<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>New Internal Transfer | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
    <script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'New Internal Transfer', 'subtitle' => 'Transfer funds between your accounts'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Back Navigation -->
            <div class="mb-6">
                <a href="{{ route('admin.accounting.cash-banking.internal-transfers.index') }}" 
                   class="inline-flex items-center text-blue-600 hover:text-blue-700 font-medium">
                    <i class="fas fa-arrow-left mr-2"></i>
                    Back to Transfers
                </a>
            </div>

            <!-- Transfer Form -->
            <form method="POST" action="{{ route('admin.accounting.cash-banking.internal-transfers.store') }}" class="space-y-6">
                @csrf

                <div class="bg-white rounded-xl shadow-sm border border-gray-200 overflow-hidden">
                    <!-- Form Header -->
                    <div class="bg-gradient-to-r from-blue-600 to-blue-700 px-6 py-4">
                        <div class="flex items-center justify-between">
                            <div>
                                <h2 class="text-xl font-bold text-white">Create New Transfer</h2>
                                <p class="text-blue-100 text-sm">Transfer funds between your cash, bank, or mobile money accounts</p>
                            </div>
                            <div class="bg-blue-500/30 p-3 rounded-lg">
                                <i class="fas fa-exchange-alt text-white text-xl"></i>
                            </div>
                        </div>
                    </div>

                    <!-- Form Body -->
                    <div class="p-6">
                        <!-- Transfer Details -->
                        <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-8">
                            <!-- Left Column -->
                            <div class="lg:col-span-2 space-y-6">
                                <!-- Date and Reference -->
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label for="transactiondate" class="block text-sm font-medium text-gray-700 mb-2">
                                            Transfer Date *
                                        </label>
                                        <div class="relative">
                                            <input type="text" name="transactiondate" id="transactiondate" required
                                                   class="pl-10 pr-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                                   value="{{ old('transactiondate', now()->format('Y-m-d')) }}">
                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <i class="fas fa-calendar text-gray-400"></i>
                                            </div>
                                        </div>
                                        @error('transactiondate')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>

                                    <div>
                                        <label for="reference" class="block text-sm font-medium text-gray-700 mb-2">
                                            Reference Number
                                        </label>
                                        <input type="text" name="reference" id="reference"
                                               class="px-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                               value="{{ old('reference') }}"
                                               placeholder="TRF-001">
                                        @error('reference')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                </div>

                                <!-- Account From -->
                                <div class="bg-gray-50 rounded-xl p-5 border border-gray-200">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-arrow-up text-red-600"></i>
                                        </div>
                                        <div>
                                            <h3 class="text-lg font-semibold text-gray-900">From Account</h3>
                                            <p class="text-sm text-gray-600">Select source account for transfer</p>
                                        </div>
                                    </div>

                                    <div>
                                        <label for="accountfrom" class="block text-sm font-medium text-gray-700 mb-2">
                                            Select Account *
                                        </label>
                                        <select name="accountfrom" id="accountfrom" required
                                                class="px-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                                onchange="updateFromAccountBalance()">
                                            <option value="">-- Select Source Account --</option>
                                            @foreach($accounts as $account)
                                            <option value="{{ $account->cashBankId }}" 
                                                    data-balance="{{ $account->current_balance }}"
                                                    data-currency="{{ $account->currency }}"
                                                    {{ old('accountfrom') == $account->cashBankId ? 'selected' : '' }}>
                                                {{ $account->accountName }} ({{ $account->accountType }}) - 
                                                {{ number_format($account->current_balance, 2) }} {{ $account->currency }}
                                            </option>
                                            @endforeach
                                        </select>
                                        @error('accountfrom')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror

                                        <!-- Account Balance Display -->
                                        <div id="fromAccountBalance" class="mt-3 p-3 bg-blue-50 rounded-lg hidden">
                                            <div class="flex items-center justify-between">
                                                <span class="text-sm font-medium text-blue-900">Available Balance:</span>
                                                <span id="fromBalanceAmount" class="text-lg font-bold text-blue-700"></span>
                                            </div>
                                        </div>
                                    </div>
                                </div>

                                <!-- Account To -->
                                <div class="bg-gray-50 rounded-xl p-5 border border-gray-200">
                                    <div class="flex items-center mb-4">
                                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                                            <i class="fas fa-arrow-down text-green-600"></i>
                                        </div>
                                        <div>
                                            <h3 class="text-lg font-semibold text-gray-900">To Account</h3>
                                            <p class="text-sm text-gray-600">Select destination account</p>
                                        </div>
                                    </div>

                                    <div>
                                        <label for="accountto" class="block text-sm font-medium text-gray-700 mb-2">
                                            Select Account *
                                        </label>
                                        <select name="accountto" id="accountto" required
                                                class="px-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                                            <option value="">-- Select Destination Account --</option>
                                            @foreach($accounts as $account)
                                            <option value="{{ $account->cashBankId }}"
                                                    {{ old('accountto') == $account->cashBankId ? 'selected' : '' }}>
                                                {{ $account->accountName }} ({{ $account->accountType }}) - 
                                                {{ number_format($account->current_balance, 2) }} {{ $account->currency }}
                                            </option>
                                            @endforeach
                                        </select>
                                        @error('accountto')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                </div>

                                <!-- Amount Details -->
                                <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                    <div>
                                        <label for="currency" class="block text-sm font-medium text-gray-700 mb-2">
                                            Currency *
                                        </label>
                                        <select name="currency" id="currency" required
                                                class="px-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500">
                                            <option value="">-- Select Currency --</option>
                                            @foreach($currencies as $currency)
                                            <option value="{{ $currency->code }}"
                                                    {{ old('currency', 'ZMW') == $currency->code ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                            @endforeach
                                        </select>
                                        @error('currency')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>

                                    <div>
                                        <label for="exchangerate" class="block text-sm font-medium text-gray-700 mb-2">
                                            Exchange Rate *
                                        </label>
                                        <input type="number" name="exchangerate" id="exchangerate" required step="0.0001" min="0.0001"
                                               class="px-4 py-3 border border-gray-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500"
                                               value="{{ old('exchangerate', 1) }}">
                                        @error('exchangerate')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                </div>

                                <!-- Amount -->
                                <div class="bg-gradient-to-r from-blue-50 to-indigo-50 rounded-xl p-5 border border-blue-200">
                                    <label for="amountsent" class="block text-sm font-medium text-gray-700 mb-2">
                                        Amount to Transfer *
                                    </label>
                                    <div class="relative">
                                        <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                            <span class="text-gray-500">ZMW</span>
                                        </div>
                                        <input type="number" name="amountsent" id="amountsent" required step="0.01" min="0.01"
                                               class="pl-16 pr-4 py-3 border-2 border-blue-300 rounded-lg w-full focus:ring-2 focus:ring-blue-500 focus:border-blue-500 text-lg font-semibold"
                                               placeholder="0.00"
                                               oninput="calculateReceivedAmount()">
                                        @error('amountsent')
                                        <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                    <div class="mt-3 flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Amount to receive:</span>
                                        <span id="receivedAmount" class="text-lg font-bold text-green-600">0.00</span>
                                    </div>
                                </div>
                            </div>

                            <!-- Right Column - Summary -->
                            <div class="space-y-6">
                                <!-- Status -->
                                <div class="bg-gray-50 rounded-xl p-5 border border-gray-200">
                                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Transfer Status</h3>
                                    <div class="space-y-3">
                                        <div class="flex items-center">
                                            <input type="radio" name="transferstatus" id="status_pending" value="Pending"
                                                   class="h-4 w-4 text-blue-600 focus:ring-blue-500 border-gray-300"
                                                   {{ old('transferstatus', 'Pending') == 'Pending' ? 'checked' : '' }}>
                                            <label for="status_pending" class="ml-3">
                                                <span class="block text-sm font-medium text-gray-700">Pending</span>
                                                <span class="block text-sm text-gray-500">Save as draft for later</span>
                                            </label>
                                        </div>
                                        <div class="flex items-center">
                                            <input type="radio" name="transferstatus" id="status_sent" value="Sent"
                                                   class="h-4 w-4 text-green-600 focus:ring-green-500 border-gray-300"
                                                   {{ old('transferstatus') == 'Sent' ? 'checked' : '' }}>
                                            <label for="status_sent" class="ml-3">
                                                <span class="block text-sm font-medium text-gray-700">Sent</span>
                                                <span class="block text-sm text-gray-500">Immediately process the transfer</span>
                                            </label>
                                        </div>
                                    </div>
                                    @error('transferstatus')
                                    <p class="mt-2 text-sm text-red-600">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Additional Details -->
                                <div class="bg-gray-50 rounded-xl p-5 border border-gray-200">
                                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Additional Information</h3>
                                    
                                    <div class="mb-4">
                                        <label for="referenceno" class="block text-sm font-medium text-gray-700 mb-2">
                                            External Reference
                                        </label>
                                        <input type="text" name="referenceno" id="referenceno"
                                               class="px-4 py-2 border border-gray-300 rounded-lg w-full focus:ring-blue-500 focus:border-blue-500"
                                               value="{{ old('referenceno') }}"
                                               placeholder="Bank reference or transaction ID">
                                    </div>

                                    <div>
                                        <label for="details" class="block text-sm font-medium text-gray-700 mb-2">
                                            Transfer Description
                                        </label>
                                        <textarea name="details" id="details" rows="3"
                                                  class="px-4 py-2 border border-gray-300 rounded-lg w-full focus:ring-blue-500 focus:border-blue-500"
                                                  placeholder="Enter purpose or description for this transfer">{{ old('details') }}</textarea>
                                    </div>
                                </div>

                                <!-- Transfer Summary -->
                                <div class="bg-gradient-to-r from-indigo-50 to-purple-50 rounded-xl p-5 border border-indigo-200">
                                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Transfer Summary</h3>
                                    <div class="space-y-3">
                                        <div class="flex justify-between items-center">
                                            <span class="text-sm text-gray-600">Source Account:</span>
                                            <span id="summaryFrom" class="text-sm font-medium text-gray-900">Not selected</span>
                                        </div>
                                        <div class="flex justify-between items-center">
                                            <span class="text-sm text-gray-600">Destination Account:</span>
                                            <span id="summaryTo" class="text-sm font-medium text-gray-900">Not selected</span>
                                        </div>
                                        <div class="flex justify-between items-center">
                                            <span class="text-sm text-gray-600">Transfer Amount:</span>
                                            <span id="summaryAmount" class="text-sm font-bold text-blue-600">0.00</span>
                                        </div>
                                        <div class="flex justify-between items-center">
                                            <span class="text-sm text-gray-600">Exchange Rate:</span>
                                            <span id="summaryRate" class="text-sm font-medium text-gray-900">1.0000</span>
                                        </div>
                                        <div class="flex justify-between items-center border-t pt-3">
                                            <span class="text-base font-medium text-gray-900">Amount Received:</span>
                                            <span id="summaryReceived" class="text-lg font-bold text-green-600">0.00</span>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="flex flex-col sm:flex-row justify-end space-y-3 sm:space-y-0 sm:space-x-4 pt-6 border-t border-gray-200">
                            <a href="{{ route('admin.accounting.cash-banking.internal-transfers.index') }}" 
                               class="px-6 py-3 border border-gray-300 text-gray-700 rounded-lg font-semibold hover:bg-gray-50 transition duration-150 text-center">
                                Cancel
                            </a>
                            <button type="submit" name="action" value="save"
                                    class="px-6 py-3 bg-blue-600 hover:bg-blue-700 text-white rounded-lg font-semibold transition duration-150 flex items-center justify-center space-x-2">
                                <i class="fas fa-save"></i>
                                <span>Save as Draft</span>
                            </button>
                            <button type="submit" name="action" value="send"
                                    class="px-6 py-3 bg-gradient-to-r from-green-600 to-emerald-600 hover:from-green-700 hover:to-emerald-700 text-white rounded-lg font-semibold transition duration-150 flex items-center justify-center space-x-2">
                                <i class="fas fa-paper-plane"></i>
                                <span>Save & Send Transfer</span>
                            </button>
                        </div>
                    </div>
                </div>
            </form>
        </main>
    </div>

    <script>
        // Initialize date picker
        flatpickr('#transactiondate', {
            dateFormat: 'Y-m-d',
            defaultDate: 'today'
        });

        // Update summary and calculations
        function updateFromAccountBalance() {
            const select = document.getElementById('accountfrom');
            const selectedOption = select.options[select.selectedIndex];
            const balanceDiv = document.getElementById('fromAccountBalance');
            const balanceSpan = document.getElementById('fromBalanceAmount');
            const summaryFrom = document.getElementById('summaryFrom');
            
            if (selectedOption.value) {
                const balance = parseFloat(selectedOption.getAttribute('data-balance'));
                const currency = selectedOption.getAttribute('data-currency');
                const accountName = selectedOption.text.split('(')[0].trim();
                
                balanceDiv.classList.remove('hidden');
                balanceSpan.textContent = `${balance.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2})} ${currency}`;
                summaryFrom.textContent = accountName;
            } else {
                balanceDiv.classList.add('hidden');
                summaryFrom.textContent = 'Not selected';
            }
        }

        function calculateReceivedAmount() {
            const amount = parseFloat(document.getElementById('amountsent').value) || 0;
            const rate = parseFloat(document.getElementById('exchangerate').value) || 1;
            const received = amount * rate;
            
            document.getElementById('receivedAmount').textContent = 
                received.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
            
            // Update summary
            document.getElementById('summaryAmount').textContent = 
                amount.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
            document.getElementById('summaryRate').textContent = 
                rate.toLocaleString('en-US', {minimumFractionDigits: 4, maximumFractionDigits: 4});
            document.getElementById('summaryReceived').textContent = 
                received.toLocaleString('en-US', {minimumFractionDigits: 2, maximumFractionDigits: 2});
        }

        // Update summary when accounts are selected
        document.getElementById('accountfrom').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            if (selectedOption.value) {
                const accountName = selectedOption.text.split('(')[0].trim();
                document.getElementById('summaryFrom').textContent = accountName;
            }
        });

        document.getElementById('accountto').addEventListener('change', function() {
            const selectedOption = this.options[this.selectedIndex];
            if (selectedOption.value) {
                const accountName = selectedOption.text.split('(')[0].trim();
                document.getElementById('summaryTo').textContent = accountName;
            }
        });

        // Update status radio buttons based on action button
        document.querySelectorAll('button[type="submit"]').forEach(button => {
            button.addEventListener('click', function() {
                const action = this.value;
                if (action === 'send') {
                    document.getElementById('status_sent').checked = true;
                } else if (action === 'save') {
                    document.getElementById('status_pending').checked = true;
                }
            });
        });

        // Initialize calculations on page load
        document.addEventListener('DOMContentLoaded', function() {
            updateFromAccountBalance();
            calculateReceivedAmount();
            
            // Set summary for selected accounts if they exist
            const fromSelect = document.getElementById('accountfrom');
            const toSelect = document.getElementById('accountto');
            
            if (fromSelect.value) {
                const fromOption = fromSelect.options[fromSelect.selectedIndex];
                document.getElementById('summaryFrom').textContent = fromOption.text.split('(')[0].trim();
            }
            
            if (toSelect.value) {
                const toOption = toSelect.options[toSelect.selectedIndex];
                document.getElementById('summaryTo').textContent = toOption.text.split('(')[0].trim();
            }
        });
    </script>
</body>
</html>