<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Create Expense Transaction | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Reuse the QuickBooks Online styling from your petty cash form */
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 12px;
            font-size: 14px;
            color: #323130;
            background: white;
            width: 100%;
            transition: all 0.2s;
        }
        
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        
        .quickbooks-input::placeholder {
            color: #A19F9D;
        }
        
        .quickbooks-select {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 12px;
            font-size: 14px;
            color: #323130;
            background: white;
            width: 100%;
            transition: all 0.2s;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='m6 9 6 6 6-6'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            background-size: 16px;
            padding-right: 36px;
        }
        
        .quickbooks-select:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        
        .quickbooks-textarea {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 12px;
            font-size: 14px;
            color: #323130;
            background: white;
            width: 100%;
            resize: vertical;
            min-height: 80px;
            transition: all 0.2s;
        }
        
        .quickbooks-textarea:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        
        .quickbooks-button-primary {
            background-color: #146CAA;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        
        .quickbooks-button-primary:hover {
            background-color: #11598C;
        }
        
        .quickbooks-button-secondary {
            background-color: white;
            color: #323130;
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .quickbooks-button-secondary:hover {
            background-color: #F3F2F1;
        }
        
        .quickbooks-label {
            display: block;
            font-size: 14px;
            font-weight: 500;
            color: #323130;
            margin-bottom: 6px;
        }
        
        .quickbooks-label-required::after {
            content: "*";
            color: #D83B01;
            margin-left: 4px;
        }
        
        .quickbooks-field-help {
            display: block;
            font-size: 12px;
            color: #605E5C;
            margin-top: 4px;
        }
        
        .quickbooks-error {
            display: block;
            font-size: 12px;
            color: #D83B01;
            margin-top: 4px;
        }
        
        .quickbooks-info-box {
            background-color: #F3F2F1;
            border: 1px solid #EDEBE9;
            border-radius: 4px;
            padding: 12px;
            margin-top: 8px;
            font-size: 13px;
            color: #605E5C;
        }
        
        .quickbooks-section {
            border-top: 1px solid #EDEBE9;
            padding-top: 24px;
            margin-top: 24px;
        }
        
        .quickbooks-card {
            background: white;
            border: 1px solid #EDEBE9;
            border-radius: 4px;
            box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
        }

        /* Animation styles */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        @keyframes fadeOut {
            from { opacity: 1; transform: translateY(0); }
            to { opacity: 0; transform: translateY(-10px); }
        }
        
        .animate-fade-in {
            animation: fadeIn 0.3s ease-out;
        }
        
        .animate-fade-out {
            animation: fadeOut 0.3s ease-in;
        }

        /* Form validation styles */
        .is-invalid {
            border-color: #D83B01 !important;
        }
        
        .is-valid {
            border-color: #107C10 !important;
        }
    </style>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'New Expense Transaction', 'subtitle' => 'Create a new expense transaction'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 quickbooks-card p-4 border-l-4 border-green-500 animate-fade-in">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3"></i>
                        <div>
                            <p class="font-medium text-gray-900">{{ session('success') }}</p>
                        </div>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 quickbooks-card p-4 border-l-4 border-red-500 animate-fade-in">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-3"></i>
                        <div>
                            <p class="font-medium text-gray-900">{{ session('error') }}</p>
                        </div>
                    </div>
                </div>
            @endif

            <!-- Form Container -->
            <div class="max-w-6xl mx-auto">
                <div class="quickbooks-card p-6">
                    <!-- Form Header with Back Button -->
                    <div class="border-b border-gray-200 pb-4 mb-6">
                        <div class="flex flex-col sm:flex-row sm:items-center sm:justify-between gap-4">
                            <div>
                                <h2 class="text-xl font-semibold text-gray-900">
                                    Expense Transaction Form
                                </h2>
                                <p class="text-gray-600 text-sm mt-1">Fill in all required fields to create a new expense transaction</p>
                            </div>
                            <div>
                                <a href="{{ route('admin.accounting.expenses.transactions.index') }}" 
                                   class="quickbooks-button-secondary flex items-center justify-center sm:justify-start">
                                    <i class="fas fa-arrow-left mr-2"></i>
                                    Back to List
                                </a>
                            </div>
                        </div>
                    </div>

                    <form action="{{ route('admin.accounting.expenses.transactions.store') }}" method="POST" class="space-y-8" id="expenseForm">
                        @csrf

                        <!-- Basic Information Section -->
                        <div>
                            <h3 class="text-lg font-semibold text-gray-900 mb-6">Basic Information</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Transaction Date -->
                                <div>
                                    <label for="transaction_date" class="quickbooks-label quickbooks-label-required">
                                        Transaction Date
                                    </label>
                                    <div class="relative">
                                        <input type="date" 
                                               id="transaction_date" 
                                               name="transaction_date" 
                                               value="{{ old('transaction_date', date('Y-m-d')) }}" 
                                               required
                                               class="quickbooks-input pl-10">
                                        <i class="fas fa-calendar-alt absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500"></i>
                                    </div>
                                    @error('transaction_date')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Vendor -->
                                <div>
                                    <label for="vendor_id" class="quickbooks-label">
                                        Vendor
                                    </label>
                                    <select id="vendor_id" 
                                            name="vendor_id" 
                                            class="quickbooks-select">
                                        <option value="">Select Vendor (Optional)</option>
                                        @foreach($vendors as $vendor)
                                            <option value="{{ $vendor->id }}" {{ old('vendor_id') == $vendor->id ? 'selected' : '' }}>
                                                {{ $vendor->display_name }} 
                                                @if($vendor->company_name)
                                                    - {{ $vendor->company_name }}
                                                @endif
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('vendor_id')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                </div>
                            </div>

                            <!-- Description -->
                            <div class="mt-6">
                                <label for="description" class="quickbooks-label quickbooks-label-required">
                                    Description
                                </label>
                                <textarea id="description" 
                                          name="description" 
                                          rows="3" 
                                          placeholder="Enter overall description"
                                          required
                                          class="quickbooks-textarea">{{ old('description') }}</textarea>
                                @error('description')
                                    <p class="quickbooks-error">{{ $message }}</p>
                                @enderror
                            </div>

                            <!-- Notes -->
                            <div class="mt-6">
                                <label for="notes" class="quickbooks-label">
                                    Notes
                                </label>
                                <textarea id="notes" 
                                          name="notes" 
                                          rows="2" 
                                          placeholder="Enter any additional notes (optional)"
                                          class="quickbooks-textarea">{{ old('notes') }}</textarea>
                                @error('notes')
                                    <p class="quickbooks-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>

                        <!-- Items Section -->
                        <div class="quickbooks-section">
                            <div class="flex justify-between items-center mb-6">
                                <h3 class="text-lg font-semibold text-gray-900">Expense Items</h3>
                                <span class="text-sm text-gray-600">Add at least one item</span>
                            </div>
                            
                            <div class="quickbooks-card p-4">
                                <div class="overflow-x-auto">
                                    <table class="quickbooks-table">
                                        <thead>
                                            <tr>
                                                <th class="text-sm font-medium text-gray-700">Chart of Account</th>
                                                <th class="text-sm font-medium text-gray-700">Description</th>
                                                <th class="text-sm font-medium text-gray-700">Amount</th>
                                                <th class="text-sm font-medium text-gray-700">Actions</th>
                                            </tr>
                                        </thead>
                                        <tbody id="itemsContainer">
                                            <!-- Items will be added here dynamically -->
                                        </tbody>
                                        <tfoot class="bg-gray-100">
                                            <tr>
                                                <td colspan="2" class="text-right text-sm font-medium text-gray-700 py-3">Total Amount:</td>
                                                <td class="py-3">
                                                    <div class="text-lg font-semibold text-gray-900" id="totalAmount">0.00</div>
                                                    <div class="text-xs text-gray-500" id="baseAmountDisplay">Base: 0.00</div>
                                                </td>
                                                <td></td>
                                            </tr>
                                        </tfoot>
                                    </table>
                                </div>
                                
                                <button type="button" 
                                        onclick="addItem()" 
                                        class="quickbooks-button-secondary mt-4 flex items-center">
                                    <i class="fas fa-plus mr-2"></i>Add Item
                                </button>
                            </div>
                        </div>

                        <!-- Currency & Exchange Rate Section -->
                        <div class="quickbooks-section">
                            <h3 class="text-lg font-semibold text-gray-900 mb-6">Currency & Exchange Rate</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Currency -->
                                <div>
                                    <label for="currency" class="quickbooks-label quickbooks-label-required">
                                        Currency
                                    </label>
                                    <select id="currency" 
                                            name="currency" 
                                            required
                                            class="quickbooks-select">
                                        <option value="">Select Currency</option>
                                        @foreach($currencies as $currency)
                                        <option value="{{ $currency->code }}" 
                                                data-symbol="{{ $currency->symbol }}"
                                                data-name="{{ $currency->name }}"
                                                {{ old('currency', $reportingCurrency?->code) == $currency->code ? 'selected' : '' }}>
                                            {{ $currency->code }} - {{ $currency->name }} 
                                        </option>
                                        @endforeach
                                    </select>
                                    @error('currency')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                    <p class="quickbooks-field-help">
                                        Default: {{ $reportingCurrency?->code }} - {{ $reportingCurrency?->name }} 
                                        ({{ $reportingCurrency?->symbol }})
                                    </p>
                                </div>

                                <!-- Exchange Rate -->
                                <div>
                                    <label for="exchange_rate" class="quickbooks-label quickbooks-label-required">
                                        Exchange Rate
                                    </label>
                                    <div class="relative">
                                        <input type="number" 
                                               id="exchange_rate" 
                                               name="exchange_rate" 
                                               value="{{ old('exchange_rate', 1) }}" 
                                               required
                                               min="0.0001" 
                                               step="0.0001"
                                               class="quickbooks-input pr-20"
                                               oninput="calculateBaseAmount()">
                                        <div class="absolute right-3 top-1/2 transform -translate-y-1/2 text-sm text-gray-600">
                                            <span id="rateText">
                                                @if($reportingCurrency)
                                                1 {{ $reportingCurrency->code }} = 
                                                @else
                                                1 USD = 
                                                @endif
                                            </span>
                                        </div>
                                    </div>
                                    @error('exchange_rate')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                    <p class="quickbooks-field-help">
                                        <span id="baseRateText">
                                            @if($reportingCurrency)
                                            Base currency: {{ $reportingCurrency->code }} ({{ $reportingCurrency->name }})
                                            @else
                                            Base currency: USD
                                            @endif
                                        </span>
                                    </p>
                                </div>
                            </div>
                        </div>

                        <!-- Payment Information Section -->
                        <div class="quickbooks-section">
                            <h3 class="text-lg font-semibold text-gray-900 mb-6">Payment Information</h3>
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Payment Method -->
                                <div>
                                    <label for="payment_method" class="quickbooks-label quickbooks-label-required">
                                        Payment Method
                                    </label>
                                    <select id="payment_method" 
                                       name="payment_method" 
                                       required
                                       class="quickbooks-select"
                                       onchange="togglePaymentFields()">
                                       <option value="">Select Payment Method</option>
                                       <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>Cash</option>
                                       <option value="mobile_money" {{ old('payment_method') == 'mobile_money' ? 'selected' : '' }}>Mobile Money</option>
                                       <option value="bank" {{ old('payment_method') == 'bank' ? 'selected' : '' }}>Bank</option>
                                    </select>
                                    @error('payment_method')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Account (for cash, bank transfer, mobile money) -->
                                <div>
                                    <label for="account_id" class="quickbooks-label" id="accountLabel">
                                        Account
                                    </label>
                                    <select id="account_id" 
                                            name="account_id" 
                                            class="quickbooks-select">
                                        <option value="">Select Account</option>
                                        @foreach($accounts as $account)
                                            <option value="{{ $account->cashBankId }}" 
                                                    data-account-type="{{ $account->accountType }}"
                                                    data-currency="{{ $account->currency }}"
                                                    data-balance="{{ $account->current_balance }}"
                                                    {{ old('account_id') == $account->cashBankId ? 'selected' : '' }}>
                                                {{ $account->accountType }} - {{ $account->accountName }} ({{ $account->currency }})
                                            </option>
                                        @endforeach
                                    </select>
                                    <div class="mt-3 quickbooks-info-box hidden" id="accountBalanceBox">
                                        <div class="flex items-center justify-between">
                                            <span class="text-sm text-gray-600">Available Balance:</span>
                                            <span id="accountBalance" class="text-sm font-medium text-gray-900">0.00</span>
                                            <span id="accountCurrency" class="text-sm text-gray-600"></span>
                                        </div>
                                    </div>
                                    @error('account_id')
                                        <p class="quickbooks-error">{{ $message }}</p>
                                    @enderror
                                </div>
                            </div>

                            <!-- Check Number (only for check payments) -->
                            <div class="mt-6 hidden" id="checkNumberField">
                                <label for="check_number" class="quickbooks-label">
                                    Check Number
                                </label>
                                <input type="text" 
                                       id="check_number" 
                                       name="check_number" 
                                       value="{{ old('check_number') }}"
                                       placeholder="Enter check number"
                                       class="quickbooks-input w-full md:w-1/2">
                                @error('check_number')
                                    <p class="quickbooks-error">{{ $message }}</p>
                                @enderror
                            </div>

                            <!-- Reference Number -->
                            <div class="mt-6">
                                <label for="reference_number" class="quickbooks-label">
                                    Reference Number
                                </label>
                                <input type="text" 
                                       id="reference_number" 
                                       name="reference_number" 
                                       value="{{ old('reference_number') }}"
                                       placeholder="Enter reference number (optional)"
                                       class="quickbooks-input w-full md:w-1/2">
                                @error('reference_number')
                                    <p class="quickbooks-error">{{ $message }}</p>
                                @enderror
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="border-t border-gray-200 pt-6 mt-6">
                            <div class="flex justify-end space-x-3">
                                <button type="submit" 
                                        class="quickbooks-button-primary flex items-center">
                                    <i class="fas fa-save mr-2"></i>
                                    Save Expense Transaction
                                </button>
                            </div>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Items management
        let itemCounter = 0;
        const itemsContainer = document.getElementById('itemsContainer');
        const totalAmountElement = document.getElementById('totalAmount');
        const baseAmountDisplay = document.getElementById('baseAmountDisplay');

        function addItem(itemData = {}) {
            itemCounter++;
            
            const itemRow = document.createElement('tr');
            itemRow.id = `itemRow${itemCounter}`;
            itemRow.innerHTML = `
                <td class="py-3">
                    <select name="items[${itemCounter}][chart_of_account_id]" 
                            class="quickbooks-select w-full item-account" 
                            onchange="validateItemField(this)">
                        <option value="">Select Chart of Account</option>
                        @foreach($chartOfAccounts as $account)
                        <option value="{{ $account->id }}" ${itemData.chart_of_account_id == {{ $account->id }} ? 'selected' : ''}>
                            {{ $account->name }} ({{ $account->code }})
                        </option>
                        @endforeach
                    </select>
                    <span class="quickbooks-error hidden" id="accountError${itemCounter}">Please select a chart of account</span>
                </td>
                <td class="py-3">
                    <input type="text" 
                           name="items[${itemCounter}][item_description]" 
                           value="${itemData.item_description || ''}"
                           required
                           class="quickbooks-input w-full item-description"
                           placeholder="Enter item description"
                           oninput="validateItemField(this)">
                    <span class="quickbooks-error hidden" id="descriptionError${itemCounter}">Please enter a description</span>
                </td>
                <td class="py-3">
                    <div class="relative">
                        <input type="number" 
                               name="items[${itemCounter}][amount]" 
                               value="${itemData.amount || ''}"
                               required
                               min="0.01" 
                               step="0.01"
                               class="quickbooks-input w-full pr-10 item-amount"
                               placeholder="0.00"
                               oninput="validateItemField(this); calculateTotal()">
                        <span class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray-600 text-sm">
                            <span id="itemCurrency${itemCounter}">
                                @if($reportingCurrency)
                                {{ $reportingCurrency->symbol }}
                                @else
                                USD
                                @endif
                            </span>
                        </span>
                    </div>
                    <span class="quickbooks-error hidden" id="amountError${itemCounter}">Please enter a valid amount</span>
                </td>
                <td class="py-3">
                    <button type="button" 
                            onclick="removeItem(${itemCounter})"
                            class="quickbooks-button-secondary p-2 text-red-600 hover:text-red-700">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;
            
            itemsContainer.appendChild(itemRow);
            calculateTotal();
            updateCurrencyDisplay();
        }

        function removeItem(itemId) {
            const row = document.getElementById(`itemRow${itemId}`);
            if (row) {
                row.remove();
                calculateTotal();
            }
        }

        function validateItemField(field) {
            const value = field.value.trim();
            const isSelect = field.tagName === 'SELECT';
            const isEmpty = isSelect ? value === '' : value === '';
            const isAmount = field.classList.contains('item-amount');
            
            field.classList.remove('is-invalid', 'is-valid');
            
            if (isEmpty) {
                field.classList.add('is-invalid');
            } else {
                field.classList.add('is-valid');
                
                if (isAmount) {
                    const amountValue = parseFloat(value);
                    if (isNaN(amountValue) || amountValue <= 0) {
                        field.classList.add('is-invalid');
                        field.classList.remove('is-valid');
                    }
                }
            }
            
            const errorId = isSelect ? `accountError${field.closest('tr').id.replace('itemRow', '')}` :
                          isAmount ? `amountError${field.closest('tr').id.replace('itemRow', '')}` :
                          `descriptionError${field.closest('tr').id.replace('itemRow', '')}`;
            
            const errorElement = document.getElementById(errorId);
            if (errorElement) {
                if (isEmpty || (isAmount && (isNaN(parseFloat(value)) || parseFloat(value) <= 0))) {
                    errorElement.classList.remove('hidden');
                } else {
                    errorElement.classList.add('hidden');
                }
            }
        }

        function calculateTotal() {
            let total = 0;
            const amountInputs = document.querySelectorAll('.item-amount');
            let hasInvalidAmounts = false;
            
            amountInputs.forEach(input => {
                const value = parseFloat(input.value) || 0;
                total += value;
                
                if (value <= 0) {
                    hasInvalidAmounts = true;
                    input.classList.add('is-invalid');
                    input.classList.remove('is-valid');
                } else {
                    input.classList.add('is-valid');
                    input.classList.remove('is-invalid');
                }
            });
            
            const formattedTotal = total.toLocaleString('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            });
            
            totalAmountElement.textContent = formattedTotal;
            calculateBaseAmount();
            updateAccountBalance();
        }

        function calculateBaseAmount() {
            const total = parseFloat(totalAmountElement.textContent.replace(/,/g, '')) || 0;
            const exchangeRate = parseFloat(document.getElementById('exchange_rate').value) || 1;
            const baseAmount = total * exchangeRate;
            
            baseAmountDisplay.textContent = `Base: ${baseAmount.toLocaleString('en-US', {
                minimumFractionDigits: 2,
                maximumFractionDigits: 2
            })}`;
        }

        function updateAccountBalance() {
            
            const paymentMethod = document.getElementById('payment_method').value;
            const select = document.getElementById('account_id');
            const balanceBox = document.getElementById('accountBalanceBox');
    
            if (!['cash', 'mobile_money', 'bank'].includes(paymentMethod)) {
                  if (balanceBox) balanceBox.classList.add('hidden');
                  return;
            }
            
            const selectedOption = select.options[select.selectedIndex];
            
            if (selectedOption && selectedOption.value) {
                const balance = parseFloat(selectedOption.getAttribute('data-balance')) || 0;
                const currency = selectedOption.getAttribute('data-currency') || '';
                
                const formattedBalance = balance.toLocaleString('en-US', {
                    minimumFractionDigits: 2,
                    maximumFractionDigits: 2
                });
                
                document.getElementById('accountBalance').textContent = formattedBalance;
                document.getElementById('accountCurrency').textContent = currency;
                
                balanceBox.classList.remove('hidden');
                
                // Check if total exceeds balance
                const total = parseFloat(totalAmountElement.textContent.replace(/,/g, '')) || 0;
                
                if (total > 0 && total > balance) {
                    document.getElementById('accountBalance').classList.add('text-red-600');
                    document.getElementById('accountBalance').classList.remove('text-green-600');
                } else if (total > 0 && total <= balance) {
                    document.getElementById('accountBalance').classList.add('text-green-600');
                    document.getElementById('accountBalance').classList.remove('text-red-600');
                } else {
                    document.getElementById('accountBalance').classList.remove('text-red-600', 'text-green-600');
                }
            } else {
                document.getElementById('accountBalance').textContent = '0.00';
                document.getElementById('accountCurrency').textContent = '';
                balanceBox.classList.remove('hidden');
            }
        }

        function togglePaymentFields() {
          const paymentMethod = document.getElementById('payment_method').value;
          const accountLabel = document.getElementById('accountLabel');
    
          // Update account label
          if (accountLabel) {
               if (['cash', 'mobile_money', 'bank'].includes(paymentMethod)) {
                   accountLabel.classList.add('quickbooks-label-required');
               } else {
                   accountLabel.classList.remove('quickbooks-label-required');
            }
          }
    
          updateAccountBalance();
        }
        function updateCurrencyDisplay() {
            const currencySelect = document.getElementById('currency');
            const selectedOption = currencySelect.options[currencySelect.selectedIndex];
            const currencySymbol = selectedOption ? selectedOption.getAttribute('data-symbol') || currencySelect.value : '';
            const currencyName = selectedOption ? selectedOption.getAttribute('data-name') || currencySelect.value : '';
            const rateText = document.getElementById('rateText');
            const baseRateText = document.getElementById('baseRateText');
            
            const reportingCurrencyCode = '{{ $reportingCurrency?->code }}' || 'USD';
            const reportingCurrencyName = '{{ $reportingCurrency?->name }}' || 'US Dollar';
            
            rateText.textContent = `1 ${reportingCurrencyCode} = `;
            baseRateText.textContent = `Base currency: ${reportingCurrencyCode} (${reportingCurrencyName})`;
            
            document.querySelectorAll('[id^="itemCurrency"]').forEach(element => {
                element.textContent = currencySymbol || reportingCurrencyCode;
            });
            
            if (currencySelect.value && currencySelect.value === reportingCurrencyCode) {
                document.getElementById('exchange_rate').value = 1;
                calculateBaseAmount();
            }
        }

        function validateExchangeRate() {
            const exchangeRateInput = document.getElementById('exchange_rate');
            const value = parseFloat(exchangeRateInput.value);
            
            exchangeRateInput.classList.remove('is-invalid', 'is-valid');
            
            if (isNaN(value) || value <= 0) {
                exchangeRateInput.classList.add('is-invalid');
                return false;
            } else {
                exchangeRateInput.classList.add('is-valid');
                return true;
            }
        }

        // Form validation
        document.getElementById('expenseForm').addEventListener('submit', function(e) {
            e.preventDefault();
            
            let isValid = true;
            let errorMessage = '';
            
            // Validate basic fields
            const requiredFields = [
                { id: 'transaction_date', name: 'Transaction Date' },
                { id: 'description', name: 'Description' },
                { id: 'currency', name: 'Currency' },
                { id: 'exchange_rate', name: 'Exchange Rate' },
                { id: 'payment_method', name: 'Payment Method' },
            ];
            
            requiredFields.forEach(field => {
                const element = document.getElementById(field.id);
                if (!element.value.trim()) {
                    element.classList.add('is-invalid');
                    isValid = false;
                    errorMessage = `Please fill in all required fields. ${field.name} is required.`;
                } else {
                    element.classList.remove('is-invalid');
                }
            });
            
            // Validate account for cash, bank transfer, mobile money
            const paymentMethod = document.getElementById('payment_method').value;
            if (['cash', 'bank_transfer', 'mobile_money'].includes(paymentMethod)) {
                const accountSelect = document.getElementById('account_id');
                if (!accountSelect.value.trim()) {
                    accountSelect.classList.add('is-invalid');
                    isValid = false;
                    errorMessage = 'Please select an account for this payment method.';
                } else {
                    accountSelect.classList.remove('is-invalid');
                }
            }
            
            // Validate exchange rate
            if (!validateExchangeRate()) {
                isValid = false;
                errorMessage = 'Please enter a valid exchange rate greater than 0.';
            }
            
            // Validate items
            const itemRows = document.querySelectorAll('#itemsContainer tr[id^="itemRow"]');
            if (itemRows.length === 0) {
                isValid = false;
                errorMessage = 'Please add at least one item.';
            }
            
            let hasValidItems = false;
            itemRows.forEach(row => {
                const accountSelect = row.querySelector('.item-account');
                const descriptionInput = row.querySelector('.item-description');
                const amountInput = row.querySelector('.item-amount');
                
                if (!accountSelect.value || !descriptionInput.value.trim() || !amountInput.value || parseFloat(amountInput.value) <= 0) {
                    isValid = false;
                    if (!errorMessage) {
                        errorMessage = 'Please ensure all items have a valid chart of account, description, and amount.';
                    }
                } else {
                    hasValidItems = true;
                }
            });
            
            if (!hasValidItems && itemRows.length > 0) {
                isValid = false;
                errorMessage = 'Please ensure all items have a valid chart of account, description, and amount.';
            }
            
            // Validate total amount
            const total = parseFloat(totalAmountElement.textContent.replace(/,/g, '')) || 0;
            if (total <= 0) {
                isValid = false;
                errorMessage = 'Total amount must be greater than 0.';
            }
            
            // Validate account balance
            const selectedOption = document.getElementById('account_id').options[document.getElementById('account_id').selectedIndex];
            const balance = selectedOption ? parseFloat(selectedOption.getAttribute('data-balance')) || 0 : 0;
            
            if (selectedOption && total > balance && ['cash', 'bank_transfer', 'mobile_money'].includes(paymentMethod)) {
                isValid = false;
                errorMessage = 'Insufficient account balance! Please select an account with sufficient funds.';
            }
            
            if (isValid) {
                this.submit();
            } else {
                showNotification(errorMessage, 'error');
                
                const firstError = document.querySelector('.is-invalid');
                if (firstError) {
                    firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    firstError.focus();
                }
            }
        });

        // Event listeners
        document.getElementById('currency').addEventListener('change', updateCurrencyDisplay);
        document.getElementById('exchange_rate').addEventListener('input', function() {
            validateExchangeRate();
            calculateBaseAmount();
        });
        document.getElementById('payment_method').addEventListener('change', togglePaymentFields);
        document.getElementById('account_id').addEventListener('change', updateAccountBalance);

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            updateCurrencyDisplay();
            togglePaymentFields();
            updateAccountBalance();
            
            // Set default date to today
            const today = new Date().toISOString().split('T')[0];
            const dateInput = document.getElementById('transaction_date');
            if (dateInput && !dateInput.value) {
                dateInput.value = today;
            }
            
            if (dateInput && dateInput.value) {
                dateInput.classList.add('is-valid');
            }
            
            // Add initial empty item
            addItem();
            
            // Auto-hide success/error messages after 5 seconds
            setTimeout(() => {
                const alerts = document.querySelectorAll('.quickbooks-card.border-l-4');
                alerts.forEach(alert => {
                    alert.classList.add('animate-fade-out');
                    setTimeout(() => {
                        alert.style.display = 'none';
                    }, 300);
                });
            }, 5000);
        });

        // Notification function
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'border-green-500 bg-green-50',
                error: 'border-red-500 bg-red-50',
                info: 'border-blue-500 bg-blue-50',
                warning: 'border-yellow-500 bg-yellow-50'
            };

            const icons = {
                success: 'fa-check-circle text-green-500',
                error: 'fa-exclamation-circle text-red-500',
                info: 'fa-info-circle text-blue-500',
                warning: 'fa-exclamation-triangle text-yellow-500'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 quickbooks-card p-4 border-l-4 ${colors[type]} shadow-lg z-50 flex items-center animate-fade-in max-w-md`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-3 text-lg"></i>
                <span class="text-gray-900">${message}</span>
                <button onclick="this.parentElement.remove()" class="ml-auto text-gray-500 hover:text-gray-700">
                    <i class="fas fa-times"></i>
                </button>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                if (notification.parentElement) {
                    notification.classList.add('animate-fade-out');
                    setTimeout(() => {
                        if (notification.parentElement) {
                            notification.remove();
                        }
                    }, 300);
                }
            }, 5000);
        }
    </script>
</body>
</html>