<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Expense Transactions | {{ config('app.name', 'Laravel') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .quickbooks-search {
            background-color: white;
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px 6px 36px;
            font-size: 14px;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Ccircle cx='11' cy='11' r='8'%3E%3C/circle%3E%3Cpath d='m21 21-4.35-4.35'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: 12px center;
        }
        .quickbooks-search:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-dropdown {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            background: white;
            padding: 6px 12px;
            font-size: 14px;
        }
        .action-btn {
            width: 32px;
            height: 32px;
            border-radius: 4px;
            display: inline-flex;
            align-items: center;
            justify-content: center;
            margin: 0 2px;
            transition: all 0.2s;
        }
        .action-btn:hover {
            transform: translateY(-1px);
        }
        .action-btn-edit {
            background-color: #E1F0FA;
            color: #146CAA;
        }
        .action-btn-edit:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-delete {
            background-color: #FEE;
            color: #D83B01;
        }
        .action-btn-delete:hover {
            background-color: #D83B01;
            color: white;
        }
        .action-btn-view {
            background-color: #F3F2F1;
            color: #323130;
        }
        .action-btn-view:hover {
            background-color: #146CAA;
            color: white;
        }
        .action-btn-post {
            background-color: #E6F4EA;
            color: #107C10;
        }
        .action-btn-post:hover {
            background-color: #107C10;
            color: white;
        }
    </style>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', [
            'title' => 'Expense Transactions',
            'subtitle' => 'Manage expense transactions'
        ])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Stats Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-exchange-alt text-blue-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Total Transactions</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $summary['total_transactions'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-check-circle text-green-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Posted</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $summary['posted_transactions'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-yellow-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-edit text-yellow-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Draft</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $summary['draft_transactions'] ?? 0 }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-purple-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-money-bill-wave text-purple-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Total Amount</p>
                            <p class="text-xl font-semibold text-gray-900">${{ number_format($summary['total_amount'] ?? 0, 2) }}</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Search and Filter Bar -->
            <div class="bg-white rounded-lg border border-gray-200 p-4 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between space-y-4 md:space-y-0">
                    <div class="flex items-center space-x-4">
                        <!-- Search -->
                        <div class="relative">
                            <input type="text" 
                                   name="search"
                                   id="searchInput"
                                   class="quickbooks-search w-64"
                                   placeholder="Search transactions"
                                   value="{{ request('search') }}"
                                   onkeypress="handleSearchKeyPress(event)">
                        </div>
                        
                        <!-- Status Filter -->
                        <select id="statusFilter" class="quickbooks-dropdown">
                            <option value="">All status</option>
                            <option value="draft" {{ request('status') == 'draft' ? 'selected' : '' }}>Draft</option>
                            <option value="posted" {{ request('status') == 'posted' ? 'selected' : '' }}>Posted</option>
                            <option value="reversed" {{ request('status') == 'reversed' ? 'selected' : '' }}>Reversed</option>
                        </select>
                        
                        <!-- Vendor Filter -->
                        <select id="vendorFilter" class="quickbooks-dropdown">
                            <option value="">All vendors</option>
                            @foreach($vendors as $vendor)
                                <option value="{{ $vendor->id }}" {{ request('vendor_id') == $vendor->id ? 'selected' : '' }}>
                                    {{ $vendor->display_name }}
                                </option>
                            @endforeach
                        </select>
                        
                        <!-- Date Range -->
                        <div class="flex space-x-2">
                            <input type="date" 
                                   id="startDate"
                                   class="quickbooks-input"
                                   value="{{ request('start_date') }}"
                                   placeholder="Start Date">
                            <span class="text-gray-500 self-center">to</span>
                            <input type="date" 
                                   id="endDate"
                                   class="quickbooks-input"
                                   value="{{ request('end_date') }}"
                                   placeholder="End Date">
                        </div>
                    </div>
                    
                    <!-- Action Buttons -->
                    <div class="flex items-center space-x-3">
                        <button onclick="applyFilters()" 
                                class="border border-gray-300 text-gray-700 px-3 py-2 rounded text-sm font-medium hover:bg-gray-50 flex items-center">
                            <i class="fas fa-filter mr-2"></i>Apply Filters
                        </button>
                        <a href="{{ route('admin.accounting.expenses.transactions.create') }}"
                           class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center">
                            <i class="fas fa-plus mr-2"></i>New Transaction
                        </a>
                    </div>
                </div>
            </div>

            <!-- Transactions Table -->
            <div class="bg-white rounded-lg border border-gray-200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Transaction #</th>
                                <th>Date</th>
                                <th>Vendor</th>
                                <th>Description</th>
                                <th>Amount</th>
                                <th>Status</th>
                                <th>Payment Method</th>
                                <th class="w-48">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @forelse($transactions as $transaction)
                                <tr class="hover:bg-gray-50 transition-colors">
                                    <td class="font-medium text-gray-900">
                                        {{ $transaction->transaction_number }}
                                    </td>
                                    <td>
                                        {{ $transaction->transaction_date->format('M d, Y') }}
                                    </td>
                                    <td>
                                        {{ $transaction->vendor->display_name ?? '—' }}
                                    </td>
                                    <td>
                                        <div class="text-gray-700">{{ Str::limit($transaction->description, 50) }}</div>
                                        @if($transaction->reference_number)
                                            <div class="text-xs text-gray-500">Ref: {{ $transaction->reference_number }}</div>
                                        @endif
                                    </td>
                                    <td>
                                        <div class="font-medium text-gray-900">
                                            ${{ number_format($transaction->amount, 2) }}
                                        </div>
                                        <div class="text-xs text-gray-500">
                                            {{ $transaction->currency }}
                                        </div>
                                    </td>
                                    <td>
                                        @php
                                            $status = $transaction->status_badge;
                                        @endphp
                                        <span class="quickbooks-badge {{ $status['class'] }}">
                                            {{ $status['label'] }}
                                        </span>
                                    </td>
                                    <td>
                                        @php
                                            $paymentMethod = $transaction->payment_method_badge;
                                        @endphp
                                        <span class="quickbooks-badge {{ $paymentMethod['class'] }}">
                                            {{ $paymentMethod['label'] }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="flex items-center">
                                            <a href="{{ route('admin.accounting.expenses.transactions.show', $transaction->expense_transaction_id) }}" 
                                               class="action-btn action-btn-view"
                                               title="View details">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            
                                            @if($transaction->canEdit())
                                                <a href="{{ route('admin.accounting.expenses.transactions.edit', $transaction->expense_transaction_id) }}" 
                                                   class="action-btn action-btn-edit"
                                                   title="Edit transaction">
                                                    <i class="fas fa-edit"></i>
                                                </a>
                                            @endif
                                            
                                            @if($transaction->canPost())
                                                <button onclick="postTransaction({{ $transaction->expense_transaction_id }})" 
                                                        class="action-btn action-btn-post"
                                                        title="Post transaction">
                                                    <i class="fas fa-check"></i>
                                                </button>
                                            @endif
                                            
                                            @if($transaction->canReverse())
                                                <button onclick="reverseTransaction({{ $transaction->expense_transaction_id }})" 
                                                        class="action-btn action-btn-reverse"
                                                        title="Reverse transaction">
                                                    <i class="fas fa-undo"></i>
                                                </button>
                                            @endif
                                            
                                            @if($transaction->canEdit())
                                                <button onclick="openDeleteModal({{ $transaction->expense_transaction_id }})" 
                                                        class="action-btn action-btn-delete"
                                                        title="Delete transaction">
                                                    <i class="fas fa-trash"></i>
                                                </button>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @empty
                                <tr>
                                    <td colspan="8" class="text-center py-8">
                                        <div class="py-16 text-center">
                                            <div class="w-16 h-16 bg-gray-100 rounded-full flex items-center justify-center mx-auto mb-4">
                                                <i class="fas fa-exchange-alt text-gray-500 text-2xl"></i>
                                            </div>
                                            <h3 class="text-lg font-medium text-gray-900 mb-2">No transactions found</h3>
                                            <p class="text-gray-600 mb-6 max-w-md mx-auto">Get started by creating your first expense transaction.</p>
                                            <div class="space-x-3">
                                                <a href="{{ route('admin.accounting.expenses.transactions.create') }}" 
                                                   class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90">
                                                    <i class="fas fa-plus mr-2"></i>Create Transaction
                                                </a>
                                            </div>
                                        </div>
                                    </td>
                                </tr>
                            @endforelse
                        </tbody>
                    </table>
                </div>
                
                <!-- Pagination -->
                @if($transactions->hasPages())
                    <div class="border-t border-gray-200 p-4 flex items-center justify-between">
                        <div class="text-sm text-gray-600">
                            Showing {{ $transactions->firstItem() }} to {{ $transactions->lastItem() }} of {{ $transactions->total() }} transactions
                        </div>
                        <div class="flex items-center space-x-1">
                            {{ $transactions->appends(request()->query())->links('vendor.pagination.tailwind') }}
                        </div>
                    </div>
                @endif
            </div>
        </main>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteModal" class="fixed inset-0 bg-black bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-red-600"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray-900">Delete Transaction</h3>
                        <p class="text-gray-600 text-sm">Are you sure you want to delete this transaction?</p>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray-200 p-6 flex justify-end space-x-3">
                <button onclick="closeDeleteModal()" class="px-4 py-2 text-gray-700 hover:bg-gray-100 rounded text-sm font-medium">
                    Cancel
                </button>
                <form id="deleteForm" method="POST" style="display: none;">
                    @csrf
                    @method('DELETE')
                </form>
                <button onclick="confirmDelete()" class="bg-red-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-red-700">
                    Delete
                </button>
            </div>
        </div>
    </div>

    <!-- Reverse Confirmation Modal -->
    <div id="reverseModal" class="fixed inset-0 bg-black bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-yellow-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-yellow-600"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray-900">Reverse Transaction</h3>
                        <p class="text-gray-600 text-sm">Enter reason for reversal:</p>
                    </div>
                </div>
                
                <div class="mt-4">
                    <textarea id="reversalReason" 
                              class="quickbooks-input w-full" 
                              rows="3" 
                              placeholder="Enter reversal reason..."></textarea>
                </div>
            </div>
            
            <div class="border-t border-gray-200 p-6 flex justify-end space-x-3">
                <button onclick="closeReverseModal()" class="px-4 py-2 text-gray-700 hover:bg-gray-100 rounded text-sm font-medium">
                    Cancel
                </button>
                <button onclick="confirmReverse()" class="bg-yellow-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-yellow-700">
                    Reverse
                </button>
            </div>
        </div>
    </div>

    <script>
        let currentDeleteId = null;
        let currentReverseId = null;

        // Filter functions
        function handleSearchKeyPress(event) {
            if (event.key === 'Enter') {
                applyFilters();
            }
        }

        function applyFilters() {
            const search = document.getElementById('searchInput').value;
            const status = document.getElementById('statusFilter').value;
            const vendor = document.getElementById('vendorFilter').value;
            const startDate = document.getElementById('startDate').value;
            const endDate = document.getElementById('endDate').value;
            
            let url = '{{ route("admin.accounting.expenses.transactions.index") }}?';
            const params = new URLSearchParams();
            
            if (search) params.append('search', search);
            if (status) params.append('status', status);
            if (vendor) params.append('vendor_id', vendor);
            if (startDate) params.append('start_date', startDate);
            if (endDate) params.append('end_date', endDate);
            
            window.location.href = url + params.toString();
        }

        // Transaction actions
        function postTransaction(transactionId) {
            if (confirm('Are you sure you want to post this transaction?')) {
                fetch(`/admin/accounting/expenses/transactions/${transactionId}/post`, {
                    method: 'POST',
                    headers: {
                        'X-CSRF-TOKEN': '{{ csrf_token() }}',
                        'Accept': 'application/json',
                    },
                })
                .then(response => response.json())
                .then(data => {
                    if (data.success) {
                        showNotification(data.message, 'success');
                        setTimeout(() => {
                            location.reload();
                        }, 1000);
                    } else {
                        showNotification(data.message, 'error');
                    }
                })
                .catch(error => {
                    showNotification('Failed to post transaction', 'error');
                });
            }
        }

        function reverseTransaction(transactionId) {
            currentReverseId = transactionId;
            document.getElementById('reverseModal').classList.remove('hidden');
        }

        function closeReverseModal() {
            document.getElementById('reverseModal').classList.add('hidden');
            currentReverseId = null;
            document.getElementById('reversalReason').value = '';
        }

        function confirmReverse() {
            const reason = document.getElementById('reversalReason').value.trim();
            
            if (!reason) {
                showNotification('Please enter a reversal reason', 'error');
                return;
            }
            
            fetch(`/admin/accounting/expenses/transactions/${currentReverseId}/reverse`, {
                method: 'POST',
                headers: {
                    'X-CSRF-TOKEN': '{{ csrf_token() }}',
                    'Accept': 'application/json',
                    'Content-Type': 'application/json',
                },
                body: JSON.stringify({
                    reversal_reason: reason
                })
            })
            .then(response => response.json())
            .then(data => {
                if (data.success) {
                    showNotification(data.message, 'success');
                    closeReverseModal();
                    setTimeout(() => {
                        location.reload();
                    }, 1000);
                } else {
                    showNotification(data.message, 'error');
                }
            })
            .catch(error => {
                showNotification('Failed to reverse transaction', 'error');
            });
        }

        // Delete modal functions
        function openDeleteModal(transactionId) {
            currentDeleteId = transactionId;
            document.getElementById('deleteModal').classList.remove('hidden');
        }

        function closeDeleteModal() {
            document.getElementById('deleteModal').classList.add('hidden');
            currentDeleteId = null;
        }

        function confirmDelete() {
            const form = document.getElementById('deleteForm');
            form.action = '{{ route("admin.accounting.expenses.transactions.destroy", ":id") }}'.replace(':id', currentDeleteId);
            form.submit();
        }

        // Utility functions
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.custom-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-blue-500',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `custom-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            // Remove after 3 seconds
            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        const style = document.createElement('style');
        style.textContent = `
            @keyframes fadeIn {
                from { opacity: 0; transform: translateY(-10px); }
                to { opacity: 1; transform: translateY(0); }
            }
            @keyframes fadeOut {
                from { opacity: 1; transform: translateY(0); }
                to { opacity: 0; transform: translateY(-10px); }
            }
            .animate-fade-in {
                animation: fadeIn 0.3s ease-out;
            }
            .animate-fade-out {
                animation: fadeOut 0.3s ease-in;
            }
            .action-btn-reverse {
                background-color: #FEF3C7;
                color: #D97706;
            }
            .action-btn-reverse:hover {
                background-color: #D97706;
                color: white;
            }
        `;
        document.head.appendChild(style);

        // Initialize date filters
        document.addEventListener('DOMContentLoaded', function() {
            // Set default date range to current month if not set
            if (!document.getElementById('startDate').value) {
                const today = new Date();
                const firstDay = new Date(today.getFullYear(), today.getMonth(), 1);
                document.getElementById('startDate').value = firstDay.toISOString().split('T')[0];
            }
            
            if (!document.getElementById('endDate').value) {
                const today = new Date();
                document.getElementById('endDate').value = today.toISOString().split('T')[0];
            }
        });
    </script>
</body>
</html>