<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>New Employee | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        /* QuickBooks Online Styling */
        .quickbooks-input {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
        }
        .quickbooks-input:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-select {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            transition: all 0.2s;
            -webkit-appearance: none;
            -moz-appearance: none;
            appearance: none;
            background-image: url("data:image/svg+xml,%3Csvg xmlns='http://www.w3.org/2000/svg' width='16' height='16' viewBox='0 0 24 24' fill='none' stroke='%23605E5C' stroke-width='2' stroke-linecap='round' stroke-linejoin='round'%3E%3Cpath d='m6 9 6 6 6-6'%3E%3C/path%3E%3C/svg%3E");
            background-repeat: no-repeat;
            background-position: right 12px center;
            background-size: 16px;
            padding-right: 36px;
        }
        .quickbooks-select:focus {
            outline: 2px solid #146CAA;
            outline-offset: -1px;
            border-color: #146CAA;
        }
        .quickbooks-textarea {
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 6px 12px;
            font-size: 14px;
            width: 100%;
            resize: vertical;
            min-height: 60px;
        }
        .quickbooks-table {
            width: 100%;
            border-collapse: separate;
            border-spacing: 0;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
            vertical-align: middle;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #f8fafc;
        }
        .quickbooks-checkbox {
            width: 18px;
            height: 18px;
            border: 2px solid #605E5C;
            border-radius: 3px;
            cursor: pointer;
            accent-color: #146CAA;
        }
        .quickbooks-checkbox:checked {
            background-color: #146CAA;
            border-color: #146CAA;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .form-section {
            background: white;
            border-radius: 8px;
            padding: 24px;
            margin-bottom: 16px;
            box-shadow: 0 1px 3px rgba(0,0,0,0.1);
        }
        .form-section h3 {
            font-size: 18px;
            font-weight: 600;
            color: #323130;
            margin-bottom: 20px;
            padding-bottom: 12px;
            border-bottom: 1px solid #EDEBE9;
        }
        .required-field::after {
            content: ' *';
            color: #D83B01;
        }
        .quickbooks-button-primary {
            background-color: #146CAA;
            color: white;
            border: none;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: background-color 0.2s;
        }
        .quickbooks-button-primary:hover {
            background-color: #11598C;
        }
        .quickbooks-button-secondary {
            background-color: white;
            color: #323130;
            border: 1px solid #8C8C8C;
            border-radius: 4px;
            padding: 8px 16px;
            font-size: 14px;
            font-weight: 500;
            cursor: pointer;
            transition: all 0.2s;
        }
        .quickbooks-button-secondary:hover {
            background-color: #F3F2F1;
        }
        .is-invalid {
            border-color: #D83B01 !important;
        }
        .is-valid {
            border-color: #107C10 !important;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'New Employee', 'subtitle' => 'Add a new employee to the payroll'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Back Button -->
            <div class="mb-6">
                <a href="{{ route('admin.accounting.payroll.employees.index') }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Employees
                </a>
            </div>

            <!-- Employee Form -->
            <div class="bg-white rounded-lg border border-gray200 shadow-sm">
                <form id="employeeForm" action="{{ route('admin.accounting.payroll.employees.store') }}" method="POST">
                    @csrf
                    
                    <!-- Personal Information Section -->
                    <div class="form-section">
                        <h3>Personal Information</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">First Name</label>
                                <input type="text" 
                                       name="first_name" 
                                       class="quickbooks-input" 
                                       placeholder="John"
                                       value="{{ old('first_name') }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Middle Name</label>
                                <input type="text" 
                                       name="middle_name" 
                                       class="quickbooks-input" 
                                       placeholder="Middle name"
                                       value="{{ old('middle_name') }}">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Last Name</label>
                                <input type="text" 
                                       name="last_name" 
                                       class="quickbooks-input" 
                                       placeholder="Doe"
                                       value="{{ old('last_name') }}"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Employee Number</label>
                                <input type="text" 
                                       name="employee_number" 
                                       class="quickbooks-input" 
                                       placeholder="EMP001"
                                       value="{{ old('employee_number', 'EMP' . date('Ymd') . rand(100, 999)) }}"
                                       required>
                                <p class="text-xs text-gray500 mt-1">Unique identifier for the employee</p>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Email Address</label>
                                <input type="email" 
                                       name="email" 
                                       class="quickbooks-input" 
                                       placeholder="john.doe@example.com"
                                       value="{{ old('email') }}"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Phone Number</label>
                                <input type="tel" 
                                       name="phone" 
                                       class="quickbooks-input" 
                                       placeholder="+260 97 123 4567"
                                       value="{{ old('phone') }}">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Date of Birth</label>
                                <input type="date" 
                                       name="date_of_birth" 
                                       class="quickbooks-input"
                                       value="{{ old('date_of_birth') }}"
                                       required>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Gender</label>
                                <select name="gender" class="quickbooks-select" required>
                                    <option value="">Select Gender</option>
                                    <option value="male" {{ old('gender') == 'male' ? 'selected' : '' }}>Male</option>
                                    <option value="female" {{ old('gender') == 'female' ? 'selected' : '' }}>Female</option>
                                    <option value="other" {{ old('gender') == 'other' ? 'selected' : '' }}>Other</option>
                                </select>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">National ID</label>
                                <input type="text" 
                                       name="national_id" 
                                       class="quickbooks-input" 
                                       placeholder="123456/78/9"
                                       value="{{ old('national_id') }}"
                                       required>
                            </div>
                        </div>
                    </div>

                    <!-- Employment Details Section -->
                    <div class="form-section">
                        <h3>Employment Details</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Hire Date</label>
                                <input type="date" 
                                       name="hire_date" 
                                       class="quickbooks-input"
                                       value="{{ old('hire_date', date('Y-m-d')) }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Employment Type</label>
                                <select name="employment_type" class="quickbooks-select" required>
                                    <option value="">Select Type</option>
                                    <option value="permanent" {{ old('employment_type') == 'permanent' ? 'selected' : '' }}>Permanent</option>
                                    <option value="contract" {{ old('employment_type') == 'contract' ? 'selected' : '' }}>Contract</option>
                                    <option value="temporary" {{ old('employment_type') == 'temporary' ? 'selected' : '' }}>Temporary</option>
                                    <option value="casual" {{ old('employment_type') == 'casual' ? 'selected' : '' }}>Casual</option>
                                </select>
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Department</label>
                                <input type="text" 
                                       name="department" 
                                       class="quickbooks-input" 
                                       placeholder="Finance"
                                       value="{{ old('department') }}">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Position/Title</label>
                                <input type="text" 
                                       name="position" 
                                       class="quickbooks-input" 
                                       placeholder="Accountant"
                                       value="{{ old('position') }}">
                            </div>
                        </div>
                    </div>

                    <!-- Earnings Section (Table-based like Petty Cash) -->
                      <!-- Earnings Section (Table-based like Petty Cash) -->
<div class="form-section">
    <div class="flex justify-between items-center mb-6">
        <h3>Earnings</h3>
        <button type="button" 
                id="addEarningRowBtn" 
                class="quickbooks-button-secondary flex items-center">
            <i class="fas fa-plus mr-2"></i> Add Row
        </button>
    </div>

    <!-- Earnings Table -->
   <!-- Earnings Table -->
    <div class="border border-gray200 rounded-lg overflow-hidden">
        <table class="quickbooks-table">
            <thead>
                <tr>
                    <th>Earning Type</th>
                    <th>Amount</th>
                    <th>Frequency</th>
                    <th>Effective Date</th>
                    <th>Taxable</th>
                    <th>Recurring</th>
                    <th>Actions</th>
                </tr>
            </thead>
            <tbody id="earningsContainer">
                <!-- Earnings will be added here dynamically -->
            </tbody>
            <tfoot class="bg-gray100">
                <tr>
                    <td colspan="2" class="text-right text-sm font-medium text-gray700 py-3">
                        <div>Total Gross Salary:</div>
                        <div class="text-xs text-gray500">Taxable Earnings:</div>
                        <div class="text-xs text-gray500">Non-Taxable Earnings:</div>
                    </td>
                    <td colspan="5" class="py-3">
                        <div class="text-lg font-semibold text-gray900" id="totalGrossSalary">0.00</div>
                        <div class="text-sm text-red-600" id="totalTaxableEarnings">0.00</div>
                        <div class="text-sm text-green-600" id="totalNonTaxableEarnings">0.00</div>
                    </td>
                </tr>
            </tfoot>
        </table>
    </div>

    <!-- Empty State Message -->
    <div id="noEarningsMessage" class="text-center py-8 text-gray500 hidden">
        <i class="fas fa-money-bill-wave text-3xl mb-3 text-gray300"></i>
        <p>No earnings added yet</p>
        <p class="text-sm mt-1">Click "Add Row" to add earnings items</p>
    </div>
</div>



    <!-- Empty State Message -->
    <div id="noEarningsMessage" class="text-center py-8 text-gray500 hidden">
        <i class="fas fa-money-bill-wave text-3xl mb-3 text-gray300"></i>
        <p>No earnings added yet</p>
        <p class="text-sm mt-1">Click "Add Row" to add earnings items</p>
    </div>
</div>

                    <!-- Statutory Information Section -->
                    <div class="form-section">
                        <h3>Statutory Information</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">NAPSA Number</label>
                                <input type="text" 
                                       name="napsa_number" 
                                       class="quickbooks-input" 
                                       placeholder="NPSA123456"
                                       value="{{ old('napsa_number') }}">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">NHIMA Number</label>
                                <input type="text" 
                                       name="nhima_number" 
                                       class="quickbooks-input" 
                                       placeholder="NHIMA123456"
                                       value="{{ old('nhima_number') }}">
                            </div>
                        </div>
                    </div>

                    <!-- Payment Details Section -->
                    <div class="form-section">
                        <h3>Payment Details</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2 required-field">Payment Method</label>
                                <select name="payment_method" class="quickbooks-select" required>
                                    <option value="">Select Method</option>
                                    <option value="bank" {{ old('payment_method') == 'bank' ? 'selected' : '' }}>Bank Transfer</option>
                                    <option value="cash" {{ old('payment_method') == 'cash' ? 'selected' : '' }}>Cash</option>
                                    <option value="mobile_money" {{ old('payment_method') == 'mobile_money' ? 'selected' : '' }}>Mobile Money</option>
                                </select>
                            </div>
                            <div id="bankFields" style="{{ old('payment_method') == 'bank' ? 'display: block;' : 'display: none;' }}">
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Name</label>
                                <input type="text" 
                                       name="bank_name" 
                                       class="quickbooks-input" 
                                       placeholder="Bank of Zambia"
                                       value="{{ old('bank_name') }}">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6" id="bankAccountFields" style="{{ old('payment_method') == 'bank' ? 'display: grid;' : 'display: none;' }}">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Account Number</label>
                                <input type="text" 
                                       name="bank_account_number" 
                                       class="quickbooks-input" 
                                       placeholder="1234567890"
                                       value="{{ old('bank_account_number') }}">
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Account Name</label>
                                <input type="text" 
                                       name="bank_account_name" 
                                       class="quickbooks-input" 
                                       placeholder="John Doe"
                                       value="{{ old('bank_account_name') }}">
                            </div>
                        </div>

                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6" id="bankBranchField" style="{{ old('payment_method') == 'bank' ? 'display: grid;' : 'display: none;' }}">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Bank Branch</label>
                                <input type="text" 
                                       name="bank_branch" 
                                       class="quickbooks-input" 
                                       placeholder="Main Branch"
                                       value="{{ old('bank_branch') }}">
                            </div>
                        </div>
                    </div>

                    <!-- Additional Information Section -->
                    <div class="form-section">
                        <h3>Additional Information</h3>
                        
                        <div class="mb-6">
                            <label class="block text-sm font-medium text-gray700 mb-2">Notes</label>
                            <textarea name="notes" 
                                      class="quickbooks-textarea" 
                                      rows="3"
                                      placeholder="Any additional notes about the employee...">{{ old('notes') }}</textarea>
                        </div>

                        <div>
                            <label class="block text-sm font-medium text-gray700 mb-4">Employee Status</label>
                            <div class="flex items-center space-x-6">
                                <div class="flex items-center">
                                    <input type="radio" id="active" name="is_active" value="1" {{ old('is_active', 1) ? 'checked' : '' }} class="quickbooks-checkbox mr-2">
                                    <label for="active" class="text-sm text-gray600">Active</label>
                                </div>
                                <div class="flex items-center">
                                    <input type="radio" id="inactive" name="is_active" value="0" {{ !old('is_active', 1) ? 'checked' : '' }} class="quickbooks-checkbox mr-2">
                                    <label for="inactive" class="text-sm text-gray600">Inactive</label>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Hidden earnings inputs will be added here by JavaScript -->
                    <div id="earningsData"></div>

                    <!-- Form Actions -->
                    <div class="border-t border-gray200 p-6">
                        <div class="flex justify-end space-x-3">
                            <a href="{{ route('admin.accounting.payroll.employees.index') }}" 
                               class="quickbooks-button-secondary">
                                Cancel
                            </a>
                            <button type="submit" 
                                    class="quickbooks-button-primary">
                                Create Employee
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    <!-- Earning Type Modal Template -->
    <template id="earningTypeModalTemplate">
        <div class="fixed inset-0 bg-gray-900 bg-opacity-50 flex items-center justify-center z-50 p-4">
            <div class="bg-white rounded-lg shadow-xl w-full max-w-2xl">
                <div class="px-6 py-4 border-b border-gray-200">
                    <h3 class="text-lg font-medium text-gray-900">Select Earning Type</h3>
                </div>
                <div class="px-6 py-4 max-h-96 overflow-y-auto">
                    <div class="grid grid-cols-1 md:grid-cols-2 gap-3">
                        <div id="earningTypeOptions">
                            <!-- Earning types will be populated here -->
                        </div>
                    </div>
                </div>
                <div class="px-6 py-4 border-t border-gray-200 flex justify-end">
                    <button type="button" class="quickbooks-button-secondary close-modal-btn">
                        Cancel
                    </button>
                </div>
            </div>
        </div>
    </template>

 <script>
    document.addEventListener('DOMContentLoaded', function() {
        // Get earning types from server
        let earningTypes = [];
        
        // Fetch earning types
        fetchEarningTypes();
        
        // Earnings management
        const earningsContainer = document.getElementById('earningsContainer');
        const noEarningsMessage = document.getElementById('noEarningsMessage');
        const addEarningRowBtn = document.getElementById('addEarningRowBtn');
        const earningsData = document.getElementById('earningsData');
        
        let earningCounter = 0;
        
        // Fetch earning types from server
        async function fetchEarningTypes() {
            try {
                const response = await fetch('{{ route("admin.accounting.payroll.api.earning-types") }}');
                if (response.ok) {
                    earningTypes = await response.json();
                    // Initialize with two rows after fetching earning types
                    addInitialRows();
                }
            } catch (error) {
                console.error('Failed to fetch earning types:', error);
                // Still try to add initial rows even if fetch fails
                addInitialRows();
            }
        }
        
        // Add two initial rows
        function addInitialRows() {
            if (earningTypes.length > 0) {
                // Add first row with Basic Salary (if available)
                const basicType = earningTypes.find(type => type.type === 'basic');
                if (basicType) {
                    addEarningRow(basicType.id, basicType.name, basicType.code, basicType.type_label, basicType.is_taxable);
                } else {
                    // Use first earning type if basic not found
                    addEarningRow(earningTypes[0].id, earningTypes[0].name, earningTypes[0].code, earningTypes[0].type_label, earningTypes[0].is_taxable);
                }
                
                // Add second row with Allowance (if available)
                const allowanceType = earningTypes.find(type => type.type === 'allowance');
                if (allowanceType) {
                    addEarningRow(allowanceType.id, allowanceType.name, allowanceType.code, allowanceType.type_label, allowanceType.is_taxable);
                } else if (earningTypes.length > 1) {
                    // Use second earning type if allowance not found
                    addEarningRow(earningTypes[1].id, earningTypes[1].name, earningTypes[1].code, earningTypes[1].type_label, earningTypes[1].is_taxable);
                } else {
                    // Add another row with the first type if only one type exists
                    addEarningRow(earningTypes[0].id, earningTypes[0].name, earningTypes[0].code, earningTypes[0].type_label, earningTypes[0].is_taxable);
                }
            } else {
                // Add two empty rows if no earning types available
                addEarningRow();
                addEarningRow();
            }
        }
        
        // Add earning row button event
        addEarningRowBtn.addEventListener('click', function() {
            addEarningRow();
        });
        
        function addEarningRow(typeId = '', typeName = '', typeCode = '', typeLabel = '', isTaxable = false) {
            // Hide "no earnings" message
            noEarningsMessage.classList.add('hidden');
            
            earningCounter++;
            const rowId = `earningRow${earningCounter}`;
            
            // Generate unique name for taxable checkbox
            const taxableCheckboxId = `taxable${earningCounter}`;
            // Generate unique name for recurring checkbox
            const recurringCheckboxId = `recurring${earningCounter}`;
            
            const row = document.createElement('tr');
            row.id = rowId;
            row.innerHTML = `
                <td>
                    <select name="earnings[${earningCounter}][earning_type_id]" 
                            class="quickbooks-select w-full earning-type-select"
                            required
                            onchange="updateEarningTypeDisplay(this, ${earningCounter})">
                        <option value="">Select Earning Type</option>
                        ${earningTypes.map(type => `
                            <option value="${type.id}" 
                                    data-type-name="${type.name}"
                                    data-type-code="${type.code}"
                                    data-type-label="${type.type_label}"
                                    data-is-taxable="${type.is_taxable}"
                                    ${typeId === type.id ? 'selected' : ''}>
                                ${type.name} (${type.code})
                            </option>
                        `).join('')}
                    </select>
                    <div class="mt-2" id="earningTypeDisplay${earningCounter}">
                        ${typeLabel ? `
                            <span class="text-xs px-2 py-1 rounded-full ${typeLabel === 'Basic Salary' ? 'bg-blue-100 text-blue-800' : 
                                                                           typeLabel === 'Allowance' ? 'bg-green-100 text-green-800' :
                                                                           typeLabel === 'Bonus' ? 'bg-yellow-100 text-yellow-800' :
                                                                           typeLabel === 'Overtime' ? 'bg-orange-100 text-orange-800' :
                                                                           'bg-gray-100 text-gray-800'}">
                                ${typeLabel}
                            </span>
                        ` : ''}
                    </div>
                    <input type="hidden" name="earnings[${earningCounter}][earning_type_name]" value="${typeName}">
                    <input type="hidden" name="earnings[${earningCounter}][earning_type_code]" value="${typeCode}">
                </td>
                <td>
                    <div class="relative">
                        <input type="number" 
                               name="earnings[${earningCounter}][amount]" 
                               class="quickbooks-input w-full pl-8 earning-amount"
                               placeholder="0.00"
                               step="0.01"
                               min="0"
                               required
                               oninput="validateEarningField(this); calculateEarningsTotal()">
                        <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 text-sm">K</span>
                    </div>
                </td>
                <td>
                    <select name="earnings[${earningCounter}][frequency]" 
                            class="quickbooks-select w-full earning-frequency"
                            required
                            onchange="validateEarningField(this); calculateEarningsTotal()">
                        <option value="monthly">Monthly</option>
                        <option value="quarterly">Quarterly</option>
                        <option value="annually">Annually</option>
                        <option value="one-time">One Time</option>
                    </select>
                </td>
                <td>
                    <input type="date" 
                           name="earnings[${earningCounter}][effective_date]" 
                           class="quickbooks-input w-full earning-effective-date"
                           value="${new Date().toISOString().split('T')[0]}"
                           required
                           onchange="validateEarningField(this)">
                </td>
                <td class="text-center">
                    <div class="flex items-center justify-center">
                        <input type="checkbox" 
                               id="${taxableCheckboxId}" 
                               name="earnings[${earningCounter}][is_taxable]" 
                               value="1"
                               class="quickbooks-checkbox taxable-checkbox"
                               ${isTaxable ? 'checked' : ''}
                               onchange="updateTaxableDisplay(this, ${earningCounter}); calculateEarningsTotal()">
                        <label for="${taxableCheckboxId}" class="sr-only">Taxable</label>
                    </div>
                    <div class="mt-1" id="taxableDisplay${earningCounter}">
                        ${isTaxable ? `
                            <span class="text-xs px-2 py-1 rounded-full bg-red-100 text-red-800">
                                Taxable
                            </span>
                        ` : `
                            <span class="text-xs px-2 py-1 rounded-full bg-green-100 text-green-800">
                                Non-Taxable
                            </span>
                        `}
                    </div>
                </td>
                <td class="text-center">
                    <div class="flex items-center justify-center">
                        <input type="checkbox" 
                               id="${recurringCheckboxId}" 
                               name="earnings[${earningCounter}][is_recurring]" 
                               value="1"
                               class="quickbooks-checkbox recurring-checkbox"
                               checked
                               onchange="updateRecurringDisplay(this, ${earningCounter})">
                        <label for="${recurringCheckboxId}" class="sr-only">Recurring</label>
                    </div>
                    <div class="mt-1" id="recurringDisplay${earningCounter}">
                        <span class="text-xs px-2 py-1 rounded-full bg-blue-100 text-blue-800">
                            Recurring
                        </span>
                    </div>
                </td>
                <td class="text-center">
                    <button type="button" 
                            onclick="removeEarningRow('${rowId}')"
                            class="quickbooks-button-secondary p-2 text-danger hover:text-danger/90"
                            title="Remove this earning">
                        <i class="fas fa-trash"></i>
                    </button>
                </td>
            `;
            
            earningsContainer.appendChild(row);
            
            // Initialize validation for pre-selected type
            if (typeId) {
                const selectElement = row.querySelector('.earning-type-select');
                validateEarningField(selectElement);
            }
            
            calculateEarningsTotal();
        }
        
        window.updateEarningTypeDisplay = function(selectElement, rowNumber) {
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const displayDiv = document.getElementById(`earningTypeDisplay${rowNumber}`);
            const typeNameInput = selectElement.closest('tr').querySelector('input[name$="[earning_type_name]"]');
            const typeCodeInput = selectElement.closest('tr').querySelector('input[name$="[earning_type_code]"]');
            const taxableCheckbox = selectElement.closest('tr').querySelector('.taxable-checkbox');
            
            if (selectedOption.value) {
                const typeName = selectedOption.dataset.typeName;
                const typeCode = selectedOption.dataset.typeCode;
                const typeLabel = selectedOption.dataset.typeLabel;
                const isTaxable = selectedOption.dataset.isTaxable === 'true';
                
                // Update display
                displayDiv.innerHTML = `
                    <span class="text-xs px-2 py-1 rounded-full ${typeLabel === 'Basic Salary' ? 'bg-blue-100 text-blue-800' : 
                                                                   typeLabel === 'Allowance' ? 'bg-green-100 text-green-800' :
                                                                   typeLabel === 'Bonus' ? 'bg-yellow-100 text-yellow-800' :
                                                                   typeLabel === 'Overtime' ? 'bg-orange-100 text-orange-800' :
                                                                   'bg-gray-100 text-gray-800'}">
                        ${typeLabel}
                    </span>
                `;
                
                // Update hidden inputs
                typeNameInput.value = typeName;
                typeCodeInput.value = typeCode;
                
                // Update taxable checkbox based on earning type
                taxableCheckbox.checked = isTaxable;
                updateTaxableDisplay(taxableCheckbox, rowNumber);
                
                // Auto-fill amount for basic salary if empty
                if (typeLabel === 'Basic Salary' || typeName.toLowerCase().includes('basic')) {
                    const amountInput = selectElement.closest('tr').querySelector('.earning-amount');
                    if (!amountInput.value || amountInput.value === '0') {
                        amountInput.value = '5000.00'; // Default basic salary
                        validateEarningField(amountInput);
                        calculateEarningsTotal();
                    }
                }
            } else {
                // Clear display if no type selected
                displayDiv.innerHTML = '';
                typeNameInput.value = '';
                typeCodeInput.value = '';
                taxableCheckbox.checked = false;
                updateTaxableDisplay(taxableCheckbox, rowNumber);
            }
            
            validateEarningField(selectElement);
        };
        
        window.updateTaxableDisplay = function(checkboxElement, rowNumber) {
            const displayDiv = document.getElementById(`taxableDisplay${rowNumber}`);
            const isTaxable = checkboxElement.checked;
            
            displayDiv.innerHTML = isTaxable ? `
                <span class="text-xs px-2 py-1 rounded-full bg-red-100 text-red-800">
                    Taxable
                </span>
            ` : `
                <span class="text-xs px-2 py-1 rounded-full bg-green-100 text-green-800">
                    Non-Taxable
                </span>
            `;
            
            calculateEarningsTotal();
        };
        
        window.updateRecurringDisplay = function(checkboxElement, rowNumber) {
            const displayDiv = document.getElementById(`recurringDisplay${rowNumber}`);
            const isRecurring = checkboxElement.checked;
            
            displayDiv.innerHTML = isRecurring ? `
                <span class="text-xs px-2 py-1 rounded-full bg-blue-100 text-blue-800">
                    Recurring
                </span>
            ` : `
                <span class="text-xs px-2 py-1 rounded-full bg-yellow-100 text-yellow-800">
                    One-time
                </span>
            `;
        };
        
        window.removeEarningRow = function(rowId) {
            const row = document.getElementById(rowId);
            if (row) {
                row.remove();
                calculateEarningsTotal();
                if (earningsContainer.children.length === 0) {
                    noEarningsMessage.classList.remove('hidden');
                }
            }
        };
        
        window.validateEarningField = function(field) {
            field.classList.remove('is-invalid', 'is-valid');
            
            if (field.tagName === 'SELECT') {
                if (field.value === '') {
                    field.classList.add('is-invalid');
                } else {
                    field.classList.add('is-valid');
                }
            } else if (field.type === 'number') {
                const value = parseFloat(field.value);
                if (isNaN(value) || value < 0) {
                    field.classList.add('is-invalid');
                } else {
                    field.classList.add('is-valid');
                }
            } else if (field.type === 'date') {
                if (!field.value) {
                    field.classList.add('is-invalid');
                } else {
                    field.classList.add('is-valid');
                }
            } else if (field.type === 'text') {
                if (field.required && !field.value.trim()) {
                    field.classList.add('is-invalid');
                } else {
                    field.classList.add('is-valid');
                }
            } else if (field.type === 'checkbox') {
                // Checkboxes are always valid
                field.classList.add('is-valid');
            }
        };
        
        window.calculateEarningsTotal = function() {
            let totalGross = 0;
            let totalTaxable = 0;
            let totalNonTaxable = 0;
            
            document.querySelectorAll('[id^="earningRow"]').forEach(row => {
                const amountInput = row.querySelector('.earning-amount');
                const frequencySelect = row.querySelector('.earning-frequency');
                const taxableCheckbox = row.querySelector('.taxable-checkbox');
                const recurringCheckbox = row.querySelector('.recurring-checkbox');
                
                const amount = parseFloat(amountInput.value) || 0;
                const frequency = frequencySelect.value;
                const isTaxable = taxableCheckbox.checked;
                const isRecurring = recurringCheckbox.checked;
                
                // Calculate monthly amount
                let monthlyAmount = 0;
                if (isRecurring) {
                    monthlyAmount = amount;
                    if (frequency === 'quarterly') monthlyAmount = amount / 3;
                    if (frequency === 'annually') monthlyAmount = amount / 12;
                    // One-time recurring? That doesn't make sense, so treat as monthly
                    if (frequency === 'one-time') monthlyAmount = amount;
                }
                // Non-recurring (one-time) earnings are not included in monthly total
                
                totalGross += monthlyAmount;
                if (isTaxable) {
                    totalTaxable += monthlyAmount;
                } else {
                    totalNonTaxable += monthlyAmount;
                }
            });
            
            document.getElementById('totalGrossSalary').textContent = formatCurrency(totalGross);
            document.getElementById('totalTaxableEarnings').textContent = formatCurrency(totalTaxable);
            document.getElementById('totalNonTaxableEarnings').textContent = formatCurrency(totalNonTaxable);
        };
        
        function formatCurrency(amount) {
            return 'K' + amount.toFixed(2).replace(/\d(?=(\d{3})+\.)/g, '$&,');
        }
        
        // Payment method toggle
        const paymentMethodSelect = document.querySelector('select[name="payment_method"]');
        const bankFields = document.getElementById('bankFields');
        const bankAccountFields = document.getElementById('bankAccountFields');
        const bankBranchField = document.getElementById('bankBranchField');
        
        function toggleBankFields() {
            const isBank = paymentMethodSelect.value === 'bank';
            
            bankFields.style.display = isBank ? 'block' : 'none';
            bankAccountFields.style.display = isBank ? 'grid' : 'none';
            bankBranchField.style.display = isBank ? 'grid' : 'none';
            
            // Add/remove required attribute
            const bankInputs = document.querySelectorAll('#bankFields input, #bankAccountFields input, #bankBranchField input');
            bankInputs.forEach(input => {
                if (isBank) {
                    input.setAttribute('required', 'required');
                } else {
                    input.removeAttribute('required');
                }
            });
        }
        
        if (paymentMethodSelect) {
            paymentMethodSelect.addEventListener('change', toggleBankFields);
            // Initial check
            toggleBankFields();
        }
        
        // Form validation
        const form = document.getElementById('employeeForm');
        form.addEventListener('submit', function(e) {
            let isValid = true;
            let errorMessage = '';
            
            // Validate required fields
            const requiredFields = form.querySelectorAll('[required]');
            requiredFields.forEach(field => {
                if (!field.value.trim()) {
                    isValid = false;
                    field.classList.add('is-invalid');
                    if (!errorMessage) {
                        errorMessage = 'Please fill in all required fields.';
                    }
                } else {
                    field.classList.remove('is-invalid');
                }
            });
            
            // Validate at least one earning
            const hasEarnings = earningsContainer.children.length > 0;
            if (!hasEarnings) {
                isValid = false;
                errorMessage = 'Please add at least one earning for the employee.';
            }
            
            // Validate all earning rows have required fields
            document.querySelectorAll('.earning-type-select').forEach((select, index) => {
                if (!select.value) {
                    isValid = false;
                    select.classList.add('is-invalid');
                    errorMessage = 'Please select an earning type for all rows.';
                }
                
                const amountInput = select.closest('tr').querySelector('.earning-amount');
                if (!amountInput.value || parseFloat(amountInput.value) <= 0) {
                    isValid = false;
                    amountInput.classList.add('is-invalid');
                    errorMessage = 'Please enter a valid amount for all earnings.';
                }
                
                const dateInput = select.closest('tr').querySelector('.earning-effective-date');
                if (!dateInput.value) {
                    isValid = false;
                    dateInput.classList.add('is-invalid');
                    errorMessage = 'Please enter an effective date for all earnings.';
                }
            });
            
            // Validate email format
            const emailField = form.querySelector('input[type="email"]');
            if (emailField && emailField.value) {
                const emailRegex = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
                if (!emailRegex.test(emailField.value)) {
                    isValid = false;
                    emailField.classList.add('is-invalid');
                    errorMessage = 'Please enter a valid email address.';
                }
            }
            
            if (!isValid) {
                e.preventDefault();
                showNotification(errorMessage, 'error');
                
                // Scroll to first error
                const firstError = document.querySelector('.is-invalid');
                if (firstError) {
                    firstError.scrollIntoView({ behavior: 'smooth', block: 'center' });
                    firstError.focus();
                }
            }
        });
    });
    
    function showNotification(message, type = 'info') {
        const existing = document.querySelector('.quickbooks-notification');
        if (existing) existing.remove();

        const colors = {
            success: 'border-success bg-green-50',
            error: 'border-danger bg-red-50',
            info: 'border-primary bg-blue-50',
            warning: 'border-warning bg-yellow-50'
        };

        const icons = {
            success: 'fa-check-circle text-success',
            error: 'fa-exclamation-circle text-danger',
            info: 'fa-info-circle text-primary',
            warning: 'fa-exclamation-triangle text-warning'
        };

        const notification = document.createElement('div');
        notification.className = `quickbooks-notification fixed top-4 right-4 bg-white border-l-4 ${colors[type]} p-4 rounded shadow-lg z-50 flex items-center max-w-md animate-fade-in`;
        notification.innerHTML = `
            <i class="fas ${icons[type]} mr-3 text-lg"></i>
            <span class="text-gray-900">${message}</span>
            <button onclick="this.parentElement.remove()" class="ml-auto text-gray-500 hover:text-gray-700">
                <i class="fas fa-times"></i>
            </button>
        `;

        document.body.appendChild(notification);

        setTimeout(() => {
            notification.classList.add('animate-fade-out');
            setTimeout(() => {
                notification.remove();
            }, 300);
        }, 5000);
    }

    // Add animation styles
    const style = document.createElement('style');
    style.textContent = `
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        @keyframes fadeOut {
            from { opacity: 1; transform: translateY(0); }
            to { opacity: 0; transform: translateY(-10px); }
        }
        .animate-fade-in {
            animation: fadeIn 0.3s ease-out;
        }
        .animate-fade-out {
            animation: fadeOut 0.3s ease-in;
        }
    `;
    document.head.appendChild(style);
</script>
</body>
</html>