<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>NHIMA Settings | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        /* Same styles as other views */
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'NHIMA Settings', 'subtitle' => 'Configure NHIMA health insurance contribution rates'])

        <main class="p-6">
            <!-- Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- NHIMA Configuration Form -->
            <div class="bg-white rounded-lg border border-gray200 p-6">
                <form id="nhimaForm" action="{{ route('admin.accounting.payroll.nhima.store') }}" method="POST">
                    @csrf
                    
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-6">NHIMA Contribution Settings</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Financial Year</label>
                                <select id="financial_year_id" name="financial_year_id" class="quickbooks-input w-full" required>
                                    <option value="">Select Financial Year</option>
                                    @foreach($financialYears as $year)
                                        <option value="{{ $year->id }}" 
                                                {{ ($nhimaSettings && $nhimaSettings->financial_year_id == $year->id) ? 'selected' : '' }}
                                                {{ ($currentFinancialYear && $year->id == $currentFinancialYear->id) ? 'selected' : '' }}>
                                            {{ $year->year }} ({{ $year->start_date->format('M Y') }} - {{ $year->end_date->format('M Y') }})
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Year Label</label>
                                <input type="text" 
                                       id="year" 
                                       name="year" 
                                       class="quickbooks-input w-full" 
                                       placeholder="e.g., 2024-2025"
                                       value="{{ $nhimaSettings->year ?? '' }}"
                                       required>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Effective From</label>
                                <input type="date" 
                                       id="start_date" 
                                       name="start_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $nhimaSettings->start_date ?? '' }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">Effective To</label>
                                <input type="date" 
                                       id="end_date" 
                                       name="end_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $nhimaSettings->end_date ?? '' }}"
                                       required>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- NHIMA Rate -->
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-6">NHIMA Contribution Rate</h3>
                        
                        <div class="bg-gray50 border border-gray200 rounded-lg p-6 mb-6">
                            <div class="max-w-md">
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Monthly Contribution Rate (%)
                                    <span class="text-danger">*</span>
                                </label>
                                <div class="relative">
                                    <input type="number" 
                                           id="monthly_rate" 
                                           name="monthly_rate" 
                                           class="quickbooks-input w-full pr-8" 
                                           step="0.01" 
                                           min="0" 
                                           max="100" 
                                           value="{{ $nhimaSettings->monthly_rate ?? '1.00' }}"
                                           required>
                                    <span class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray500">%</span>
                                </div>
                                <p class="text-sm text-gray500 mt-2">
                                    Current Zambian NHIMA rate: 1% of basic salary (capped at K3,750 monthly contribution)
                                </p>
                            </div>
                        </div>
                        
                        <div class="modal-info">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <p class="font-medium mb-1">NHIMA Contribution Calculation:</p>
                                <p class="text-sm">Monthly NHIMA = Basic Salary × NHIMA Rate (up to maximum of K3,750)</p>
                                <p class="text-sm mt-1">Example: For basic salary of K10,000: K10,000 × 1% = K100</p>
                                <p class="text-sm mt-1">Note: Employee contributes 1% and employer contributes 1% of basic salary</p>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Form Actions -->
                    <div class="flex justify-between items-center">
                        <div>
                            <a href="{{ route('admin.accounting.payroll.index') }}" 
                               class="text-primary hover:text-primary/90 font-medium text-sm">
                                <i class="fas fa-arrow-left mr-1"></i>Back to Payroll
                            </a>
                        </div>
                        <div class="flex space-x-3">
                            <button type="button" 
                                    onclick="resetForm()" 
                                    class="px-4 py-2 border border-gray300 rounded text-sm font-medium text-gray700 hover:bg-gray50">
                                Reset to Defaults
                            </button>
                            <button type="submit" 
                                    class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                Save NHIMA Settings
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Current NHIMA Settings Display -->
            @if($nhimaSettings)
                <div class="mt-8 bg-white rounded-lg border border-gray200">
                    <div class="border-b border-gray200 p-6">
                        <h3 class="text-lg font-semibold text-gray900">Current NHIMA Settings for {{ $nhimaSettings->year }}</h3>
                        <p class="text-gray600 text-sm">Effective: {{ $nhimaSettings->start_date->format('M d, Y') }} to {{ $nhimaSettings->end_date->format('M d, Y') }}</p>
                    </div>

                    <div class="p-6">
                        <div class="bg-gray50 rounded-lg p-6">
                            <h4 class="text-lg font-semibold text-gray900 mb-4">Contribution Details</h4>
                            <div class="space-y-4">
                                <div class="flex justify-between items-center">
                                    <span class="text-gray600">Monthly Contribution Rate:</span>
                                    <span class="font-semibold text-gray900">{{ $nhimaSettings->monthly_rate }}%</span>
                                </div>
                                <div class="flex justify-between items-center">
                                    <span class="text-gray600">Annual Rate:</span>
                                    <span class="font-semibold text-gray900">{{ $nhimaSettings->annual_rate }}%</span>
                                </div>
                                <div class="border-t border-gray200 pt-4">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                        <div class="bg-white rounded p-4">
                                            <h5 class="text-sm font-medium text-gray700 mb-2">Employee Contribution</h5>
                                            <p class="text-sm text-gray600">1% of basic salary × Monthly Rate</p>
                                            <p class="text-sm text-gray500 mt-1">Max: K3,750 per month</p>
                                        </div>
                                        <div class="bg-white rounded p-4">
                                            <h5 class="text-sm font-medium text-gray700 mb-2">Employer Contribution</h5>
                                            <p class="text-sm text-gray600">1% of basic salary × Monthly Rate</p>
                                            <p class="text-sm text-gray500 mt-1">Max: K3,750 per month</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            @endif
        </main>
    </div>

    <script>
        // Load settings when financial year changes
        document.getElementById('financial_year_id').addEventListener('change', function() {
            const yearId = this.value;
            
            if (!yearId) return;
            
            fetch(`{{ route('admin.accounting.payroll.get-settings') }}?financial_year_id=${yearId}&setting_type=nhima`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.setting) {
                        // Populate form with existing settings
                        document.getElementById('year').value = data.setting.year;
                        document.getElementById('start_date').value = data.setting.start_date;
                        document.getElementById('end_date').value = data.setting.end_date;
                        document.getElementById('monthly_rate').value = data.setting.monthly_rate;
                        
                        showNotification('Loaded existing NHIMA settings', 'success');
                    } else {
                        // Set default values for new year
                        const selectedOption = this.options[this.selectedIndex];
                        const yearText = selectedOption.text.split(' ')[0];
                        document.getElementById('year').value = yearText;
                        
                        // Reset to default Zambian NHIMA rate
                        resetForm();
                    }
                })
                .catch(error => {
                    console.error('Error loading settings:', error);
                    showNotification('Failed to load settings', 'error');
                });
        });

        function resetForm() {
            if (confirm('Are you sure you want to reset to default Zambian NHIMA rate?')) {
                document.getElementById('monthly_rate').value = '1.00';
                showNotification('Reset to default Zambian NHIMA rate', 'info');
            }
        }

        function showNotification(message, type = 'info') {
            // Similar notification function as in other views
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Initialize on page load
        document.addEventListener('DOMContentLoaded', function() {
            // Set default dates if not already set
            const startDateInput = document.getElementById('start_date');
            const endDateInput = document.getElementById('end_date');
            
            if (startDateInput && !startDateInput.value) {
                startDateInput.value = '{{ date("Y") }}-01-01';
            }
            
            if (endDateInput && !endDateInput.value) {
                endDateInput.value = '{{ date("Y") }}-12-31';
            }
        });
    </script>
</body>
</html>