<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>PAYE Tax Bands | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <style>
        .quickbooks-table { border-collapse: separate; border-spacing: 0; width: 100%; }
        .quickbooks-table th { background-color: #F3F2F1; color: #323130; font-weight: 600; font-size: 13px; text-transform: uppercase; padding: 12px 16px; border-bottom: 1px solid #EDEBE9; text-align: left; }
        .quickbooks-table td { padding: 12px 16px; border-bottom: 1px solid #EDEBE9; font-size: 14px; color: #323130; }
        .quickbooks-table tbody tr:hover { background-color: #F3F2F1; }
        .quickbooks-input { border: 1px solid #8C8C8C; border-radius: 4px; padding: 6px 12px; font-size: 14px; }
        .quickbooks-input:focus { outline: 2px solid #146CAA; outline-offset: -1px; border-color: #146CAA; }
        .quickbooks-badge { display: inline-flex; align-items: center; padding: 4px 8px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .action-btn { width: 32px; height: 32px; border-radius: 4px; display: inline-flex; align-items: center; justify-content: center; margin: 0 2px; transition: all 0.2s; }
        .action-btn:hover { transform: translateY(-1px); }
        .action-btn-edit { background-color: #E1F0FA; color: #146CAA; }
        .action-btn-edit:hover { background-color: #146CAA; color: white; }
        .action-btn-delete { background-color: #FEE; color: #D83B01; }
        .action-btn-delete:hover { background-color: #D83B01; color: white; }
        .modal-backdrop { background-color: rgba(0, 0, 0, 0.5); }
        .modal-content { max-height: 90vh; overflow-y: auto; }
        .modal-divider { border-top: 1px solid #EDEBE9; margin: 20px 0; }
        .modal-info { background-color: #F3F2F1; border: 1px solid #EDEBE9; border-radius: 4px; padding: 12px; margin-top: 8px; font-size: 13px; color: #605E5C; }
        .modal-info i { color: #605E5C; margin-right: 8px; }
        .tax-band-row { transition: all 0.2s; }
        .tax-band-row:hover { background-color: #F9FAFB; }
        .tax-band-highlight { background-color: #E1F0FA !important; animation: highlight 2s ease-out; }
        @keyframes highlight {
            0% { background-color: #E1F0FA; }
            100% { background-color: transparent; }
        }
        .editable-cell { cursor: pointer; position: relative; }
        .editable-cell:hover { background-color: #F3F2F1; }
        .editable-input { position: absolute; top: 0; left: 0; width: 100%; height: 100%; border: 2px solid #146CAA; border-radius: 4px; padding: 8px; font-size: 14px; z-index: 10; }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'PAYE Tax Bands', 'subtitle' => 'Configure PAYE tax bands for different financial years'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Current Settings Info -->
            <div class="mb-8 bg-white rounded-lg border border-gray200 p-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h2 class="text-xl font-semibold text-gray900">{{ $company->name }}</h2>
                        <p class="text-gray600 text-sm mt-1">
                            Current Financial Year: 
                            <span class="font-medium">{{ $currentFinancialYear->year ?? 'Not Set' }}</span>
                            @if($currentFinancialYear)
                                <span class="text-gray500 ml-2">
                                    ({{ $currentFinancialYear->start_date->format('M d, Y') }} - {{ $currentFinancialYear->end_date->format('M d, Y') }})
                                </span>
                            @endif
                        </p>
                    </div>
                    <div class="mt-4 md:mt-0">
                        @if($payeSettings)
                            <span class="quickbooks-badge bg-green-100 text-green-800">{{ $taxBands->count() }} Bands Configured</span>
                        @else
                            <span class="quickbooks-badge bg-yellow-100 text-yellow-800">Not Configured</span>
                        @endif
                    </div>
                </div>
            </div>

            <!-- Configuration Form -->
            <div class="bg-white rounded-lg border border-gray200 p-6 mb-8">
                <form id="payeForm" action="{{ route('admin.accounting.payroll.paye.store') }}" method="POST">
                    @csrf
                    
                    <div class="mb-8">
                        <h3 class="text-lg font-semibold text-gray900 mb-6">Select Financial Year</h3>
                        
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Financial Year
                                    <span class="text-danger">*</span>
                                </label>
                                <select id="financial_year_id" name="financial_year_id" class="quickbooks-input w-full" required onchange="loadPayeSettings()">
                                    <option value="">Select Financial Year</option>
                                    @foreach($financialYears as $year)
                                        <option value="{{ $year->id }}" 
                                                {{ ($currentFinancialYear && $year->id == $currentFinancialYear->id) ? 'selected' : '' }}
                                                data-start-date="{{ $year->start_date->format('Y-m-d') }}"
                                                data-end-date="{{ $year->end_date->format('Y-m-d') }}">
                                            {{ $year->year }} ({{ $year->start_date->format('M Y') }} - {{ $year->end_date->format('M Y') }})
                                        </option>
                                    @endforeach
                                </select>
                            </div>
                            
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Year Label
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="text" 
                                       id="year" 
                                       name="year" 
                                       class="quickbooks-input w-full" 
                                       placeholder="e.g., 2025"
                                       value="{{ $payeSettings->year ?? '' }}"
                                       required>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-2 gap-4 mt-4">
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Effective From
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="date" 
                                       id="start_date" 
                                       name="start_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $payeSettings->start_date ?? '' }}"
                                       required>
                            </div>
                            <div>
                                <label class="block text-sm font-medium text-gray700 mb-2">
                                    Effective To
                                    <span class="text-danger">*</span>
                                </label>
                                <input type="date" 
                                       id="end_date" 
                                       name="end_date" 
                                       class="quickbooks-input w-full" 
                                       value="{{ $payeSettings->end_date ?? '' }}"
                                       required>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Tax Bands Configuration -->
                    <div>
                        <div class="flex items-center justify-between mb-6">
                            <div>
                                <h3 class="text-lg font-semibold text-gray900">PAYE Tax Bands</h3>
                                <p class="text-gray600 text-sm">Configure tax bands for the selected financial year</p>
                            </div>
                            <div class="flex space-x-2">
                                <button type="button" 
                                        onclick="addTaxBand()" 
                                        class="bg-primary text-white px-4 py-2 rounded text-sm font-medium hover:bg-primary/90 flex items-center">
                                    <i class="fas fa-plus mr-2"></i>Add Tax Band
                                </button>
                                <button type="button" 
                                        onclick="resetToDefaultBands()" 
                                        class="bg-gray-100 text-gray-700 px-4 py-2 rounded text-sm font-medium hover:bg-gray-200 flex items-center">
                                    <i class="fas fa-undo mr-2"></i>Reset to Defaults
                                </button>
                            </div>
                        </div>

                        <div class="overflow-x-auto mb-4">
                            <table class="quickbooks-table" id="taxBandsTable">
                                <thead>
                                    <tr>
                                        <th class="w-16">Band #</th>
                                        <th>Description</th>
                                        <th>Monthly Range (ZMW)</th>
                                        <th>Annual Range (ZMW)</th>
                                        <th class="w-24">Tax Rate %</th>
                                        <th class="w-24">Actions</th>
                                    </tr>
                                </thead>
                                <tbody id="taxBandsBody">
                                    <!-- Tax bands will be loaded here -->
                                </tbody>
                            </table>
                        </div>

                        <div class="modal-info">
                            <i class="fas fa-info-circle"></i>
                            <div>
                                <p class="font-medium mb-1">Important Notes:</p>
                                <ul class="text-sm list-disc pl-4 space-y-1">
                                    <li>Tax bands must be in ascending order (from lowest to highest income)</li>
                                    <li>For the highest tax band, use 999999999.99 for the "to" amount</li>
                                    <li>Monthly and annual ranges must be consistent (annual = monthly × 12)</li>
                                    <li>Tax rates must be between 0% and 100%</li>
                                </ul>
                            </div>
                        </div>
                    </div>

                    <div class="modal-divider"></div>

                    <!-- Form Actions -->
                    <div class="flex justify-between items-center">
                        <div>
                            <a href="{{ route('admin.accounting.payroll.index') }}" 
                               class="text-primary hover:text-primary/90 font-medium text-sm">
                                <i class="fas fa-arrow-left mr-1"></i>Back to Payroll
                            </a>
                        </div>
                        <div class="flex space-x-3">
                            <button type="button" 
                                    onclick="validateAndSubmit()" 
                                    class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                                <i class="fas fa-save mr-2"></i>Save Tax Bands
                            </button>
                        </div>
                    </div>
                </form>
            </div>

            <!-- Current Tax Bands Display -->
            @if($payeSettings && $taxBands->count() > 0)
                <div class="bg-white rounded-lg border border-gray200 mt-8">
                    <div class="border-b border-gray200 p-6">
                        <div class="flex items-center justify-between">
                            <div>
                                <h3 class="text-lg font-semibold text-gray900">Current Tax Bands for {{ $payeSettings->year }}</h3>
                                <p class="text-gray600 text-sm">Effective: {{ $payeSettings->start_date->format('M d, Y') }} to {{ $payeSettings->end_date->format('M d, Y') }}</p>
                            </div>
                            <div class="flex space-x-2">
                                <button onclick="exportTaxBands()" 
                                        class="text-primary hover:text-primary/90 text-sm font-medium flex items-center">
                                    <i class="fas fa-download mr-1"></i> Export
                                </button>
                                <button onclick="printTaxBands()" 
                                        class="text-primary hover:text-primary/90 text-sm font-medium flex items-center">
                                    <i class="fas fa-print mr-1"></i> Print
                                </button>
                            </div>
                        </div>
                    </div>

                    <div class="overflow-x-auto">
                        <table class="quickbooks-table">
                            <thead>
                                <tr>
                                    <th>Band</th>
                                    <th>Description</th>
                                    <th>Monthly Range (ZMW)</th>
                                    <th>Annual Range (ZMW)</th>
                                    <th>Tax Rate</th>
                                    <th>Monthly Tax Example</th>
                                </tr>
                            </thead>
                            <tbody>
                                @foreach($taxBands as $band)
                                    <tr>
                                        <td class="font-medium text-gray900">{{ $band->band_number }}</td>
                                        <td>{{ $band->description }}</td>
                                        <td>{{ $band->monthly_range }}</td>
                                        <td>{{ $band->annual_range }}</td>
                                        <td>
                                            <span class="quickbooks-badge bg-red-100 text-red-800">
                                                {{ $band->tax_rate_formatted }}
                                            </span>
                                        </td>
                                        <td class="text-sm text-gray600">
                                            @if($band->tax_rate > 0)
                                                Salary K{{ number_format($band->monthly_to, 0) }} = K{{ number_format($band->calculateMonthlyTax($band->monthly_to), 2) }} tax
                                            @else
                                                Tax-free
                                            @endif
                                        </td>
                                    </tr>
                                @endforeach
                            </tbody>
                        </table>
                    </div>
                    
                    <div class="border-t border-gray200 p-4">
                        <div class="text-sm text-gray600">
                            <i class="fas fa-calculator mr-1"></i>
                            Total Bands: {{ $taxBands->count() }} | 
                            Highest Rate: {{ $taxBands->max('tax_rate') }}% |
                            Lowest Rate: {{ $taxBands->min('tax_rate') }}%
                        </div>
                    </div>
                </div>
            @endif
        </main>
    </div>

    <!-- Delete Confirmation Modal -->
    <div id="deleteBandModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-md w-full">
            <div class="p-6">
                <div class="flex items-center mb-4">
                    <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                        <i class="fas fa-exclamation-triangle text-danger"></i>
                    </div>
                    <div>
                        <h3 class="text-lg font-semibold text-gray900">Delete Tax Band</h3>
                        <p class="text-gray600 text-sm">Are you sure you want to delete this tax band?</p>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end space-x-3">
                <button onclick="closeDeleteBandModal()" class="px-4 py-2 text-gray700 hover:bg-gray100 rounded text-sm font-medium">
                    Cancel
                </button>
                <button onclick="confirmDeleteBand()" class="bg-danger text-white px-4 py-2 rounded text-sm font-medium hover:bg-danger/90">
                    Delete
                </button>
            </div>
        </div>
    </div>

    <script>
        let taxBandCount = 0;
        let currentDeleteBandIndex = null;
        let isEditing = false;

        // Load PAYE settings when page loads
        document.addEventListener('DOMContentLoaded', function() {
            loadPayeSettings();
            
            // Set default dates if not set
            const startDateInput = document.getElementById('start_date');
            const endDateInput = document.getElementById('end_date');
            
            if (startDateInput && !startDateInput.value) {
                startDateInput.value = '{{ date("Y") }}-01-01';
            }
            
            if (endDateInput && !endDateInput.value) {
                endDateInput.value = '{{ date("Y") }}-12-31';
            }
        });

        // Load PAYE settings for selected financial year
        function loadPayeSettings() {
            const yearSelect = document.getElementById('financial_year_id');
            const yearId = yearSelect.value;
            
            if (!yearId) {
                clearTaxBands();
                return;
            }
            
            // Get selected option data
            const selectedOption = yearSelect.options[yearSelect.selectedIndex];
            const yearText = selectedOption.text.split(' ')[0];
            const startDate = selectedOption.getAttribute('data-start-date');
            const endDate = selectedOption.getAttribute('data-end-date');
            
            // Update year and dates
            document.getElementById('year').value = yearText;
            document.getElementById('start_date').value = startDate;
            document.getElementById('end_date').value = endDate;
            
            // Show loading
            showNotification('Loading PAYE settings...', 'info');
            
            // Fetch settings from server
            fetch(`{{ route('admin.accounting.payroll.get-settings') }}?financial_year_id=${yearId}&setting_type=paye`)
                .then(response => response.json())
                .then(data => {
                    if (data.success && data.setting && data.tax_bands) {
                        // Load existing tax bands
                        loadTaxBands(data.tax_bands);
                        isEditing = true;
                        showNotification('Loaded existing PAYE settings', 'success');
                    } else {
                        // Load default Zambian PAYE bands
                        loadDefaultBands();
                        isEditing = false;
                        showNotification('Loaded default Zambian PAYE bands', 'info');
                    }
                })
                .catch(error => {
                    console.error('Error loading settings:', error);
                    loadDefaultBands();
                    showNotification('Loaded default Zambian PAYE bands', 'info');
                });
        }

        // Load default Zambian PAYE bands
        function loadDefaultBands() {
            const defaultBands = [
                {
                    band_number: 1,
                    description: 'First Band',
                    monthly_from: '0.00',
                    monthly_to: '5100.00',
                    annual_from: '0.00',
                    annual_to: '61200.00',
                    tax_rate: '0.00'
                },
                {
                    band_number: 2,
                    description: 'Second Band',
                    monthly_from: '5100.01',
                    monthly_to: '7100.00',
                    annual_from: '61201.00',
                    annual_to: '85200.00',
                    tax_rate: '20.00'
                },
                {
                    band_number: 3,
                    description: 'Third Band',
                    monthly_from: '7100.01',
                    monthly_to: '9200.00',
                    annual_from: '85201.00',
                    annual_to: '110400.00',
                    tax_rate: '30.00'
                },
                {
                    band_number: 4,
                    description: 'Fourth Band',
                    monthly_from: '9200.01',
                    monthly_to: '999999999.99',
                    annual_from: '110401.00',
                    annual_to: '999999999.99',
                    tax_rate: '37.00'
                }
            ];
            
            loadTaxBands(defaultBands);
        }

        // Load tax bands into the table
        function loadTaxBands(bands) {
            clearTaxBands();
            taxBandCount = bands.length;
            
            bands.forEach((band, index) => {
                addTaxBandRow(index, band);
            });
            
            // Highlight the table
            const tableBody = document.getElementById('taxBandsBody');
            tableBody.classList.add('tax-band-highlight');
            setTimeout(() => {
                tableBody.classList.remove('tax-band-highlight');
            }, 2000);
        }

        // Clear all tax bands
        function clearTaxBands() {
            const tableBody = document.getElementById('taxBandsBody');
            tableBody.innerHTML = '';
            taxBandCount = 0;
        }

        // Add a new tax band row
        function addTaxBand() {
            const nextBandNumber = taxBandCount + 1;
            const newBand = {
                band_number: nextBandNumber,
                description: `Band ${nextBandNumber}`,
                monthly_from: '0.00',
                monthly_to: '0.00',
                annual_from: '0.00',
                annual_to: '0.00',
                tax_rate: '0.00'
            };
            
            addTaxBandRow(taxBandCount, newBand);
            taxBandCount++;
            
            // Scroll to the new row
            const newRow = document.querySelector('#taxBandsBody tr:last-child');
            if (newRow) {
                newRow.scrollIntoView({ behavior: 'smooth', block: 'center' });
                newRow.classList.add('tax-band-highlight');
            }
            
            showNotification('New tax band added', 'info');
        }

        // Add tax band row to table
        function addTaxBandRow(index, band) {
            const tableBody = document.getElementById('taxBandsBody');
            
            const row = document.createElement('tr');
            row.className = 'tax-band-row';
            row.innerHTML = `
                <td>
                    <input type="number" 
                           name="tax_bands[${index}][band_number]" 
                           value="${band.band_number}" 
                           class="quickbooks-input w-16 text-center band-number" 
                           min="1" 
                           readonly>
                </td>
                <td>
                    <input type="text" 
                           name="tax_bands[${index}][description]" 
                           value="${band.description}" 
                           class="quickbooks-input w-full band-description" 
                           placeholder="Band description"
                           required>
                </td>
                <td>
                    <div class="flex items-center space-x-2">
                        <div class="relative">
                            <span class="absolute left-2 top-1/2 transform -translate-y-1/2 text-xs text-gray500">K</span>
                            <input type="number" 
                                   name="tax_bands[${index}][monthly_from]" 
                                   value="${band.monthly_from}" 
                                   class="quickbooks-input w-24 pl-6 monthly-from" 
                                   step="0.01" 
                                   min="0" 
                                   required
                                   onchange="updateAnnualFrom(${index})">
                        </div>
                        <span class="text-gray500">to</span>
                        <div class="relative">
                            <span class="absolute left-2 top-1/2 transform -translate-y-1/2 text-xs text-gray500">K</span>
                            <input type="number" 
                                   name="tax_bands[${index}][monthly_to]" 
                                   value="${band.monthly_to}" 
                                   class="quickbooks-input w-24 pl-6 monthly-to" 
                                   step="0.01" 
                                   min="0" 
                                   required
                                   onchange="updateAnnualTo(${index})">
                        </div>
                    </div>
                </td>
                <td>
                    <div class="flex items-center space-x-2">
                        <div class="relative">
                            <span class="absolute left-2 top-1/2 transform -translate-y-1/2 text-xs text-gray500">K</span>
                            <input type="number" 
                                   name="tax_bands[${index}][annual_from]" 
                                   value="${band.annual_from}" 
                                   class="quickbooks-input w-24 pl-6 annual-from" 
                                   step="0.01" 
                                   min="0" 
                                   required>
                        </div>
                        <span class="text-gray500">to</span>
                        <div class="relative">
                            <span class="absolute left-2 top-1/2 transform -translate-y-1/2 text-xs text-gray500">K</span>
                            <input type="number" 
                                   name="tax_bands[${index}][annual_to]" 
                                   value="${band.annual_to}" 
                                   class="quickbooks-input w-24 pl-6 annual-to" 
                                   step="0.01" 
                                   min="0" 
                                   required>
                        </div>
                    </div>
                </td>
                <td>
                    <div class="relative">
                        <input type="number" 
                               name="tax_bands[${index}][tax_rate]" 
                               value="${band.tax_rate}" 
                               class="quickbooks-input w-full pr-8 tax-rate" 
                               step="0.01" 
                               min="0" 
                               max="100" 
                               required>
                        <span class="absolute right-3 top-1/2 transform -translate-y-1/2 text-gray500">%</span>
                    </div>
                </td>
                <td>
                    <div class="flex items-center space-x-1">
                        <button type="button" 
                                onclick="moveBandUp(${index})" 
                                class="text-gray-400 hover:text-blue-600 p-1"
                                title="Move up">
                            <i class="fas fa-arrow-up"></i>
                        </button>
                        <button type="button" 
                                onclick="moveBandDown(${index})" 
                                class="text-gray-400 hover:text-blue-600 p-1"
                                title="Move down">
                            <i class="fas fa-arrow-down"></i>
                        </button>
                        <button type="button" 
                                onclick="deleteBand(${index})" 
                                class="text-red-400 hover:text-red-600 p-1"
                                title="Delete band">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                </td>
            `;
            
            tableBody.appendChild(row);
        }

        // Update annual from based on monthly from
        function updateAnnualFrom(index) {
            const monthlyFrom = document.querySelector(`input[name="tax_bands[${index}][monthly_from]"]`);
            const annualFrom = document.querySelector(`input[name="tax_bands[${index}][annual_from]"]`);
            
            if (monthlyFrom && annualFrom) {
                const monthlyValue = parseFloat(monthlyFrom.value) || 0;
                annualFrom.value = (monthlyValue * 12).toFixed(2);
            }
        }

        // Update annual to based on monthly to
        function updateAnnualTo(index) {
            const monthlyTo = document.querySelector(`input[name="tax_bands[${index}][monthly_to]"]`);
            const annualTo = document.querySelector(`input[name="tax_bands[${index}][annual_to]"]`);
            
            if (monthlyTo && annualTo) {
                const monthlyValue = parseFloat(monthlyTo.value) || 0;
                annualTo.value = (monthlyValue * 12).toFixed(2);
            }
        }

        // Move band up
        function moveBandUp(index) {
            if (index <= 0) return;
            
            swapBands(index, index - 1);
            renumberBands();
            showNotification('Band moved up', 'success');
        }

        // Move band down
        function moveBandDown(index) {
            if (index >= taxBandCount - 1) return;
            
            swapBands(index, index + 1);
            renumberBands();
            showNotification('Band moved down', 'success');
        }

        // Swap two bands
        function swapBands(index1, index2) {
            const rows = document.querySelectorAll('#taxBandsBody tr');
            const tableBody = document.getElementById('taxBandsBody');
            
            // Get the rows
            const row1 = rows[index1];
            const row2 = rows[index2];
            
            // Swap their positions
            if (index1 < index2) {
                tableBody.insertBefore(row2, row1);
            } else {
                tableBody.insertBefore(row1, row2);
            }
            
            // Update the input names
            updateRowInputNames(row1, index2);
            updateRowInputNames(row2, index1);
        }

        // Update input names for a row
        function updateRowInputNames(row, newIndex) {
            const inputs = row.querySelectorAll('input');
            inputs.forEach(input => {
                const name = input.name;
                input.name = name.replace(/tax_bands\[\d+\]/, `tax_bands[${newIndex}]`);
            });
        }

        // Renumber all bands
        function renumberBands() {
            const rows = document.querySelectorAll('#taxBandsBody tr');
            rows.forEach((row, index) => {
                const bandNumberInput = row.querySelector('.band-number');
                if (bandNumberInput) {
                    bandNumberInput.value = index + 1;
                }
            });
        }

        // Delete band
        function deleteBand(index) {
            currentDeleteBandIndex = index;
            document.getElementById('deleteBandModal').classList.remove('hidden');
        }

        function closeDeleteBandModal() {
            document.getElementById('deleteBandModal').classList.add('hidden');
            currentDeleteBandIndex = null;
        }

        function confirmDeleteBand() {
            if (currentDeleteBandIndex !== null) {
                const row = document.querySelector(`#taxBandsBody tr:nth-child(${currentDeleteBandIndex + 1})`);
                if (row) {
                    row.remove();
                    taxBandCount--;
                    
                    // Renumber remaining bands
                    renumberBands();
                    
                    // Update input names for all rows
                    const rows = document.querySelectorAll('#taxBandsBody tr');
                    rows.forEach((row, index) => {
                        updateRowInputNames(row, index);
                    });
                    
                    showNotification('Tax band deleted', 'success');
                }
            }
            
            closeDeleteBandModal();
        }

        // Reset to default bands
        function resetToDefaultBands() {
            if (confirm('Are you sure you want to reset to default Zambian PAYE bands? This will replace all current bands.')) {
                loadDefaultBands();
                showNotification('Reset to default Zambian PAYE bands', 'info');
            }
        }

        // Validate and submit form
        function validateAndSubmit() {
            // Basic validation
            const rows = document.querySelectorAll('#taxBandsBody tr');
            if (rows.length === 0) {
                showNotification('Please add at least one tax band', 'error');
                return;
            }
            
            let isValid = true;
            let lastMonthlyTo = -1;
            let lastAnnualTo = -1;
            
            rows.forEach((row, index) => {
                const monthlyFrom = parseFloat(row.querySelector('.monthly-from').value) || 0;
                const monthlyTo = parseFloat(row.querySelector('.monthly-to').value) || 0;
                const annualFrom = parseFloat(row.querySelector('.annual-from').value) || 0;
                const annualTo = parseFloat(row.querySelector('.annual-to').value) || 0;
                const taxRate = parseFloat(row.querySelector('.tax-rate').value) || 0;
                
                // Check monthly range
                if (monthlyFrom >= monthlyTo) {
                    showNotification(`Band ${index + 1}: Monthly "From" must be less than "To"`, 'error');
                    isValid = false;
                }
                
                // Check annual range
                if (annualFrom >= annualTo) {
                    showNotification(`Band ${index + 1}: Annual "From" must be less than "To"`, 'error');
                    isValid = false;
                }
                
                // Check band continuity
                if (monthlyFrom <= lastMonthlyTo) {
                    showNotification(`Band ${index + 1}: Monthly range must start after previous band ends`, 'error');
                    isValid = false;
                }
                
                if (annualFrom <= lastAnnualTo) {
                    showNotification(`Band ${index + 1}: Annual range must start after previous band ends`, 'error');
                    isValid = false;
                }
                
                // Check tax rate
                if (taxRate < 0 || taxRate > 100) {
                    showNotification(`Band ${index + 1}: Tax rate must be between 0% and 100%`, 'error');
                    isValid = false;
                }
                
                lastMonthlyTo = monthlyTo;
                lastAnnualTo = annualTo;
            });
            
            if (isValid) {
                // Show confirmation for editing existing settings
                if (isEditing) {
                    if (confirm('Update existing PAYE settings for this financial year?')) {
                        document.getElementById('payeForm').submit();
                    }
                } else {
                    document.getElementById('payeForm').submit();
                }
            }
        }

        // Export tax bands
        function exportTaxBands() {
            const year = document.getElementById('year').value || 'Current';
            const financialYear = document.getElementById('financial_year_id').selectedOptions[0]?.text || '';
            
            let csvContent = "data:text/csv;charset=utf-8,";
            csvContent += "PAYE Tax Bands\n";
            csvContent += `Financial Year: ${financialYear}\n`;
            csvContent += `Year: ${year}\n`;
            csvContent += `Generated: ${new Date().toLocaleDateString()}\n\n`;
            csvContent += "Band,Description,Monthly From,Monthly To,Annual From,Annual To,Tax Rate (%)\n";
            
            const rows = document.querySelectorAll('#taxBandsBody tr');
            rows.forEach(row => {
                const bandNumber = row.querySelector('.band-number').value;
                const description = row.querySelector('.band-description').value;
                const monthlyFrom = row.querySelector('.monthly-from').value;
                const monthlyTo = row.querySelector('.monthly-to').value;
                const annualFrom = row.querySelector('.annual-from').value;
                const annualTo = row.querySelector('.annual-to').value;
                const taxRate = row.querySelector('.tax-rate').value;
                
                csvContent += `${bandNumber},"${description}",${monthlyFrom},${monthlyTo},${annualFrom},${annualTo},${taxRate}\n`;
            });
            
            const encodedUri = encodeURI(csvContent);
            const link = document.createElement("a");
            link.setAttribute("href", encodedUri);
            link.setAttribute("download", `PAYE_Tax_Bands_${year}.csv`);
            document.body.appendChild(link);
            link.click();
            document.body.removeChild(link);
            
            showNotification('Tax bands exported successfully', 'success');
        }

        // Print tax bands
        function printTaxBands() {
            const printContent = document.getElementById('taxBandsBody').parentElement.outerHTML;
            const originalContent = document.body.innerHTML;
            const year = document.getElementById('year').value || 'Current';
            const financialYear = document.getElementById('financial_year_id').selectedOptions[0]?.text || '';
            
            document.body.innerHTML = `
                <!DOCTYPE html>
                <html>
                <head>
                    <title>PAYE Tax Bands - ${year}</title>
                    <style>
                        body { font-family: Arial, sans-serif; margin: 20px; }
                        h1 { color: #146CAA; margin-bottom: 5px; }
                        .header { text-align: center; margin-bottom: 20px; }
                        table { width: 100%; border-collapse: collapse; margin-top: 20px; }
                        th { background-color: #f3f3f3; padding: 10px; text-align: left; font-size: 12px; }
                        td { padding: 8px; border-bottom: 1px solid #ddd; font-size: 12px; }
                        .footer { margin-top: 30px; font-size: 11px; color: #666; text-align: center; }
                        @media print {
                            .no-print { display: none; }
                        }
                    </style>
                </head>
                <body>
                    <div class="header">
                        <h1>PAYE Tax Bands</h1>
                        <p>Financial Year: ${financialYear}</p>
                        <p>Year: ${year}</p>
                        <p>Generated on: ${new Date().toLocaleDateString()}</p>
                    </div>
                    ${printContent}
                    <div class="footer">
                        <p>Generated by IMMIA FINANCE Payroll System</p>
                        <p>This is an official document</p>
                    </div>
                    <div class="no-print" style="margin-top: 20px; text-align: center;">
                        <button onclick="window.print()" style="padding: 10px 20px; background: #146CAA; color: white; border: none; border-radius: 4px; cursor: pointer;">
                            Print Document
                        </button>
                        <button onclick="window.close()" style="padding: 10px 20px; background: #666; color: white; border: none; border-radius: 4px; cursor: pointer; margin-left: 10px;">
                            Close
                        </button>
                    </div>
                </body>
                </html>
            `;
            
            window.print();
            document.body.innerHTML = originalContent;
        }

        // Utility functions
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-primary',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Add animation styles
        if (!document.querySelector('style[data-paye]')) {
            const style = document.createElement('style');
            style.setAttribute('data-paye', 'true');
            style.textContent = `
                @keyframes fadeIn {
                    from { opacity: 0; transform: translateY(-10px); }
                    to { opacity: 1; transform: translateY(0); }
                }
                @keyframes fadeOut {
                    from { opacity: 1; transform: translateY(0); }
                    to { opacity: 0; transform: translateY(-10px); }
                }
                .animate-fade-in {
                    animation: fadeIn 0.3s ease-out;
                }
                .animate-fade-out {
                    animation: fadeOut 0.3s ease-in;
                }
            `;
            document.head.appendChild(style);
        }
    </script>
</body>
</html>