<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Preliminary Payroll Summary - {{ $period->period_name }} | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <style>
        .quickbooks-table { border-collapse: separate; border-spacing: 0; width: 100%; }
        .quickbooks-table th { background-color: #F3F2F1; color: #323130; font-weight: 600; font-size: 13px; text-transform: uppercase; padding: 12px 16px; border-bottom: 1px solid #EDEBE9; text-align: left; }
        .quickbooks-table td { padding: 12px 16px; border-bottom: 1px solid #EDEBE9; font-size: 14px; color: #323130; vertical-align: middle; }
        .quickbooks-table tbody tr:hover { background-color: #F3F2F1; }
        .quickbooks-input { border: 1px solid #8C8C8C; border-radius: 4px; padding: 6px 12px; font-size: 14px; }
        .quickbooks-input:focus { outline: 2px solid #146CAA; outline-offset: -1px; border-color: #146CAA; }
        .quickbooks-badge { display: inline-flex; align-items: center; padding: 4px 8px; border-radius: 12px; font-size: 12px; font-weight: 600; }
        .action-btn { width: 32px; height: 32px; border-radius: 4px; display: inline-flex; align-items: center; justify-content: center; margin: 0 2px; transition: all 0.2s; }
        .action-btn:hover { transform: translateY(-1px); }
        .action-btn-edit { background-color: #E1F0FA; color: #146CAA; }
        .action-btn-edit:hover { background-color: #146CAA; color: white; }
        .action-btn-delete { background-color: #FEE; color: #D83B01; }
        .action-btn-delete:hover { background-color: #D83B01; color: white; }
        .action-btn-view { background-color: #F3F2F1; color: #323130; }
        .action-btn-view:hover { background-color: #146CAA; color: white; }
        .action-btn-adjust { background-color: #146CAA; color: white; }
        .action-btn-adjust:hover { background-color: #0d5c95; color: white; }
        .action-btn-calculate { background-color: #8B5CF6; color: white; }
        .action-btn-calculate:hover { background-color: #7C3AED; color: white; }
        .action-btn-approve { background-color: #10B981; color: white; }
        .action-btn-approve:hover { background-color: #059669; color: white; }
        
        /* Loading Spinner */
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        .animate-spin {
            animation: spin 1s linear infinite;
        }
        
        /* Notification Animations */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        @keyframes fadeOut {
            from { opacity: 1; transform: translateY(0); }
            to { opacity: 0; transform: translateY(-10px); }
        }
        .animate-fade-in { animation: fadeIn 0.3s ease-out; }
        .animate-fade-out { animation: fadeOut 0.3s ease-in; }
        
        /* Print Styles */
        @media print {
            .no-print { display: none !important; }
            .quickbooks-table th { background-color: #f0f0f0 !important; }
            .action-btn { display: none !important; }
        }
    </style>
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Preliminary Payroll Summary', 'subtitle' => $period->period_name])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Period Info Card -->
            <div class="bg-white rounded-lg border border-gray-200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h1 class="text-xl font-bold text-gray-900">{{ $period->period_name }}</h1>
                        <div class="flex flex-wrap items-center gap-4 mt-2">
                            <span class="inline-flex items-center text-sm text-gray-600">
                                <i class="fas fa-calendar-alt mr-1"></i>
                                {{ $period->start_date->format('M d, Y') }} - {{ $period->end_date->format('M d, Y') }}
                            </span>
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                {{ $period->status === 'open' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-800' }}">
                                {{ ucfirst($period->status) }}
                            </span>
                            <span class="inline-flex items-center text-sm text-gray-600">
                                <i class="fas fa-money-bill-wave mr-1"></i>
                                Payment Date: {{ $period->payment_date->format('M d, Y') }}
                            </span>
                            @if($payrollEntries->count() > 0)
                                <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-purple-100 text-purple-800">
                                    <i class="fas fa-calculator mr-1"></i>
                                    Payroll Calculated
                                </span>
                            @endif
                        </div>
                    </div>
                    <div class="mt-4 md:mt-0 flex flex-col md:flex-row gap-3">
                        <a href="{{ route('admin.accounting.payroll.preparation.index') }}" 
                           class="text-primary hover:text-primary/90 font-medium text-sm flex items-center justify-center no-print">
                            <i class="fas fa-arrow-left mr-2"></i>Back to Preparation
                        </a>
                        <button onclick="window.print()" 
                                class="bg-gray-200 text-gray-700 px-4 py-2 rounded text-sm font-medium hover:bg-gray-300 flex items-center justify-center no-print">
                            <i class="fas fa-print mr-2"></i> Print Summary
                        </button>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-blue-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-users text-blue-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Total Employees</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $summaryTotals['total_employees'] }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-sync-alt text-green-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Total Recurring</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_recurring'], 2) }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-yellow-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-plus-circle text-yellow-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Total Non-Recurring</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_non_recurring'], 2) }}</p>
                        </div>
                    </div>
                </div>
                <div class="bg-white rounded-lg border border-gray-200 p-4">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-blue-50 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-money-bill-wave text-blue-600 text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray-600">Grand Total</p>
                            <p class="text-xl font-semibold text-gray-900">{{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_gross'], 2) }}</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Employees Table with QuickBooks Styling -->
            <div class="bg-white rounded-lg border border-gray-200 overflow-hidden">
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Employee #</th>
                                <th>Name</th>
                                <th>NRC</th>
                                <th>Email</th>
                                <th>Phone</th>
                                <th>Basic Salary</th>
                                <th>Allowances</th>
                                <th>Recurring Earnings</th>
                                <th>Non-Recurring</th>
                                <th>Gross Total</th>
                                <th>Status</th>
                                <th class="no-print">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($employeeSummaries as $summary)
                                @php
                                    $employee = $summary['employee'];
                                @endphp
                                <tr class="hover:bg-gray-100 transition-colors">
                                    <td class="font-medium">
                                        <span class="quickbooks-badge bg-blue-100 text-blue-800">
                                            {{ $employee->employee_number }}
                                        </span>
                                    </td>
                                    <td>
                                        <div class="font-medium text-gray-900">{{ $employee->first_name }} {{ $employee->last_name }}</div>
                                        <div class="text-xs text-gray-500">{{ $employee->employment_type }}</div>
                                    </td>
                                    <td class="text-gray-700">
                                        {{ $employee->national_id ?? '—' }}
                                    </td>
                                    <td class="text-gray-700">
                                        {{ $employee->email }}
                                    </td>
                                    <td class="text-gray-700">
                                        {{ $employee->phone ?? '—' }}
                                    </td>
                                    <td class="text-gray-700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($summary['basic_salary'], 2) }}
                                    </td>
                                    <td class="text-gray-700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($summary['allowances'], 2) }}
                                    </td>
                                    <td class="text-gray-700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($summary['total_recurring'], 2) }}
                                    </td>
                                    <td class="text-gray-700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($summary['total_non_recurring'], 2) }}
                                        @if($summary['adjustment_count'] > 0)
                                            <div class="text-xs text-gray-500">
                                                ({{ $summary['adjustment_count'] }} adjustment{{ $summary['adjustment_count'] > 1 ? 's' : '' }})
                                            </div>
                                        @endif
                                    </td>
                                    <td class="text-gray-700 font-bold">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($summary['gross_total'], 2) }}
                                    </td>
                                    <td>
                                        @if($summary['has_pending_adjustments'])
                                            <span class="quickbooks-badge bg-yellow-100 text-yellow-800">
                                                <i class="fas fa-clock mr-1"></i>Pending
                                            </span>
                                        @elseif($summary['adjustment_count'] > 0)
                                            <span class="quickbooks-badge bg-green-100 text-green-800">
                                                <i class="fas fa-check mr-1"></i>Ready
                                            </span>
                                        @else
                                            <span class="quickbooks-badge bg-gray-100 text-gray-800">
                                                No Adjustments
                                            </span>
                                        @endif
                                    </td>
                                    <td class="no-print">
                                        <div class="flex items-center">
                                            <a href="{{ route('admin.accounting.payroll.preparation.create', ['periodId' => $period->id, 'employeeId' => $employee->id]) }}" 
                                               class="action-btn action-btn-adjust"
                                               title="Payroll Adjustments">
                                                <i class="fas fa-edit"></i>
                                            </a>
                                            @if($summary['adjustment_count'] > 0)
                                            <a href="{{ route('admin.accounting.payroll.preparation.create', ['periodId' => $period->id, 'employeeId' => $employee->id]) }}#adjustmentsSection" 
                                               class="action-btn action-btn-view ml-1"
                                               title="View Adjustments">
                                                <i class="fas fa-eye"></i>
                                            </a>
                                            @endif
                                        </div>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                        <tfoot class="bg-gray-100">
                            <tr>
                                <td colspan="5" class="text-right font-medium text-gray-700 py-3">
                                    <div>TOTALS:</div>
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_basic_salary'], 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_allowances'], 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_recurring'], 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_non_recurring'], 2) }}
                                </td>
                                <td class="font-bold text-blue-600 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($summaryTotals['total_gross'], 2) }}
                                </td>
                                <td colspan="2" class="py-3">
                                    <div class="text-sm text-gray-600">
                                        {{ $summaryTotals['total_employees'] }} employee(s)
                                    </div>
                                    @if($summaryTotals['total_pending_adjustments'] > 0)
                                        <div class="text-sm text-yellow-600 mt-1">
                                            {{ $summaryTotals['total_pending_adjustments'] }} pending adjustment(s)
                                        </div>
                                    @endif
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
            </div>

            <!-- CALCULATED PAYROLL SECTION -->
            @if($payrollEntries->count() > 0)
            <div class="mt-8 bg-white rounded-lg border border-gray-200 overflow-hidden">
                <div class="bg-gray-50 px-6 py-4 border-b border-gray-200">
                    <h2 class="text-lg font-semibold text-gray-900">
                        <i class="fas fa-calculator mr-2 text-purple-600"></i>
                        Calculated Payroll Summary
                    </h2>
                    <p class="text-sm text-gray-600 mt-1">
                        Net Pay, PAYE, NAPSA, and NHIMA calculations for this period
                    </p>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Employee</th>
                                <th>Gross Earnings</th>
                                <th>PAYE Tax</th>
                                <th>NAPSA (Emp)</th>
                                <th>NHIMA (Emp)</th>
                                <th>Total Deductions</th>
                                <th>Net Pay</th>
                                <th>Status</th>
                                <th class="no-print">Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($payrollEntries as $entry)
                            <tr>
                                <td class="font-medium">
                                    <div class="font-medium text-gray-900">{{ $entry->employee->first_name }} {{ $entry->employee->last_name }}</div>
                                    <div class="text-xs text-gray-500">{{ $entry->employee->employee_number }}</div>
                                </td>
                                <td class="font-medium">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->gross_earnings, 2) }}
                                    <div class="text-xs text-gray-500">
                                        Taxable: {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->total_taxable_earnings, 2) }}
                                    </div>
                                </td>
                                <td class="font-medium">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->paye_tax, 2) }}
                                </td>
                                <td class="font-medium">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->napsa_employee, 2) }}
                                    <div class="text-xs text-gray-500">
                                        Basic: {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->basic_salary_for_napsa, 2) }}
                                    </div>
                                </td>
                                <td class="font-medium">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->nhima_employee, 2) }}
                                </td>
                                <td class="font-medium text-red-600">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->total_deductions, 2) }}
                                </td>
                                <td class="font-bold text-green-600">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->net_pay, 2) }}
                                </td>
                                <td>
                                    {!! $entry->status_badge !!}
                                </td>
                                <td class="no-print">
                                    <div class="flex items-center space-x-1">
                                        <button onclick="viewEarningsBreakdown({{ $entry->id }})" 
                                                class="action-btn action-btn-view"
                                                title="View Earnings Breakdown">
                                            <i class="fas fa-list"></i>
                                        </button>
                                        @if($entry->canApprove())
                                        <button onclick="approveEntry({{ $entry->id }})" 
                                                class="action-btn action-btn-approve ml-1"
                                                title="Approve">
                                            <i class="fas fa-check"></i>
                                        </button>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                            @endforeach
                        </tbody>
                        <tfoot class="bg-gray-100">
                            <tr>
                                <td class="font-bold text-gray-900 py-3">TOTALS:</td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('gross_earnings'), 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('paye_tax'), 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('napsa_employee'), 2) }}
                                </td>
                                <td class="font-bold text-gray-900 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('nhima_employee'), 2) }}
                                </td>
                                <td class="font-bold text-red-600 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('total_deductions'), 2) }}
                                </td>
                                <td class="font-bold text-green-600 py-3">
                                    {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('net_pay'), 2) }}
                                </td>
                                <td colspan="2" class="py-3">
                                    <div class="text-sm text-gray-600">
                                        Employer NAPSA: {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('napsa_employer'), 2) }}
                                    </div>
                                    <div class="text-sm text-gray-600 mt-1">
                                        Employer NHIMA: {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollEntries->sum('nhima_employer'), 2) }}
                                    </div>
                                </td>
                            </tr>
                        </tfoot>
                    </table>
                </div>
                
                <!-- Action Buttons for Calculated Payroll -->
                <div class="bg-gray-50 px-6 py-4 border-t border-gray-200 no-print">
                    <div class="flex justify-between items-center">
                        <div class="text-sm text-gray-600">
                            {{ $payrollEntries->count() }} payroll entries calculated
                        </div>
                        <div class="flex space-x-3">
                            <button onclick="exportPayroll({{ $period->id }})" 
                                    class="bg-gray-200 text-gray-700 px-4 py-2 rounded text-sm font-medium hover:bg-gray-300 flex items-center">
                                <i class="fas fa-download mr-2"></i> Export Payroll
                            </button>
                            @if($payrollEntries->where('calculation_status', 'calculated')->count() > 0)
                            <button onclick="bulkApprovePayroll({{ $period->id }})" 
                                    class="bg-green-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-green-700 flex items-center">
                                <i class="fas fa-check-double mr-2"></i> Bulk Approve
                            </button>
                            @endif
                            <a href="{{ route('admin.accounting.payroll.processing.create', $period->id) }}" 
                               class="bg-blue-600 text-white px-4 py-2 rounded text-sm font-medium hover:bg-blue-700 flex items-center">
                                <i class="fas fa-forward mr-2"></i> Process Payments
                            </a>
                        </div>
                    </div>
                </div>
            </div>
            @endif

            <!-- Additional Summary Information -->
            <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mt-6">
                <!-- Taxable vs Non-Taxable Breakdown -->
                <div class="bg-white rounded-lg border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Tax Breakdown</h3>
                    <div class="space-y-4">
                        @php
                            // Calculate taxable vs non-taxable amounts
                            $totalTaxable = 0;
                            $totalNonTaxable = 0;
                            foreach ($employeeSummaries as $summary) {
                                $totalTaxable += $summary['total_recurring'];
                                $totalNonTaxable += $summary['total_non_recurring'];
                            }
                        @endphp
                        <div class="flex justify-between items-center">
                            <span class="text-sm text-gray-600">Taxable Earnings:</span>
                            <span class="font-medium text-red-600">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalTaxable, 2) }}</span>
                        </div>
                        <div class="flex justify-between items-center">
                            <span class="text-sm text-gray-600">Non-Taxable Earnings:</span>
                            <span class="font-medium text-green-600">{{ $company->currency_symbol ?? 'K' }}{{ number_format($totalNonTaxable, 2) }}</span>
                        </div>
                        <div class="border-t border-gray-200 pt-4">
                            <div class="flex justify-between items-center">
                                <span class="text-sm font-medium text-gray-700">Taxable Percentage:</span>
                                <span class="font-bold text-blue-600">
                                    @if($summaryTotals['total_gross'] > 0)
                                        {{ number_format(($totalTaxable / $summaryTotals['total_gross']) * 100, 1) }}%
                                    @else
                                        0%
                                    @endif
                                </span>
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Adjustment Status Summary -->
                <div class="bg-white rounded-lg border border-gray-200 p-6">
                    <h3 class="text-lg font-semibold text-gray-900 mb-4">Adjustment Status</h3>
                    <div class="space-y-3">
                        <div class="flex justify-between items-center">
                            <span class="text-sm text-gray-600">Employees with Adjustments:</span>
                            <span class="font-medium">
                                {{ collect($employeeSummaries)->where('adjustment_count', '>', 0)->count() }}/{{ $summaryTotals['total_employees'] }}
                            </span>
                        </div>
                        <div class="flex justify-between items-center">
                            <span class="text-sm text-gray-600">Employees Pending Approval:</span>
                            <span class="font-medium text-yellow-600">
                                {{ collect($employeeSummaries)->where('has_pending_adjustments', true)->count() }}
                            </span>
                        </div>
                        <div class="flex justify-between items-center">
                            <span class="text-sm text-gray-600">Employees Ready for Processing:</span>
                            <span class="font-medium text-green-600">
                                {{ collect($employeeSummaries)->where('adjustment_count', '>', 0)->where('has_pending_adjustments', false)->count() }}
                            </span>
                        </div>
                        <div class="border-t border-gray-200 pt-4">
                            <div class="text-sm text-gray-600 mb-2">Completion Status:</div>
                            <div class="w-full bg-gray-200 rounded-full h-2.5">
                                @php
                                    $completedPercent = $summaryTotals['total_employees'] > 0 ? 
                                        (collect($employeeSummaries)->where('adjustment_count', '>', 0)->count() / $summaryTotals['total_employees']) * 100 : 0;
                                @endphp
                                <div class="bg-blue-600 h-2.5 rounded-full" style="width: {{ $completedPercent }}%"></div>
                            </div>
                            <div class="text-xs text-gray-500 mt-1 text-right">{{ number_format($completedPercent, 1) }}% Complete</div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Action Buttons -->
            <div class="flex flex-wrap justify-between items-center mt-6 gap-4 no-print">
                <div class="flex space-x-3">
                    <a href="{{ route('admin.accounting.payroll.preparation.index') }}" 
                       class="bg-gray-200 text-gray-700 px-6 py-3 rounded text-sm font-medium hover:bg-gray-300 flex items-center">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Preparation
                    </a>
                    <a href="{{ route('admin.accounting.payroll.preparation.create', $period->id) }}" 
                       class="bg-blue-100 text-blue-700 px-6 py-3 rounded text-sm font-medium hover:bg-blue-200 flex items-center">
                        <i class="fas fa-plus-circle mr-2"></i> All Adjustments
                    </a>
                </div>
                
                <div class="flex space-x-3">
                    @if($summaryTotals['total_pending_adjustments'] > 0)
                        <a href="{{ route('admin.accounting.payroll.preparation.create', ['periodId' => $period->id, 'view' => 'pending']) }}" 
                           class="bg-yellow-100 text-yellow-700 px-6 py-3 rounded text-sm font-medium hover:bg-yellow-200 flex items-center">
                            <i class="fas fa-clock mr-2"></i> View Pending ({{ $summaryTotals['total_pending_adjustments'] }})
                        </a>
                    @endif
                    
                    <!-- CALCULATE PAYROLL BUTTON -->
                    @if($payrollEntries->count() === 0)
                    <button id="calculatePayrollBtn" 
                            data-period-id="{{ $period->id }}"
                            class="bg-purple-600 text-white px-6 py-3 rounded text-sm font-medium hover:bg-purple-700 flex items-center no-print">
                        <i class="fas fa-calculator mr-2"></i> Calculate Payroll
                    </button>
                    @else
                    <a href="{{ route('admin.accounting.payroll.processing.create', $period->id) }}" 
                       class="bg-green-600 text-white px-6 py-3 rounded text-sm font-medium hover:bg-green-700 flex items-center">
                        <i class="fas fa-calculator mr-2"></i> Process Payroll
                    </a>
                    @endif
                </div>
            </div>

            <!-- Quick Help Section -->
            <div class="mt-8 bg-blue-50 border border-blue-200 rounded-lg p-4 no-print">
                <div class="flex items-start">
                    <i class="fas fa-info-circle text-blue-500 mt-0.5 mr-3"></i>
                    <div>
                        <h4 class="font-medium text-blue-900 mb-2">How to Use This Summary</h4>
                        <ul class="text-sm text-blue-800 space-y-1">
                            <li>• Click the <i class="fas fa-edit text-blue-600 mx-1"></i> button to add or edit adjustments for an employee</li>
                            <li>• Employees with <span class="bg-yellow-100 text-yellow-800 px-1 rounded text-xs">Pending</span> status have unapproved adjustments</li>
                            <li>• Employees with <span class="bg-green-100 text-green-800 px-1 rounded text-xs">Ready</span> status have approved adjustments</li>
                            <li>• Click "Calculate Payroll" to compute PAYE, NAPSA, NHIMA, and Net Pay for all employees</li>
                            <li>• After calculation, review and approve payroll entries before processing payments</li>
                            <li>• Use the print button to generate a printable version of this summary</li>
                        </ul>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <!-- Earnings Breakdown Modal -->
    <div id="earningsBreakdownModal" class="fixed inset-0 bg-gray-500 bg-opacity-75 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg shadow-xl w-full max-w-4xl mx-4 max-h-[90vh] overflow-hidden">
            <div class="px-6 py-4 border-b border-gray-200">
                <h3 class="text-lg font-semibold text-gray-900">Earnings Breakdown</h3>
                <p class="text-sm text-gray-600 mt-1" id="modalEmployeeName"></p>
            </div>
            <div class="p-6 overflow-y-auto max-h-[60vh]" id="modalContent">
                <!-- Content will be loaded via AJAX -->
            </div>
            <div class="px-6 py-4 border-t border-gray-200 bg-gray-50">
                <div class="flex justify-end">
                    <button onclick="closeModal()" 
                            class="bg-gray-200 text-gray-700 px-4 py-2 rounded text-sm font-medium hover:bg-gray-300">
                        Close
                    </button>
                </div>
            </div>
        </div>
    </div>

      <script>
        // Set base URL for API calls
        const baseUrl = '{{ url("") }}';
        const apiBaseUrl = baseUrl + '/admin/accounting/payroll';

        // Utility function to show notifications
        function showNotification(message, type = 'info') {
            const existing = document.querySelector('.quickbooks-notification');
            if (existing) existing.remove();

            const colors = {
                success: 'bg-green-500',
                error: 'bg-red-500',
                info: 'bg-blue-500',
                warning: 'bg-yellow-500'
            };

            const icons = {
                success: 'fa-check-circle',
                error: 'fa-exclamation-circle',
                info: 'fa-info-circle',
                warning: 'fa-exclamation-triangle'
            };

            const notification = document.createElement('div');
            notification.className = `quickbooks-notification fixed top-4 right-4 ${colors[type]} text-white px-4 py-3 rounded shadow-lg z-50 flex items-center animate-fade-in`;
            notification.innerHTML = `
                <i class="fas ${icons[type]} mr-2"></i>
                <span>${message}</span>
            `;

            document.body.appendChild(notification);

            setTimeout(() => {
                notification.classList.add('animate-fade-out');
                setTimeout(() => {
                    notification.remove();
                }, 300);
            }, 3000);
        }

        // Search functionality for the table
        function searchTable() {
            const input = document.getElementById('searchInput');
            const filter = input.value.toLowerCase();
            const table = document.querySelector('.quickbooks-table');
            const rows = table.getElementsByTagName('tbody')[0].getElementsByTagName('tr');

            for (let i = 0; i < rows.length; i++) {
                const cells = rows[i].getElementsByTagName('td');
                let found = false;
                
                for (let j = 0; j < cells.length; j++) {
                    if (cells[j]) {
                        const text = cells[j].textContent || cells[j].innerText;
                        if (text.toLowerCase().indexOf(filter) > -1) {
                            found = true;
                            break;
                        }
                    }
                }
                
                rows[i].style.display = found ? '' : 'none';
            }
        }

        // Sort table by column
        function sortTable(columnIndex) {
            const table = document.querySelector('.quickbooks-table');
            const tbody = table.getElementsByTagName('tbody')[0];
            const rows = Array.from(tbody.getElementsByTagName('tr'));
            
            // Get current sort direction
            const header = table.getElementsByTagName('th')[columnIndex];
            const isAscending = !header.classList.contains('sorted-desc');
            
            // Remove sort classes from all headers
            Array.from(table.getElementsByTagName('th')).forEach(th => {
                th.classList.remove('sorted-asc', 'sorted-desc');
            });
            
            // Add sort class to current header
            header.classList.add(isAscending ? 'sorted-desc' : 'sorted-asc');
            
            // Sort rows
            rows.sort((a, b) => {
                let aValue = a.getElementsByTagName('td')[columnIndex].textContent.trim();
                let bValue = b.getElementsByTagName('td')[columnIndex].textContent.trim();
                
                // Handle numeric values
                if (columnIndex >= 5 && columnIndex <= 9) { // Amount columns
                    aValue = parseFloat(aValue.replace(/[^\d.-]/g, '')) || 0;
                    bValue = parseFloat(bValue.replace(/[^\d.-]/g, '')) || 0;
                }
                
                if (isAscending) {
                    return aValue > bValue ? 1 : -1;
                } else {
                    return aValue < bValue ? 1 : -1;
                }
            });
            
            // Reappend rows in sorted order
            rows.forEach(row => tbody.appendChild(row));
        }

        // Export to CSV
        function exportToCSV() {
            const table = document.querySelector('.quickbooks-table');
            const rows = table.querySelectorAll('tr');
            const csv = [];
            
            for (let i = 0; i < rows.length; i++) {
                const row = [];
                const cols = rows[i].querySelectorAll('td, th');
                
                for (let j = 0; j < cols.length; j++) {
                    // Skip action column
                    if (j === cols.length - 1 && cols[j].classList.contains('no-print')) {
                        continue;
                    }
                    
                    let data = cols[j].textContent.trim();
                    data = data.replace(/"/g, '""');
                    row.push('"' + data + '"');
                }
                
                csv.push(row.join(','));
            }
            
            const csvString = csv.join('\n');
            const blob = new Blob([csvString], { type: 'text/csv' });
            const url = window.URL.createObjectURL(blob);
            const a = document.createElement('a');
            a.href = url;
            a.download = 'payroll-summary-{{ $period->period_name }}-{{ date("Y-m-d") }}.csv';
            document.body.appendChild(a);
            a.click();
            document.body.removeChild(a);
            
            showNotification('Summary exported successfully', 'success');
        }

        // ============================================
        // PAYROLL CALCULATION FUNCTIONS
        // ============================================

        // Calculate Payroll Functionality
        document.getElementById('calculatePayrollBtn')?.addEventListener('click', function() {
            const periodId = this.getAttribute('data-period-id');
            const companyId = {{ $company->id }};
            
            // Show confirmation dialog
            if (!confirm('Are you sure you want to calculate payroll for this period?\n\nThis will compute PAYE, NAPSA, NHIMA, and Net Pay for all employees.')) {
                return;
            }
            
            // Show loading
            showNotification('Calculating payroll... This may take a moment.', 'info');
            
            // Disable button during calculation
            const originalText = this.innerHTML;
            this.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Calculating...';
            this.disabled = true;
            
            // Call the calculation endpoint
            fetch(`${apiBaseUrl}/calculate/${periodId}`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                },
                body: JSON.stringify({
                    company_id: companyId,
                    period_id: periodId
                })
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showNotification(data.message || 'Payroll calculated successfully!', 'success');
                    
                    // Reload the page to show the calculated payroll
                    setTimeout(() => {
                        window.location.reload();
                    }, 1500);
                } else {
                    showNotification(data.message || 'Failed to calculate payroll.', 'error');
                    // Re-enable button
                    this.innerHTML = originalText;
                    this.disabled = false;
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('An error occurred while calculating payroll. Please check if the route exists.', 'error');
                // Re-enable button
                this.innerHTML = originalText;
                this.disabled = false;
            });
        });

        // View earnings breakdown for a payroll entry
        function viewEarningsBreakdown(entryId) {
            fetch(`${apiBaseUrl}/entry/${entryId}/breakdown`, {
                headers: {
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    document.getElementById('modalEmployeeName').textContent = data.employee_name;
                    document.getElementById('modalContent').innerHTML = data.html;
                    document.getElementById('earningsBreakdownModal').classList.remove('hidden');
                } else {
                    showNotification(data.message || 'Failed to load earnings breakdown.', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to load earnings breakdown. Please check the route.', 'error');
            });
        }

        // Close modal
        function closeModal() {
            document.getElementById('earningsBreakdownModal').classList.add('hidden');
        }

        // Approve a single payroll entry
        function approveEntry(entryId) {
            if (!confirm('Are you sure you want to approve this payroll entry?')) {
                return;
            }
            
            fetch(`${apiBaseUrl}/entry/${entryId}/approve`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showNotification(data.message || 'Entry approved successfully.', 'success');
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    showNotification(data.message || 'Failed to approve entry.', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to approve entry. Please check the route.', 'error');
            });
        }

        // Bulk approve payroll entries
        function bulkApprovePayroll(periodId) {
            if (!confirm('Are you sure you want to approve all calculated payroll entries?\n\nThis will mark all entries as approved for payment processing.')) {
                return;
            }
            
            showNotification('Approving payroll entries...', 'info');
            
            fetch(`${apiBaseUrl}/period/${periodId}/bulk-approve`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-CSRF-TOKEN': document.querySelector('meta[name="csrf-token"]').content
                }
            })
            .then(response => {
                if (!response.ok) {
                    throw new Error(`HTTP error! status: ${response.status}`);
                }
                return response.json();
            })
            .then(data => {
                if (data.success) {
                    showNotification(data.message || 'All entries approved successfully.', 'success');
                    setTimeout(() => window.location.reload(), 1000);
                } else {
                    showNotification(data.message || 'Failed to approve entries.', 'error');
                }
            })
            .catch(error => {
                console.error('Error:', error);
                showNotification('Failed to bulk approve. Please check the route.', 'error');
            });
        }

        // Export payroll
        function exportPayroll(periodId) {
            window.open(`${apiBaseUrl}/period/${periodId}/export`, '_blank');
        }

        // Initialize tooltips
        document.addEventListener('DOMContentLoaded', function() {
            // Add sort indicators to table headers
            const headers = document.querySelectorAll('.quickbooks-table th');
            headers.forEach((header, index) => {
                if (!header.classList.contains('no-print') && index < headers.length - 1) {
                    header.style.cursor = 'pointer';
                    header.title = 'Click to sort';
                    header.addEventListener('click', () => sortTable(index));
                }
            });
            
            // Add search input if needed
            const tableContainer = document.querySelector('.bg-white.rounded-lg.border');
            const searchDiv = document.createElement('div');
            searchDiv.className = 'p-4 border-b border-gray-200 no-print';
            searchDiv.innerHTML = `
                <div class="flex items-center justify-between">
                    <div class="text-sm text-gray-600">
                        Showing {{ $summaryTotals['total_employees'] }} employees
                    </div>
                    <div class="flex items-center space-x-2">
                        <div class="relative">
                            <input type="text" 
                                   id="searchInput" 
                                   placeholder="Search employees..." 
                                   class="quickbooks-input pl-10"
                                   onkeyup="searchTable()">
                            <i class="fas fa-search absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-400"></i>
                        </div>
                        <button onclick="exportToCSV()" 
                                class="bg-gray-200 text-gray-700 px-4 py-2 rounded text-sm font-medium hover:bg-gray-300 flex items-center">
                            <i class="fas fa-download mr-2"></i> Export Summary
                        </button>
                    </div>
                </div>
            `;
            
            tableContainer.insertBefore(searchDiv, tableContainer.firstChild);
            
            // Show notification if there are pending adjustments
            @if($summaryTotals['total_pending_adjustments'] > 0)
                setTimeout(() => {
                    showNotification('{{ $summaryTotals["total_pending_adjustments"] }} pending adjustments need approval before processing payroll.', 'warning');
                }, 1000);
            @endif

            // Show notification if payroll is already calculated
            @if($payrollEntries->count() > 0)
                setTimeout(() => {
                    showNotification('Payroll has been calculated for this period. Review and approve entries before processing payments.', 'info');
                }, 1500);
            @endif

            // Debug: Log the API base URL
            console.log('API Base URL:', apiBaseUrl);
            console.log('Period ID: {{ $period->id }}');
            console.log('Calculate Route:', `${apiBaseUrl}/calculate/{{ $period->id }}`);
        });

        // Print-friendly styles
        const printStyles = `
            <style>
                @media print {
                    body * {
                        visibility: hidden;
                    }
                    .quickbooks-table, .quickbooks-table * {
                        visibility: visible;
                    }
                    .quickbooks-table {
                        position: absolute;
                        left: 0;
                        top: 0;
                        width: 100%;
                    }
                    .action-btn, .no-print {
                        display: none !important;
                    }
                    th {
                        background-color: #f0f0f0 !important;
                        -webkit-print-color-adjust: exact;
                    }
                }
            </style>
        `;
        
        // Add print styles to head
        document.head.insertAdjacentHTML('beforeend', printStyles);
    </script>

</body>
</html>