<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Review Payroll | {{ config('app.name', 'IMMIA FINANCE') }}</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script>
        tailwind.config = {
            theme: {
                extend: {
                    colors: {
                        primary: '#146CAA',
                        primaryLight: '#E1F0FA',
                        secondary: '#6b7280',
                        success: '#107C10',
                        warning: '#F68C1E',
                        danger: '#D83B01',
                        info: '#0078D4',
                        gray50: '#FAF9F8',
                        gray100: '#F3F2F1',
                        gray200: '#EDEBE9',
                        gray300: '#E1DFDD',
                        gray400: '#C8C6C4',
                        gray500: '#A19F9D',
                        gray600: '#605E5C',
                        gray700: '#323130',
                        gray800: '#201F1E',
                        gray900: '#000000'
                    }
                }
            }
        }
    </script>
    <style>
        .quickbooks-table {
            border-collapse: separate;
            border-spacing: 0;
            width: 100%;
        }
        .quickbooks-table th {
            background-color: #F3F2F1;
            color: #323130;
            font-weight: 600;
            font-size: 13px;
            text-transform: uppercase;
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            text-align: left;
        }
        .quickbooks-table td {
            padding: 12px 16px;
            border-bottom: 1px solid #EDEBE9;
            font-size: 14px;
            color: #323130;
        }
        .quickbooks-table tbody tr:hover {
            background-color: #F3F2F1;
        }
        .quickbooks-badge {
            display: inline-flex;
            align-items: center;
            padding: 4px 8px;
            border-radius: 12px;
            font-size: 12px;
            font-weight: 600;
        }
        .summary-card {
            background: white;
            border-radius: 8px;
            padding: 20px;
            border: 1px solid #EDEBE9;
            margin-bottom: 16px;
        }
        .summary-card h3 {
            font-size: 16px;
            font-weight: 600;
            color: #323130;
            margin-bottom: 16px;
            padding-bottom: 12px;
            border-bottom: 1px solid #EDEBE9;
        }
        .summary-row {
            display: flex;
            justify-content: space-between;
            padding: 8px 0;
            border-bottom: 1px solid #F3F2F1;
        }
        .summary-row:last-child {
            border-bottom: none;
        }
        .summary-total {
            border-top: 2px solid #EDEBE9;
            margin-top: 8px;
            padding-top: 8px;
            font-weight: 600;
        }
    </style>
</head>
<body class="bg-gray50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Review Payroll', 'subtitle' => 'Review and approve payroll calculations'])

        <main class="p-6">
            <!-- Success/Error Messages -->
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-2"></i>
                        <span class="text-green-800 font-semibold">{{ session('success') }}</span>
                    </div>
                </div>
            @endif

            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 rounded-lg p-4">
                    <div class="flex items-center">
                        <i class="fas fa-exclamation-circle text-red-500 mr-2"></i>
                        <span class="text-red-800 font-semibold">{{ session('error') }}</span>
                    </div>
                </div>
            @endif

            <!-- Back Button -->
            <div class="mb-6">
                <a href="{{ route('admin.accounting.payroll.processing.index') }}" 
                   class="text-primary hover:text-primary/90 font-medium text-sm flex items-center">
                    <i class="fas fa-arrow-left mr-2"></i>Back to Payroll Processing
                </a>
            </div>

            <!-- Payroll Run Header -->
            <div class="bg-white rounded-lg border border-gray200 p-6 mb-6">
                <div class="flex flex-col md:flex-row md:items-center justify-between">
                    <div>
                        <h1 class="text-2xl font-bold text-gray900">{{ $payrollRun->run_number }}</h1>
                        <div class="flex items-center space-x-3 mt-2">
                            <span class="quickbooks-badge bg-blue-100 text-blue-800">
                                {{ $payrollRun->payrollPeriod->period_name ?? 'N/A' }}
                            </span>
                            @php
                                $statusColors = [
                                    'draft' => 'bg-gray-100 text-gray-800',
                                    'processing' => 'bg-yellow-100 text-yellow-800',
                                    'calculated' => 'bg-blue-100 text-blue-800',
                                    'approved' => 'bg-green-100 text-green-800',
                                    'paid' => 'bg-success text-white',
                                    'reversed' => 'bg-red-100 text-red-800'
                                ];
                                $statusColor = $statusColors[$payrollRun->status] ?? 'bg-gray-100 text-gray-800';
                            @endphp
                            <span class="quickbooks-badge {{ $statusColor }}">
                                {{ ucfirst($payrollRun->status) }}
                            </span>
                        </div>
                        <p class="text-gray600 text-sm mt-2">
                            Period: {{ $payrollRun->payrollPeriod->formatted_date_range ?? 'N/A' }}
                            | Cutoff: {{ $payrollRun->payrollPeriod->cutoff_date->format('M d, Y') ?? 'N/A' }}
                            | Payment: {{ $payrollRun->payrollPeriod->payment_date->format('M d, Y') ?? 'N/A' }}
                        </p>
                    </div>
                    <div class="mt-4 md:mt-0 flex space-x-3">
                        @if($payrollRun->status === 'calculated')
                            <form action="{{ route('admin.accounting.payroll.processing.approve', $payrollRun->id) }}" method="POST">
                                @csrf
                                <button type="submit" 
                                        class="bg-success text-white px-4 py-2 rounded text-sm font-medium hover:bg-success/90 flex items-center">
                                    <i class="fas fa-check mr-2"></i>Approve Payroll
                                </button>
                            </form>
                        @endif
                        <button onclick="exportPayroll()" 
                                class="border border-gray300 text-gray700 px-4 py-2 rounded text-sm font-medium hover:bg-gray50 flex items-center">
                            <i class="fas fa-download mr-2"></i>Export
                        </button>
                    </div>
                </div>
            </div>

            <!-- Summary Cards -->
            <div class="grid grid-cols-1 md:grid-cols-4 gap-4 mb-6">
                <div class="summary-card">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-primaryLight rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-users text-primary text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Employees</p>
                            <p class="text-xl font-semibold text-gray900">{{ $payrollRun->payrollEntries->count() }}</p>
                        </div>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-green-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-money-bill-wave text-success text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Gross Salary</p>
                            <p class="text-xl font-semibold text-gray900">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_gross_salary, 2) }}</p>
                        </div>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-red-100 rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-percentage text-danger text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Total Deductions</p>
                            <p class="text-xl font-semibold text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_paye_tax + $payrollRun->total_napsa_employee + $payrollRun->total_nhima_employee + $payrollRun->total_other_deductions, 2) }}</p>
                        </div>
                    </div>
                </div>
                <div class="summary-card">
                    <div class="flex items-center">
                        <div class="w-10 h-10 bg-success text-white rounded-lg flex items-center justify-center mr-3">
                            <i class="fas fa-money-check-alt text-lg"></i>
                        </div>
                        <div>
                            <p class="text-sm text-gray600">Net Pay</p>
                            <p class="text-xl font-semibold text-success">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_net_pay, 2) }}</p>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Detailed Summary -->
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6 mb-6">
                <!-- Earnings Summary -->
                <div class="summary-card">
                    <h3>Earnings Summary</h3>
                    <div class="space-y-2">
                        <div class="summary-row">
                            <span class="text-gray600">Basic Salary</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('basic_salary'), 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">Housing Allowance</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('housing_allowance'), 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">Transport Allowance</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('transport_allowance'), 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">Other Allowances</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->payrollEntries->sum('other_allowances'), 2) }}</span>
                        </div>
                        <div class="summary-row summary-total">
                            <span class="text-gray900">Total Gross Salary</span>
                            <span class="text-gray900 font-bold">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_gross_salary, 2) }}</span>
                        </div>
                    </div>
                </div>

                <!-- Deductions Summary -->
                <div class="summary-card">
                    <h3>Deductions Summary</h3>
                    <div class="space-y-2">
                        <div class="summary-row">
                            <span class="text-gray600">PAYE Tax</span>
                            <span class="font-medium text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_paye_tax, 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">NAPSA (Employee)</span>
                            <span class="font-medium text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_napsa_employee, 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">NHIMA (Employee)</span>
                            <span class="font-medium text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_nhima_employee, 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">Other Deductions</span>
                            <span class="font-medium text-danger">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_other_deductions, 2) }}</span>
                        </div>
                        <div class="summary-row summary-total">
                            <span class="text-gray900">Total Deductions</span>
                            <span class="text-gray900 font-bold text-danger">
                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_paye_tax + $payrollRun->total_napsa_employee + $payrollRun->total_nhima_employee + $payrollRun->total_other_deductions, 2) }}
                            </span>
                        </div>
                    </div>
                </div>

                <!-- Employer Contributions -->
                <div class="summary-card">
                    <h3>Employer Contributions</h3>
                    <div class="space-y-2">
                        <div class="summary-row">
                            <span class="text-gray600">NAPSA (Employer)</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_napsa_employer, 2) }}</span>
                        </div>
                        <div class="summary-row">
                            <span class="text-gray600">NHIMA (Employer)</span>
                            <span class="font-medium">{{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_nhima_employer, 2) }}</span>
                        </div>
                        <div class="summary-row summary-total">
                            <span class="text-gray900">Total Employer Cost</span>
                            <span class="text-gray900 font-bold">
                                {{ $company->currency_symbol ?? 'K' }}{{ number_format($payrollRun->total_gross_salary + $payrollRun->total_napsa_employer + $payrollRun->total_nhima_employer, 2) }}
                            </span>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Payroll Entries Table -->
            <div class="bg-white rounded-lg border border-gray200 overflow-hidden mb-6">
                <div class="border-b border-gray200 p-6">
                    <h3 class="text-lg font-semibold text-gray900">Employee Payroll Details</h3>
                    <p class="text-gray600 text-sm mt-1">Detailed breakdown for each employee</p>
                </div>
                
                <div class="overflow-x-auto">
                    <table class="quickbooks-table">
                        <thead>
                            <tr>
                                <th>Employee</th>
                                <th>Basic Salary</th>
                                <th>Gross Salary</th>
                                <th>PAYE</th>
                                <th>NAPSA</th>
                                <th>NHIMA</th>
                                <th>Other Deductions</th>
                                <th>Net Pay</th>
                                <th>Actions</th>
                            </tr>
                        </thead>
                        <tbody>
                            @foreach($payrollRun->payrollEntries as $entry)
                                <tr class="hover:bg-gray100 transition-colors">
                                    <td>
                                        <div class="flex items-center">
                                            <div class="w-8 h-8 bg-primaryLight rounded-full flex items-center justify-center mr-3">
                                                <span class="text-primary font-semibold text-sm">
                                                    {{ strtoupper(substr($entry->employee->first_name, 0, 1)) }}
                                                </span>
                                            </div>
                                            <div>
                                                <div class="font-medium text-gray900">{{ $entry->employee->first_name }} {{ $entry->employee->last_name }}</div>
                                                <div class="text-xs text-gray500">{{ $entry->employee->employee_number }}</div>
                                            </div>
                                        </div>
                                    </td>
                                    <td class="text-gray700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->basic_salary, 2) }}
                                    </td>
                                    <td class="text-gray700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->gross_salary, 2) }}
                                    </td>
                                    <td class="text-danger font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->paye_tax, 2) }}
                                    </td>
                                    <td class="text-gray700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->napsa_employee, 2) }}
                                    </td>
                                    <td class="text-gray700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->nhima_employee, 2) }}
                                    </td>
                                    <td class="text-gray700 font-medium">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->other_deductions, 2) }}
                                    </td>
                                    <td class="text-success font-bold">
                                        {{ $company->currency_symbol ?? 'K' }}{{ number_format($entry->net_pay, 2) }}
                                    </td>
                                    <td>
                                        <button onclick="viewEntryDetails({{ $entry->id }})" 
                                                class="text-primary hover:text-primary/90 text-sm font-medium">
                                            View Details
                                        </button>
                                    </td>
                                </tr>
                            @endforeach
                        </tbody>
                    </table>
                </div>
            </div>

            <!-- Approval Actions -->
            @if($payrollRun->status === 'calculated')
                <div class="bg-white rounded-lg border border-gray200 p-6">
                    <div class="flex flex-col md:flex-row md:items-center justify-between">
                        <div>
                            <h3 class="text-lg font-semibold text-gray900">Ready for Approval</h3>
                            <p class="text-gray600 text-sm mt-1">
                                Review all calculations before approving. Once approved, payroll can be processed for payment.
                            </p>
                        </div>
                        <div class="mt-4 md:mt-0 flex space-x-3">
                            <a href="{{ route('admin.accounting.payroll.processing.index') }}" 
                               class="px-4 py-2 border border-gray300 rounded text-sm font-medium text-gray700 hover:bg-gray50">
                                Cancel
                            </a>
                            <form action="{{ route('admin.accounting.payroll.processing.approve', $payrollRun->id) }}" method="POST">
                                @csrf
                                <button type="submit" 
                                        class="px-4 py-2 bg-success text-white rounded text-sm font-medium hover:bg-success/90 flex items-center">
                                    <i class="fas fa-check mr-2"></i>Approve Payroll
                                </button>
                            </form>
                        </div>
                    </div>
                </div>
            @endif
        </main>
    </div>

    <!-- Entry Details Modal -->
    <div id="entryDetailsModal" class="fixed inset-0 bg-gray900 bg-opacity-50 hidden items-center justify-center z-50 p-4">
        <div class="bg-white rounded-lg shadow-lg max-w-2xl w-full">
            <div class="p-6">
                <div class="flex items-center justify-between mb-4">
                    <div>
                        <h3 class="text-lg font-semibold text-gray900" id="entryEmployeeName"></h3>
                        <p class="text-gray600 text-sm" id="entryEmployeeNumber"></p>
                    </div>
                    <button onclick="closeEntryModal()" class="text-gray500 hover:text-gray700">
                        <i class="fas fa-times text-lg"></i>
                    </button>
                </div>
                
                <div class="space-y-6">
                    <!-- Earnings -->
                    <div>
                        <h4 class="text-sm font-medium text-gray700 mb-3">Earnings</h4>
                        <div class="space-y-2">
                            <div class="summary-row">
                                <span class="text-gray600">Basic Salary</span>
                                <span class="font-medium" id="entryBasicSalary"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">Housing Allowance</span>
                                <span class="font-medium" id="entryHousingAllowance"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">Transport Allowance</span>
                                <span class="font-medium" id="entryTransportAllowance"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">Other Allowances</span>
                                <span class="font-medium" id="entryOtherAllowances"></span>
                            </div>
                            <div class="summary-row summary-total">
                                <span class="text-gray900">Gross Salary</span>
                                <span class="text-gray900 font-bold" id="entryGrossSalary"></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Deductions -->
                    <div>
                        <h4 class="text-sm font-medium text-gray700 mb-3">Deductions</h4>
                        <div class="space-y-2">
                            <div class="summary-row">
                                <span class="text-gray600">PAYE Tax</span>
                                <span class="font-medium text-danger" id="entryPayeTax"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">NAPSA (Employee)</span>
                                <span class="font-medium text-danger" id="entryNapsaEmployee"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">NHIMA (Employee)</span>
                                <span class="font-medium text-danger" id="entryNhimaEmployee"></span>
                            </div>
                            <div class="summary-row">
                                <span class="text-gray600">Other Deductions</span>
                                <span class="font-medium text-danger" id="entryOtherDeductions"></span>
                            </div>
                            <div class="summary-row summary-total">
                                <span class="text-gray900">Total Deductions</span>
                                <span class="text-gray900 font-bold text-danger" id="entryTotalDeductions"></span>
                            </div>
                        </div>
                    </div>
                    
                    <!-- Net Pay -->
                    <div class="bg-gray50 p-4 rounded-lg">
                        <div class="flex items-center justify-between">
                            <span class="text-lg font-semibold text-gray900">Net Pay</span>
                            <span class="text-2xl font-bold text-success" id="entryNetPay"></span>
                        </div>
                    </div>
                </div>
            </div>
            
            <div class="border-t border-gray200 p-6 flex justify-end">
                <button onclick="closeEntryModal()" class="px-4 py-2 bg-primary text-white rounded text-sm font-medium hover:bg-primary/90">
                    Close
                </button>
            </div>
        </div>
    </div>

    <script>
        let entryData = @json($payrollRun->payrollEntries->map(function($entry) {
            return [
                'id' => $entry->id,
                'employee_name' => $entry->employee->first_name . ' ' . $entry->employee->last_name,
                'employee_number' => $entry->employee->employee_number,
                'basic_salary' => number_format($entry->basic_salary, 2),
                'housing_allowance' => number_format($entry->housing_allowance, 2),
                'transport_allowance' => number_format($entry->transport_allowance, 2),
                'other_allowances' => number_format($entry->other_allowances, 2),
                'gross_salary' => number_format($entry->gross_salary, 2),
                'paye_tax' => number_format($entry->paye_tax, 2),
                'napsa_employee' => number_format($entry->napsa_employee, 2),
                'nhima_employee' => number_format($entry->nhima_employee, 2),
                'other_deductions' => number_format($entry->other_deductions, 2),
                'total_deductions' => number_format($entry->total_deductions, 2),
                'net_pay' => number_format($entry->net_pay, 2)
            ];
        }));
        
        function viewEntryDetails(entryId) {
            const entry = entryData.find(e => e.id == entryId);
            if (!entry) return;
            
            const currencySymbol = '{{ $company->currency_symbol ?? "K" }}';
            
            document.getElementById('entryEmployeeName').textContent = entry.employee_name;
            document.getElementById('entryEmployeeNumber').textContent = 'Employee #: ' + entry.employee_number;
            document.getElementById('entryBasicSalary').textContent = currencySymbol + entry.basic_salary;
            document.getElementById('entryHousingAllowance').textContent = currencySymbol + entry.housing_allowance;
            document.getElementById('entryTransportAllowance').textContent = currencySymbol + entry.transport_allowance;
            document.getElementById('entryOtherAllowances').textContent = currencySymbol + entry.other_allowances;
            document.getElementById('entryGrossSalary').textContent = currencySymbol + entry.gross_salary;
            document.getElementById('entryPayeTax').textContent = currencySymbol + entry.paye_tax;
            document.getElementById('entryNapsaEmployee').textContent = currencySymbol + entry.napsa_employee;
            document.getElementById('entryNhimaEmployee').textContent = currencySymbol + entry.nhima_employee;
            document.getElementById('entryOtherDeductions').textContent = currencySymbol + entry.other_deductions;
            document.getElementById('entryTotalDeductions').textContent = currencySymbol + entry.total_deductions;
            document.getElementById('entryNetPay').textContent = currencySymbol + entry.net_pay;
            
            document.getElementById('entryDetailsModal').classList.remove('hidden');
        }
        
        function closeEntryModal() {
            document.getElementById('entryDetailsModal').classList.add('hidden');
        }
        
        function exportPayroll() {
            const exportType = confirm('Export as:\n\nClick OK for Excel\nClick Cancel for PDF');
            
            if (exportType) {
                // Export to Excel
                window.open(`/admin/accounting/payroll/processing/export/{{ $payrollRun->id }}?format=excel`, '_blank');
            } else {
                // Export to PDF
                window.open(`/admin/accounting/payroll/processing/export/{{ $payrollRun->id }}?format=pdf`, '_blank');
            }
        }
        
        // Close modal on escape key
        document.addEventListener('keydown', function(e) {
            if (e.key === 'Escape') {
                closeEntryModal();
            }
        });
    </script>
</body>
</html>