@php
    // Define page variables
    $pageTitle = 'Loan Products';
    $pageSubtitle = 'Manage product terms, rates, fees, and penalties';

    // Sidebar/Header Stats integration
    $stats_sidebar = [
        'pending_applications' => $stats['totalApplications'] ?? 0, 
    ];
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | ZedBankOS</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; 
            --zedbank-blue-dark: #005A94; 
            --zedbank-blue-medium: #005A94; 
        }

        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .border-zedbank-blue { border-color: var(--zedbank-blue); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Table Styles */
        .quickbooks-table { border-collapse: separate; border-spacing: 0; width: 100%; }
        .quickbooks-table th { background-color: #F3F2F1; color: #323130; font-weight: 600; font-size: 11px; text-transform: uppercase; padding: 12px 16px; border-bottom: 1px solid #EDEBE9; text-align: left; }
        .quickbooks-table td { padding: 12px 16px; border-bottom: 1px solid #EDEBE9; font-size: 13px; color: #323130; }
        .quickbooks-table tbody tr:hover { background-color: #F9FAFB; }
        
        .action-btn { width: 30px; height: 30px; border-radius: 4px; display: inline-flex; align-items: center; justify-content: center; transition: all 0.2s; }
        .status-badge { display: inline-flex; align-items: center; padding: 2px 10px; border-radius: 12px; font-size: 11px; font-weight: 600; }
        
        /* Modal Style */
        .modal-overlay { display: none; position: fixed; inset: 0; background: rgba(0, 0, 0, 0.5); backdrop-filter: blur(4px); z-index: 1000; align-items: center; justify-content: center; }
        .modal-overlay.active { display: flex !important; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    @include('accounting.partials.sidebar', ['stats' => $stats_sidebar])

    <div class="lg:ml-64 min-h-screen flex flex-col">
        
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6 flex-grow">
            <div class="max-w-7xl mx-auto">
                
                @if(session('success'))
                    <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg flex justify-between items-center shadow-sm">
                        <span><i class="fas fa-check-circle mr-2"></i>{{ session('success') }}</span>
                        <button onclick="this.parentElement.remove()"><i class="fas fa-times"></i></button>
                    </div>
                @endif

                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
                    <div class="flex flex-col md:flex-row justify-between items-center gap-4">
                        <div class="w-full md:w-auto flex items-center gap-4">
                            <div class="relative">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <i class="fas fa-search text-gray-400 text-xs"></i>
                                </div>
                                <input 
                                    type="text" 
                                    id="searchInput" 
                                    class="block w-full md:w-80 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-zedbank-blue focus:border-zedbank-blue focus:ring-1 text-sm outline-none" 
                                    placeholder="Search products, rates, or status..."
                                >
                            </div>
                        </div>
                        <div class="flex flex-wrap gap-2 w-full md:w-auto">
                            <a href="{{ route('admin.accounting.products.create') }}" class="px-4 py-2 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark transition flex items-center gap-2 text-sm font-semibold shadow-sm">
                                <i class="fas fa-plus"></i>
                                <span>New Product</span>
                            </a>
                            <button class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2 text-sm">
                                <i class="fas fa-download"></i>
                                <span>Export</span>
                            </button>
                        </div>
                    </div>
                </div>

                <div class="bg-white shadow-sm rounded-lg border border-gray-200 overflow-hidden">
                    <div class="overflow-x-auto">
                        <table class="quickbooks-table" id="dataTable">
                            <thead>
                                <tr>
                                    <th class="px-4 py-3">Product Details</th>
                                    <th class="px-4 py-3">Daily</th>
                                    <th class="px-4 py-3">Weekly</th>
                                    <th class="px-4 py-3">Monthly</th>
                                    <th class="px-4 py-3">Yearly</th>
                                    <th class="px-4 py-3">Proc. Fee</th>
                                    <th class="px-4 py-3">Penalty</th>
                                    <th class="px-4 py-3">Status</th>
                                    <th class="px-4 py-3 text-center">Actions</th>
                                </tr>
                            </thead>

                            <tbody>
                                @forelse($loanTypes as $loanType)
                                    <tr class="product-row border-b last:border-0 transition">
                                        <td class="px-4 py-3">
                                            <div class="font-bold text-gray-900">{{ $loanType->product }}</div>
                                            <div class="text-[10px] text-gray-500 uppercase font-bold tracking-tighter">
                                                {{ str_replace('_', ' ', $loanType->interest_method) }}
                                            </div>
                                        </td>
                                        <td class="px-4 py-3 text-zedbank-blue font-medium">{{ number_format($loanType->dailyrate, 2) }}%</td>
                                        <td class="px-4 py-3 text-zedbank-blue font-medium">{{ number_format($loanType->weeklyrate, 2) }}%</td>
                                        <td class="px-4 py-3 text-zedbank-blue font-bold bg-blue-50/30">{{ number_format($loanType->monthlyrate, 2) }}%</td>
                                        <td class="px-4 py-3 text-zedbank-blue font-medium">{{ number_format($loanType->yearlyrate, 2) }}%</td>
                                        <td class="px-4 py-3">
                                            <div class="text-xs font-semibold">{{ number_format($loanType->processing_fee, 2) }}</div>
                                            <div class="text-[9px] text-gray-400 italic capitalize">{{ str_replace('_', ' ', $loanType->processing_fee_basis) }}</div>
                                        </td>
                                        <td class="px-4 py-3">
                                            <div class="text-xs font-semibold text-red-600">{{ number_format($loanType->penalty_rate, 2) }}</div>
                                            <div class="text-[9px] text-gray-400 italic capitalize">{{ str_replace('_', ' ', $loanType->penalty_basis) }}</div>
                                        </td>
                                        <td class="px-4 py-3">
                                            <span class="status-badge {{ $loanType->status === 'Active' ? 'bg-green-100 text-green-800' : 'bg-gray-100 text-gray-600' }}">
                                                {{ $loanType->status }}
                                            </span>
                                        </td>
                                        <td class="px-4 py-3">
                                            <div class="flex items-center justify-center space-x-1">
                                                <a href="{{ route('admin.accounting.products.show', $loanType->id) }}" class="action-btn bg-gray-100 text-gray-600 hover:bg-zedbank-blue hover:text-white" title="View"><i class="fas fa-eye text-[10px]"></i></a>
                                                <a href="{{ route('admin.accounting.products.edit', $loanType->id) }}" class="action-btn bg-blue-50 text-blue-600 hover:bg-blue-600 hover:text-white" title="Edit"><i class="fas fa-edit text-[10px]"></i></a>
                                                <a href="{{ route('admin.accounting.products.assign-requirements', $loanType->id) }}" class="action-btn bg-purple-50 text-purple-600 hover:bg-purple-600 hover:text-white" title="Requirements"><i class="fas fa-clipboard-list text-[10px]"></i></a>
                                                <button onclick="toggleStatus({{ $loanType->id }})" class="action-btn bg-green-50 text-green-600 hover:bg-green-600 hover:text-white" title="Toggle Status"><i class="fas fa-power-off text-[10px]"></i></button>
                                                <button onclick="confirmDelete({{ $loanType->id }})" class="action-btn bg-red-50 text-red-600 hover:bg-red-600 hover:text-white" title="Delete"><i class="fas fa-trash text-[10px]"></i></button>
                                            </div>
                                        </td>
                                    </tr>
                                @empty
                                    <tr>
                                        <td colspan="9" class="px-4 py-12 text-center text-gray-500 italic">
                                            <i class="fas fa-boxes text-3xl mb-3 block text-gray-300"></i>
                                            No loan products found. Start by creating a new product.
                                        </td>
                                    </tr>
                                @endforelse
                            </tbody>
                        </table>
                    </div>

                    @if($loanTypes->hasPages())
                        <div class="px-4 py-4 border-t border-gray-200 bg-gray-50">
                            {{ $loanTypes->links() }}
                        </div>
                    @endif
                </div>
            </div>
        </main>
    </div>

    <div id="deleteModal" class="modal-overlay">
        <div class="bg-white rounded-lg p-6 max-w-sm w-full shadow-xl animate-fade-in">
            <div class="flex items-center justify-center text-red-600 mb-4">
                <i class="fas fa-exclamation-triangle text-4xl"></i>
            </div>
            <h3 class="text-lg font-bold text-gray-900 mb-2 text-center">Delete Product?</h3>
            <p class="text-xs text-gray-600 mb-6 text-center leading-relaxed">Are you sure you want to delete this product? This action cannot be undone and will remove all associated settings.</p>
            <div class="flex justify-center gap-3">
                <button onclick="closeModal()" class="px-4 py-2 border rounded-lg text-xs font-semibold hover:bg-gray-50 transition">Cancel</button>
                <form id="deleteForm" method="POST">
                    @csrf @method('DELETE')
                    <button type="submit" class="px-4 py-2 bg-red-600 text-white rounded-lg text-xs font-semibold hover:bg-red-700 transition">Delete Permanently</button>
                </form>
            </div>
        </div>
    </div>

    <script>
        // Toggle Status via AJAX/Fetch
        function toggleStatus(id) {
            if(confirm('Change product status?')) {
                fetch(`/admin/accounting/products/${id}/toggle-status`, { 
                    method: 'POST', 
                    headers: { 'X-CSRF-TOKEN': '{{ csrf_token() }}', 'Content-Type': 'application/json' } 
                }).then(() => location.reload());
            }
        }

        // Delete Modal Logic
        function confirmDelete(id) {
            document.getElementById('deleteForm').action = `/admin/accounting/products/${id}`;
            document.getElementById('deleteModal').classList.add('active');
        }
        function closeModal() { 
            document.getElementById('deleteModal').classList.remove('active'); 
        }

        // Real-time Search
        document.getElementById('searchInput').addEventListener('keyup', function() {
            let filter = this.value.toLowerCase();
            document.querySelectorAll('.product-row').forEach(row => {
                row.style.display = row.innerText.toLowerCase().includes(filter) ? '' : 'none';
            });
        });
    </script>
</body>
</html>