<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Add Currency Rate | IMMIA FINANCE</title>
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
</head>
<body class="bg-gray-50">
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64">
        @include('accounting.partials.header', ['title' => 'Add Currency Rate', 'subtitle' => 'Set exchange rate between currencies'])

        <main class="p-6">
            <div class="max-w-2xl mx-auto">
                <!-- Back Button -->
                <div class="mb-6">
                    <a href="{{ route('admin.accounting.settings.currency-rates.index') }}" 
                       class="inline-flex items-center text-blue-600 hover:text-blue-700 font-medium">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Back to Currency Rates
                    </a>
                </div>

                <!-- Success/Error Messages -->
                @if($errors->any())
                    <div class="bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded-lg mb-6">
                        <ul class="list-disc list-inside">
                            @foreach($errors->all() as $error)
                                <li>{{ $error }}</li>
                            @endforeach
                        </ul>
                    </div>
                @endif

                <!-- Debug: Show what's being submitted -->
                @if(session('debug'))
                    <div class="bg-yellow-100 border border-yellow-400 text-yellow-700 px-4 py-3 rounded-lg mb-6">
                        <pre>{{ session('debug') }}</pre>
                    </div>
                @endif

                <!-- Form Card -->
                <div class="bg-white rounded-xl shadow-sm border">
                    <div class="px-6 py-4 border-b">
                        <h3 class="text-lg font-semibold text-gray-900">Set Exchange Rate</h3>
                        <p class="text-sm text-gray-600 mt-1">Set exchange rate between two currencies</p>
                    </div>
                    
                    <form action="{{ route('admin.accounting.settings.currency-rates.store') }}" method="POST">
                        @csrf
                        <div class="p-6 space-y-6">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Base Currency -->
                                <div>
                                    <label for="base_currency_id" class="block text-sm font-medium text-gray-700 mb-2">
                                        Base Currency *
                                    </label>
                                    <select name="base_currency_id" id="base_currency_id" 
                                            class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                        <option value="">Select Base Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->id }}" {{ old('base_currency_id') == $currency->id ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('base_currency_id')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>

                                <!-- Target Currency -->
                                <div>
                                    <label for="target_currency_id" class="block text-sm font-medium text-gray-700 mb-2">
                                        Target Currency *
                                    </label>
                                    <select name="target_currency_id" id="target_currency_id" 
                                            class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                        <option value="">Select Target Currency</option>
                                        @foreach($currencies as $currency)
                                            <option value="{{ $currency->id }}" {{ old('target_currency_id') == $currency->id ? 'selected' : '' }}>
                                                {{ $currency->code }} - {{ $currency->name }}
                                            </option>
                                        @endforeach
                                    </select>
                                    @error('target_currency_id')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>
                            </div>

                            <div class="grid grid-cols-1 md:grid-cols-2 gap-6">
                                <!-- Exchange Rate -->
                                <div>
                                    <label for="rate" class="block text-sm font-medium text-gray-700 mb-2">
                                        Exchange Rate *
                                    </label>
                                    <input type="number" name="rate" id="rate" step="0.000001" min="0.000001"
                                           value="{{ old('rate') }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent"
                                           placeholder="0.000000" required>
                                    @error('rate')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                    <p class="text-xs text-gray-500 mt-1">1 Base = X Target</p>
                                </div>

                                <!-- Rate Date -->
                                <div>
                                    <label for="rate_date" class="block text-sm font-medium text-gray-700 mb-2">
                                        Rate Date *
                                    </label>
                                    <input type="date" name="rate_date" id="rate_date" 
                                           value="{{ old('rate_date', date('Y-m-d')) }}"
                                           class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent" required>
                                    @error('rate_date')
                                        <p class="text-red-500 text-xs mt-1">{{ $message }}</p>
                                    @enderror
                                </div>
                            </div>

                            <!-- Rate Display -->
                            <div class="bg-gray-50 rounded-lg p-4">
                                <h4 class="text-sm font-medium text-gray-700 mb-2">Rate Preview</h4>
                                <p class="text-lg font-semibold text-gray-900" id="ratePreview">
                                    1 <span id="baseCurrencyCode">[Base]</span> = <span id="rateValue">0.000000</span> <span id="targetCurrencyCode">[Target]</span>
                                </p>
                            </div>
                        </div>

                        <!-- Form Actions -->
                        <div class="px-6 py-4 bg-gray-50 border-t rounded-b-xl flex justify-end space-x-3">
                            <a href="{{ route('admin.accounting.settings.currency-rates.index') }}" 
                               class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 font-medium">
                                Cancel
                            </a>
                            <button type="submit" 
                                    class="bg-blue-600 hover:bg-blue-700 text-white px-6 py-2 rounded-lg font-medium flex items-center space-x-2">
                                <i class="fas fa-save"></i>
                                <span>Save Exchange Rate</span>
                            </button>
                        </div>
                    </form>
                </div>
            </div>
        </main>
    </div>

    <script>
        // Set default date to today
        document.addEventListener('DOMContentLoaded', function() {
            const today = new Date().toISOString().split('T')[0];
            if (!document.getElementById('rate_date').value) {
                document.getElementById('rate_date').value = today;
            }

            // Update rate preview
            const baseCurrencySelect = document.getElementById('base_currency_id');
            const targetCurrencySelect = document.getElementById('target_currency_id');
            const rateInput = document.getElementById('rate');
            const rateValue = document.getElementById('rateValue');
            const baseCurrencyCode = document.getElementById('baseCurrencyCode');
            const targetCurrencyCode = document.getElementById('targetCurrencyCode');

            function updateRatePreview() {
                const baseCurrency = baseCurrencySelect.options[baseCurrencySelect.selectedIndex];
                const targetCurrency = targetCurrencySelect.options[targetCurrencySelect.selectedIndex];
                const baseCode = baseCurrency.text.split(' - ')[0] || '[Base]';
                const targetCode = targetCurrency.text.split(' - ')[0] || '[Target]';
                const rate = rateInput.value || '0.000000';
                
                baseCurrencyCode.textContent = baseCode;
                targetCurrencyCode.textContent = targetCode;
                rateValue.textContent = parseFloat(rate).toFixed(6);
            }

            baseCurrencySelect.addEventListener('change', updateRatePreview);
            targetCurrencySelect.addEventListener('change', updateRatePreview);
            rateInput.addEventListener('input', updateRatePreview);
            
            // Initial update
            updateRatePreview();
        });
    </script>
</body>
</html>