@extends('layouts.admin')

@section('title', 'Application Requirements')
@section('page-title', 'Application Requirements')
@section('page-description', 'Manage application requirements for your company')

@section('content')
<div class="max-w-7xl mx-auto">
    <!-- Header Section -->
    <div class="mb-8">
        <h1 class="text-3xl font-bold text-gray-800 mb-2">Application Requirements</h1>
        <p class="text-gray-600">Manage document requirements for loan applications</p>
        @if(auth()->user()->company)
            <div class="mt-2 inline-flex items-center px-3 py-1 bg-blue-100 text-blue-800 rounded-full text-sm">
                <i class="fas fa-building mr-2"></i>
                {{ auth()->user()->company->name }}
            </div>
        @endif
    </div>

    <!-- Search and Controls Section -->
    <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
        <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
            <div class="w-full md:w-auto">
                <div class="relative">
                    <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                        <i class="fas fa-search text-gray-400"></i>
                    </div>
                    <input 
                        type="text" 
                        id="searchInput" 
                        class="block w-full md:w-80 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary" 
                        placeholder="Search by document name..."
                    >
                </div>
            </div>
            <div class="flex flex-wrap gap-2 w-full md:w-auto">
                <button 
                    type="button"
                    onclick="openAddModal()"
                    class="px-4 py-2 bg-primary text-white rounded-lg hover:bg-secondary transition flex items-center gap-2">
                    <i class="fas fa-plus"></i>
                    <span>Add Requirement</span>
                </button>
                <button id="filterButton" type="button" class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2">
                    <i class="fas fa-filter"></i>
                    <span>Filter by Status</span>
                </button>
            </div>
        </div>
        
        <!-- Filter Options (Hidden by default) -->
        <div id="filterOptions" class="hidden mt-4 border-t border-gray-200 pt-4">
            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                <div>
                    <label for="statusFilter" class="block text-sm font-medium text-gray-700 mb-1">Status</label>
                    <select id="statusFilter" class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary">
                        <option value="">All Status</option>
                        <option value="active">Active</option>
                        <option value="inactive">Inactive</option>
                    </select>
                </div>
                <div class="flex items-end">
                    <button id="clearFilters" type="button" class="px-4 py-2 bg-gray-100 border border-gray-300 rounded-lg hover:bg-gray-200 transition">
                        Clear Filters
                    </button>
                </div>
            </div>
        </div>
    </div>

    @if(session('success'))
        <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
            <i class="fas fa-check-circle mr-2"></i>{{ session('success') }}
        </div>
    @endif

    @if(session('error'))
        <div class="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
            <i class="fas fa-exclamation-circle mr-2"></i>{{ session('error') }}
        </div>
    @endif

    <!-- Table Section -->
    <div class="overflow-x-auto bg-white shadow-sm rounded-lg border border-gray-200">
        <table class="min-w-full text-sm" id="requirementsTable">
            <thead class="bg-gray-50 border-b border-gray-200">
                <tr>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Document Name</th>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Description</th>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Status</th>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Created By</th>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Created At</th>
                    <th class="px-4 py-3 text-left text-gray-600 font-medium">Actions</th>
                </tr>
            </thead>

            <tbody>
                @forelse($requirements as $requirement)
                    <tr class="border-b hover:bg-gray-50 transition" 
                        data-status="{{ $requirement->status }}">
                        <td class="px-4 py-3 text-gray-700 font-medium">{{ $requirement->documentname }}</td>
                        <td class="px-4 py-3 text-gray-700">
                            {{ $requirement->description ?? 'No description' }}
                        </td>
                        <td class="px-4 py-3">
                            <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                                {{ $requirement->status == 'active' ? 'bg-green-100 text-green-800' : 'bg-red-100 text-red-800' }}">
                                {{ ucfirst($requirement->status) }}
                            </span>
                        </td>
                        <td class="px-4 py-3 text-gray-700">{{ $requirement->createdby }}</td>
                        <td class="px-4 py-3 text-gray-700">
                            {{ \Carbon\Carbon::parse($requirement->createdat)->format('M d, Y H:i') }}
                        </td>
                        <td class="px-4 py-3">
                            <div class="flex space-x-2">
                                <!-- Edit Button -->
                                <button 
                                    type="button"
                                    onclick="openEditModal(this)"
                                    data-id="{{ $requirement->id }}"
                                    data-documentname="{{ $requirement->documentname }}"
                                    data-description="{{ $requirement->description }}"
                                    data-status="{{ $requirement->status }}"
                                    class="text-blue-600 hover:text-blue-800 p-1 rounded transition" 
                                    title="Edit">
                                    <i class="fas fa-edit"></i>
                                </button>
                                
                                <!-- Block/Delete Button -->
                                <form method="POST" 
                                      action="{{ route('application-requirements.block', $requirement->id) }}" 
                                      class="inline"
                                      onsubmit="return confirm('Are you sure you want to {{ $requirement->status == 'active' ? 'deactivate' : 'activate' }} this requirement?')">
                                    @csrf
                                    @method('DELETE')
                                    <button type="submit" 
                                            class="text-red-600 hover:text-red-800 p-1 rounded transition" 
                                            title="{{ $requirement->status == 'active' ? 'Deactivate' : 'Activate' }}">
                                        <i class="fas fa-ban"></i>
                                    </button>
                                </form>
                            </div>
                        </td>
                    </tr>
                @empty
                    <tr>
                        <td colspan="6" class="px-4 py-8 text-center text-gray-500">
                            <div class="flex flex-col items-center justify-center">
                                <i class="fas fa-file-alt text-4xl text-gray-300 mb-4"></i>
                                <p class="text-lg">No application requirements found</p>
                                <p class="text-sm">Get started by adding your first requirement</p>
                            </div>
                        </td>
                    </tr>
                @endforelse
            </tbody>
        </table>
    </div>

    <!-- Add Requirement Modal -->
    <div id="addRequirementModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Add New Requirement</h3>
                
                @if(auth()->user()->company)
                    <div class="mb-4 p-3 bg-blue-50 border border-blue-200 rounded-lg">
                        <p class="text-sm text-blue-700">
                            <i class="fas fa-info-circle mr-1"></i>
                            This requirement will be added to: <strong>{{ auth()->user()->company->name }}</strong>
                        </p>
                    </div>
                @endif
                
                <form action="{{ route('application-requirements.store') }}" method="POST">
                    @csrf
                    <div class="space-y-4">
                        <div>
                            <label for="documentname" class="block text-sm font-medium text-gray-700 mb-1">Document Name *</label>
                            <input type="text" 
                                   class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary" 
                                   id="documentname" 
                                   name="documentname" 
                                   required
                                   placeholder="e.g., Identification Proof">
                        </div>
                        
                        <div>
                            <label for="description" class="block text-sm font-medium text-gray-700 mb-1">Description</label>
                            <textarea class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary" 
                                      id="description" 
                                      name="description" 
                                      rows="3"
                                      placeholder="Optional description of the document requirement"></textarea>
                        </div>
                    </div>
                    
                    <div class="mt-6 flex justify-end space-x-3">
                        <button type="button" 
                                onclick="closeAddModal()"
                                class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition">Cancel</button>
                        <button type="submit" 
                                class="px-4 py-2 bg-primary text-white rounded-lg hover:bg-secondary transition">Save Requirement</button>
                    </div>
                </form>
            </div>
        </div>
    </div>

    <!-- Edit Requirement Modal -->
    <div id="editRequirementModal" class="fixed inset-0 bg-gray-600 bg-opacity-50 overflow-y-auto h-full w-full hidden z-50">
        <div class="relative top-20 mx-auto p-5 border w-96 shadow-lg rounded-md bg-white">
            <div class="mt-3">
                <h3 class="text-lg font-semibold text-gray-800 mb-4">Edit Requirement</h3>
                
                <form id="editRequirementForm" method="POST">
                    @csrf
                    @method('PUT')
                    <input type="hidden" id="edit_id" name="id">
                    
                    <div class="space-y-4">
                        <div>
                            <label for="edit_documentname" class="block text-sm font-medium text-gray-700 mb-1">Document Name *</label>
                            <input type="text" 
                                   class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary" 
                                   id="edit_documentname" 
                                   name="documentname" 
                                   required>
                        </div>
                        
                        <div>
                            <label for="edit_description" class="block text-sm font-medium text-gray-700 mb-1">Description</label>
                            <textarea class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary" 
                                      id="edit_description" 
                                      name="description" 
                                      rows="3"></textarea>
                        </div>
                        
                        <div>
                            <label for="edit_status" class="block text-sm font-medium text-gray-700 mb-1">Status *</label>
                            <select class="w-full border border-gray-300 rounded-lg px-3 py-2 focus:ring-primary focus:border-primary" 
                                    id="edit_status" 
                                    name="status" 
                                    required>
                                <option value="active">Active</option>
                                <option value="inactive">Inactive</option>
                            </select>
                        </div>
                    </div>
                    
                    <div class="mt-6 flex justify-end space-x-3">
                        <button type="button" 
                                onclick="closeEditModal()"
                                class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition">Cancel</button>
                        <button type="submit" 
                                class="px-4 py-2 bg-primary text-white rounded-lg hover:bg-secondary transition">Update Requirement</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
</div>
@endsection

@section('scripts')
<script>
    // Search functionality
    document.getElementById('searchInput')?.addEventListener('input', function() {
        const searchTerm = this.value.toLowerCase();
        const rows = document.querySelectorAll('#requirementsTable tbody tr');
        let visibleCount = 0;
        
        rows.forEach(row => {
            const rowText = row.textContent.toLowerCase();
            if (rowText.includes(searchTerm)) {
                row.style.display = '';
                visibleCount++;
            } else {
                row.style.display = 'none';
            }
        });
    });

    // Toggle filter options
    document.getElementById('filterButton')?.addEventListener('click', function() {
        const filterOptions = document.getElementById('filterOptions');
        filterOptions.classList.toggle('hidden');
    });

    // Filter functionality
    const statusFilter = document.getElementById('statusFilter');
    const tableRows = document.querySelectorAll('#requirementsTable tbody tr');
    
    function filterTable() {
        const selectedStatus = statusFilter?.value || '';
        
        tableRows.forEach(row => {
            const status = row.getAttribute('data-status');
            
            let showRow = true;
            
            if (selectedStatus && status !== selectedStatus) {
                showRow = false;
            }
            
            row.style.display = showRow ? '' : 'none';
        });
    }
    
    if (statusFilter) {
        statusFilter.addEventListener('change', filterTable);
    }

    // Clear filters
    document.getElementById('clearFilters')?.addEventListener('click', function() {
        if (statusFilter) statusFilter.value = '';
        
        // Show all rows
        tableRows.forEach(row => {
            row.style.display = '';
        });
        
        // Hide filter options
        document.getElementById('filterOptions')?.classList.add('hidden');
    });

    // Modal functions
    function openAddModal() {
        document.getElementById('addRequirementModal').classList.remove('hidden');
        document.getElementById('addRequirementModal').classList.add('flex');
        
        // Clear form fields
        document.getElementById('documentname').value = '';
        document.getElementById('description').value = '';
    }
    
    function closeAddModal() {
        document.getElementById('addRequirementModal').classList.add('hidden');
        document.getElementById('addRequirementModal').classList.remove('flex');
    }
    
    function openEditModal(button) {
        const id = button.getAttribute('data-id');
        const documentname = button.getAttribute('data-documentname');
        const description = button.getAttribute('data-description');
        const status = button.getAttribute('data-status');
        
        document.getElementById('edit_id').value = id;
        document.getElementById('edit_documentname').value = documentname;
        document.getElementById('edit_description').value = description || '';
        document.getElementById('edit_status').value = status;
        
        // Update form action
        const form = document.getElementById('editRequirementForm');
        if (form) {
            form.action = `/application-requirements/${id}`;
        }
        
        document.getElementById('editRequirementModal').classList.remove('hidden');
        document.getElementById('editRequirementModal').classList.add('flex');
    }
    
    function closeEditModal() {
        document.getElementById('editRequirementModal').classList.add('hidden');
        document.getElementById('editRequirementModal').classList.remove('flex');
    }

    // Close modals when clicking outside
    document.addEventListener('click', function(event) {
        const addModal = document.getElementById('addRequirementModal');
        const editModal = document.getElementById('editRequirementModal');
        
        if (addModal && !addModal.classList.contains('hidden')) {
            if (event.target === addModal) {
                closeAddModal();
            }
        }
        
        if (editModal && !editModal.classList.contains('hidden')) {
            if (event.target === editModal) {
                closeEditModal();
            }
        }
    });

    // Close modals with Escape key
    document.addEventListener('keydown', function(event) {
        if (event.key === 'Escape') {
            closeAddModal();
            closeEditModal();
        }
    });

    // Initialize tooltips
    document.addEventListener('DOMContentLoaded', function() {
        // Simple tooltip implementation
        const elementsWithTitle = document.querySelectorAll('[title]');
        elementsWithTitle.forEach(element => {
            element.addEventListener('mouseenter', function(e) {
                const tooltip = document.createElement('div');
                tooltip.className = 'fixed bg-gray-800 text-white text-xs rounded py-1 px-2 z-50';
                tooltip.textContent = this.getAttribute('title');
                tooltip.style.top = (e.pageY + 10) + 'px';
                tooltip.style.left = (e.pageX + 10) + 'px';
                tooltip.id = 'custom-tooltip';
                document.body.appendChild(tooltip);
            });
            
            element.addEventListener('mouseleave', function() {
                const tooltip = document.getElementById('custom-tooltip');
                if (tooltip) {
                    tooltip.remove();
                }
            });
        });
    });
</script>

<style>
    .bg-primary {
        background-color: #3b82f6;
    }
    .bg-secondary {
        background-color: #1e40af;
    }
    .focus\:ring-primary:focus {
        --tw-ring-color: #3b82f6;
    }
    .focus\:border-primary:focus {
        border-color: #3b82f6;
    }
    
    /* Modal animations */
    .fixed.inset-0 {
        transition: opacity 0.3s ease;
    }
    
    .fixed.inset-0.hidden {
        opacity: 0;
        pointer-events: none;
    }
    
    .fixed.inset-0.flex {
        opacity: 1;
        pointer-events: auto;
    }
    
    .relative.top-20.mx-auto {
        animation: modalSlideIn 0.3s ease;
    }
    
    @keyframes modalSlideIn {
        from {
            opacity: 0;
            transform: translateY(-20px);
        }
        to {
            opacity: 1;
            transform: translateY(0);
        }
    }
</style>
@endsection