@php
    // Define page variables
    $pageTitle = 'New Loan Application';
    $pageSubtitle = 'Create a new loan application for a customer';
    
    // Define theme colors
    $themeBlue = '#0077C5';
    $themeBlueDark = '#005A94';
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>

    <style>
        :root {
            --zedbank-blue: {{ $themeBlue }}; 
            --zedbank-blue-dark: {{ $themeBlueDark }}; 
        }

        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }

        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        .compact-input { padding: 10px 14px; font-size: 0.875rem; } 
        .compact-label { font-size: 0.875rem; } 
        .detail-label { font-size: 0.75rem; color: #6b7280; } 
        .detail-value { font-size: 0.875rem; font-weight: 600; } 
        
        .loading-spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #0077C5;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        .backdate-warning {
            display: none;
            animation: fadeIn 0.3s ease-in;
        }
        
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }

        /* Document Requirements Styles */
        .document-requirement-card {
            border-left: 4px solid transparent;
            transition: all 0.2s ease;
        }
        
        .document-requirement-card.mandatory {
            border-left-color: #dc3545;
            background-color: #fef2f2;
        }
        
        .document-requirement-card.optional {
            border-left-color: #6c757d;
            background-color: #f8f9fa;
        }
        
        .document-badge {
            display: inline-flex;
            align-items: center;
            padding: 0.25rem 0.75rem;
            border-radius: 9999px;
            font-size: 0.75rem;
            font-weight: 600;
        }
        
        .document-badge.mandatory {
            background-color: #fee2e2;
            color: #dc2626;
        }
        
        .document-badge.optional {
            background-color: #e5e7eb;
            color: #4b5563;
        }
        
        .document-upload-zone {
            border: 2px dashed #d1d5db;
            border-radius: 0.5rem;
            padding: 2rem;
            text-align: center;
            cursor: pointer;
            transition: all 0.2s ease;
        }
        
        .document-upload-zone:hover {
            border-color: #0077C5;
            background-color: #f0f9ff;
        }
        
        .document-upload-zone.drag-over {
            border-color: #0077C5;
            background-color: #e0f2fe;
        }
        
        .document-preview {
            display: flex;
            align-items: center;
            justify-content: space-between;
            background-color: #f0f9ff;
            border: 1px solid #bae6fd;
            border-radius: 0.5rem;
            padding: 1rem;
            margin-top: 1rem;
        }
        
        .document-progress {
            height: 0.25rem;
            background-color: #e5e7eb;
            border-radius: 0.125rem;
            overflow: hidden;
            margin-top: 0.5rem;
        }
        
        .document-progress-bar {
            height: 100%;
            background-color: #10b981;
            transition: width 0.3s ease;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64 min-h-screen">
        
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            <div class="max-w-7xl mx-auto">
                <div class="mb-6">
                    <a href="{{ route('admin.applications.index') }}" 
                    class="inline-flex items-center text-zedbank-blue hover:text-zedbank-blue-dark font-medium">
                        <i class="fas fa-arrow-left mr-2"></i>
                        Back to Applications
                    </a>
                </div>

                {{-- Display validation errors --}}
                @if ($errors->any())
                <div class="mb-6 bg-red-50 border-l-4 border-red-500 p-4 rounded">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-exclamation-triangle text-red-500"></i>
                        </div>
                        <div class="ml-3">
                            <h3 class="text-sm font-medium text-red-800">
                                There were {{ $errors->count() }} error(s) with your submission
                            </h3>
                            <div class="mt-2 text-sm text-red-700">
                                <ul class="list-disc pl-5 space-y-1">
                                    @foreach ($errors->all() as $error)
                                        <li>{{ $error }}</li>
                                    @endforeach
                                </ul>
                            </div>
                        </div>
                    </div>
                </div>
                @endif

                <form action="{{ route('admin.applications.store') }}" method="POST" id="application-form" enctype="multipart/form-data">
                    @csrf
                    {{-- Add hidden input for action tracking --}}
                    <input type="hidden" name="action" id="form_action" value="submit">
                    
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        <div class="lg:col-span-2 space-y-6">
                            
                            {{-- CARD 1: CUSTOMER SELECTION --}}
                            <div class="bg-white rounded-xl shadow-sm border mb-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-user-check mr-3 text-zedbank-blue"></i>
                                        Select Customer
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <select name="customer_id" id="loan_customer_select" 
                                            class="w-full compact-input border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                            required>
                                        <option value="">Select Customer</option>
                                        @foreach($customers as $customer)
                                            <option value="{{ $customer->id }}" {{ old('customer_id') == $customer->id ? 'selected' : '' }}>
                                                {{ $customer->first_name }} {{ $customer->surname }} ({{ $customer->customer_number }})
                                            </option>
                                        @endforeach
                                    </select>
                                </div>
                            </div>

                            {{-- CARD 2: CUSTOMER INFORMATION --}}
                            <div class="bg-white rounded-xl shadow-sm border">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-address-card mr-3 text-zedbank-blue"></i>
                                        Customer Information (Auto-filled)
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">Title</label>
                                             <input type="text" name="title" id="title" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('title') }}"></div>
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">First Name</label>
                                             <input type="text" name="first_name" id="first_name" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('first_name') }}"></div>
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">Surname</label>
                                             <input type="text" name="surname" id="surname" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('surname') }}"></div>
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">NRC No.</label>
                                             <input type="text" name="nrc_no" id="nrc_no" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('nrc_no') }}"></div>
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">Mobile No.</label>
                                             <input type="text" name="phone" id="phone" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('phone') }}"></div>
                                        <div><label class="block text-gray-700 compact-label font-medium mb-2">Email</label>
                                             <input type="email" name="email" id="email" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('email') }}"></div>
                                    </div>
                                    <div class="mt-4 pt-4 border-t border-gray-200">
                                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4">
                                            <div class="md:col-span-2"><label class="block text-gray-700 compact-label font-medium mb-2">Physical Address</label>
                                                 <input type="text" name="physical_address" id="physical_address" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('physical_address') }}"></div>
                                            <div><label class="block text-gray-700 compact-label font-medium mb-2">Town</label>
                                                 <input type="text" name="town" id="town" class="w-full compact-input border border-gray-300 rounded-lg" value="{{ old('town') }}"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 3: EMPLOYMENT DETAILS --}}
                            <div class="bg-white rounded-xl shadow-sm border">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-briefcase mr-3 text-zedbank-blue"></i>
                                        Employment Details
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        {{-- Column 1 --}}
                                        <div class="space-y-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Job Title</label>
                                                <input type="text" name="job_title" id="job_title" value="{{ old('job_title') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Employer Name</label>
                                                <input type="text" name="employer_name" id="employer_name" value="{{ old('employer_name') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Employment Type</label>
                                                <select name="employment_type" id="employment_type" class="w-full compact-input border border-gray-300 rounded-lg">
                                                    <option value="">Select Type</option>
                                                    <option value="permanent" {{ old('employment_type') == 'permanent' ? 'selected' : '' }}>Permanent</option>
                                                    <option value="contract" {{ old('employment_type') == 'contract' ? 'selected' : '' }}>Contract</option>
                                                    <option value="temporary" {{ old('employment_type') == 'temporary' ? 'selected' : '' }}>Temporary</option>
                                                    <option value="casual" {{ old('employment_type') == 'casual' ? 'selected' : '' }}>Casual</option>
                                                    <option value="self_employed" {{ old('employment_type') == 'self_employed' ? 'selected' : '' }}>Self Employed</option>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Gross Salary</label>
                                                <input type="number" name="gross_salary" id="gross_salary" value="{{ old('gross_salary') }}" step="0.01" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Employee Number</label>
                                                <input type="text" name="employee_number" id="employee_number" value="{{ old('employee_number') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>

                                        {{-- Column 2 --}}
                                        <div class="space-y-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Net Salary</label>
                                                <input type="number" name="net_salary" id="net_salary" value="{{ old('net_salary') }}" step="0.01" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Years of Employment</label>
                                                <input type="number" name="employment_years" id="employment_years" value="{{ old('employment_years') }}" step="0.1" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Contract Expiry Date</label>
                                                <input type="date" name="contract_expiry_date" id="contract_expiry_date" value="{{ old('contract_expiry_date') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Retirement Year</label>
                                                <input type="number" name="retirement_year" id="retirement_year" value="{{ old('retirement_year') }}" min="{{ date('Y') }}" max="2100" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>
                                    </div>

                                    {{-- Employer Address Section --}}
                                    <div class="mt-6 pt-4 border-t border-gray-200">
                                        <h3 class="font-bold text-gray-800 mb-3">Employer Address</h3>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Physical Address</label>
                                                <input type="text" name="employer_physical_address" id="employer_physical_address" value="{{ old('employer_physical_address') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Postal Address</label>
                                                <input type="text" name="employer_postal_address" id="employer_postal_address" value="{{ old('employer_postal_address') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Town/Province</label>
                                                <input type="text" name="employer_town_province" id="employer_town_province" value="{{ old('employer_town_province') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 4: NEXT OF KIN DETAILS --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-users mr-3 text-zedbank-blue"></i>
                                        Next of Kin Details
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        {{-- Personal Details --}}
                                        <div class="space-y-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Surname</label>
                                                <input type="text" name="kin_surname" id="kin_surname" value="{{ old('kin_surname') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">First Name</label>
                                                <input type="text" name="kin_first_name" id="kin_first_name" value="{{ old('kin_first_name') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Relationship</label>
                                                <select name="kin_relationship" id="kin_relationship" class="w-full compact-input border border-gray-300 rounded-lg">
                                                    <option value="">Select Relationship</option>
                                                    <option value="spouse" {{ old('kin_relationship') == 'spouse' ? 'selected' : '' }}>Spouse</option>
                                                    <option value="parent" {{ old('kin_relationship') == 'parent' ? 'selected' : '' }}>Parent</option>
                                                    <option value="child" {{ old('kin_relationship') == 'child' ? 'selected' : '' }}>Child</option>
                                                    <option value="sibling" {{ old('kin_relationship') == 'sibling' ? 'selected' : '' }}>Sibling</option>
                                                    <option value="relative" {{ old('kin_relationship') == 'relative' ? 'selected' : '' }}>Relative</option>
                                                    <option value="friend" {{ old('kin_relationship') == 'friend' ? 'selected' : '' }}>Friend</option>
                                                    <option value="other" {{ old('kin_relationship') == 'other' ? 'selected' : '' }}>Other</option>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Email</label>
                                                <input type="email" name="kin_email" id="kin_email" value="{{ old('kin_email') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>

                                        {{-- Contact Details --}}
                                        <div class="space-y-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Telephone</label>
                                                <input type="text" name="kin_telephone" id="kin_telephone" value="{{ old('kin_telephone') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Country</label>
                                                <select name="kin_country" id="kin_country" class="w-full compact-input border border-gray-300 rounded-lg">
                                                    <option value="Zambia" {{ old('kin_country', 'Zambia') == 'Zambia' ? 'selected' : '' }}>Zambia</option>
                                                    <option value="Other" {{ old('kin_country') == 'Other' ? 'selected' : '' }}>Other</option>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Province</label>
                                                <input type="text" name="kin_province" id="kin_province" value="{{ old('kin_province') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>
                                    </div>

                                    {{-- Address Section --}}
                                    <div class="mt-6 pt-4 border-t border-gray-200">
                                        <h3 class="font-bold text-gray-800 mb-3">Next of Kin Address</h3>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Physical Address</label>
                                                <input type="text" name="kin_physical_address" id="kin_physical_address" value="{{ old('kin_physical_address') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Postal Address</label>
                                                <input type="text" name="kin_postal_address" id="kin_postal_address" value="{{ old('kin_postal_address') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2">Town</label>
                                                <input type="text" name="kin_town" id="kin_town" value="{{ old('kin_town') }}" class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 5: LOAN DETAILS --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-file-invoice-dollar mr-3 text-zedbank-blue"></i>
                                        Loan Details
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div class="space-y-4">
                                            {{-- 1. Loan Product --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="product_id">
                                                    <span class="text-red-500">*</span> Loan Product
                                                </label>
                                                <select name="product_id" id="product_id" 
                                                        class="w-full compact-input border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                                        onchange="updateInterestRateFromProduct(this); calculateInstallment(); loadDocumentRequirements(this.value);"
                                                        required>
                                                    <option value="">Select Loan Product</option>
                                                    @foreach($loanProducts as $product)
                                                        <option value="{{ $product->id }}" 
                                                                data-daily-rate="{{ $product->dailyrate ?? 0 }}"
                                                                data-weekly-rate="{{ $product->weeklyrate ?? 0 }}"
                                                                data-monthly-rate="{{ $product->monthlyrate ?? 0 }}"
                                                                data-yearly-rate="{{ $product->yearlyrate ?? 0 }}"
                                                                data-admin-fee="{{ $product->adminfee ?? 0 }}"
                                                                data-insurance-fee="{{ $product->insurancefee ?? 0 }}"
                                                                data-processing-fee="{{ $product->processing_fee ?? 0 }}"
                                                                data-processing-fee-basis="{{ $product->processing_fee_basis ?? 'initial_amount' }}"
                                                                data-penalty-rate="{{ $product->penalty_rate ?? 0 }}"
                                                                data-penalty-basis="{{ $product->penalty_basis ?? 'percentage_of_due' }}"
                                                                data-interest-method="{{ $product->interest_method ?? 'reducing_balance' }}"
                                                                {{ old('product_id') == $product->id ? 'selected' : '' }}>
                                                            {{ $product->product }} ({{ $product->yearlyrate }}%)
                                                        </option>
                                                    @endforeach
                                                </select>
                                            </div>

                                            {{-- 2. Payment Frequency --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="payment_frequency">
                                                    <span class="text-red-500">*</span> Payment Frequency
                                                </label>
                                                <select name="payment_frequency" id="payment_frequency" 
                                                        class="w-full compact-input border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                                        onchange="updateTenureLabel(); updateInterestRateFromProduct(document.getElementById('product_id')); calculateInstallment();"
                                                        required>
                                                    <option value="">Select Frequency</option>
                                                    <option value="one_off" {{ old('payment_frequency') == 'one_off' ? 'selected' : '' }}>One-off Payment</option>
                                                    <option value="daily" {{ old('payment_frequency') == 'daily' ? 'selected' : '' }}>Daily</option>
                                                    <option value="weekly" {{ old('payment_frequency') == 'weekly' ? 'selected' : '' }}>Weekly</option>
                                                    <option value="monthly" {{ old('payment_frequency') == 'monthly' ? 'selected' : '' }}>Monthly</option>
                                                    <option value="quarterly" {{ old('payment_frequency') == 'quarterly' ? 'selected' : '' }}>Quarterly</option>
                                                </select>
                                            </div>

                                            {{-- 3. Interest Method --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="interest_method">
                                                    <span class="text-red-500">*</span> Interest Method
                                                </label>
                                                <select name="interest_method" id="interest_method" 
                                                        class="w-full compact-input border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors"
                                                        onchange="calculateInstallment()"
                                                        required>
                                                    <option value="">Select Method</option>
                                                    <option value="simple_interest" {{ old('interest_method') == 'simple_interest' ? 'selected' : '' }}>Simple Interest</option>
                                                    <option value="reducing_balance" {{ old('interest_method') == 'reducing_balance' ? 'selected' : '' }}>Reducing Balance</option>
                                                </select>
                                            </div>

                                            {{-- 4. Interest Rate --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="interest_rate">
                                                    <span class="text-red-500">*</span> Interest Rate (%)
                                                </label>
                                                <div class="relative">
                                                    <input type="number" name="interest_rate" id="interest_rate" value="{{ old('interest_rate') }}" step="0.01" class="w-full px-4 compact-input border border-gray-300 rounded-lg" oninput="calculateInstallment()" required>
                                                    <span class="absolute inset-y-0 right-0 pr-3 flex items-center text-gray-500">%</span>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div class="space-y-4">
                                            {{-- 5. Currency --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="currency_id">
                                                    <span class="text-red-500">*</span> Currency
                                                </label>
                                                <select name="currency_id" id="currency_id" class="w-full compact-input border border-gray-300 rounded-lg" onchange="updateCurrencySymbols(this); calculateInstallment();" required>
                                                    <option value="">Select Currency</option>
                                                    @foreach($currencies as $currency)
                                                        <option value="{{ $currency->id }}" data-symbol="{{ $currency->symbol }}" {{ old('currency_id') == $currency->id ? 'selected' : '' }}>{{ $currency->code }} - {{ $currency->name }}</option>
                                                    @endforeach
                                                </select>
                                            </div>

                                            {{-- 6. Loan Tenure --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="loan_tenure">
                                                    <span class="text-red-500">*</span> <span id="tenure-label-text">Loan Tenure (Months)</span>
                                                </label>
                                                <input type="number" name="loan_tenure" id="loan_tenure" value="{{ old('loan_tenure') }}" class="w-full compact-input border border-gray-300 rounded-lg" oninput="calculateInstallment()" required>
                                            </div>
                                            
                                            {{-- 7. Loan Amount --}}
                                            <div>
                                                <label class="block text-gray-700 compact-label font-medium mb-2" for="loan_amount">
                                                    <span class="text-red-500">*</span> Loan Amount
                                                </label>
                                                <div class="relative">
                                                    <span class="absolute inset-y-0 left-0 pl-3 flex items-center text-gray-500 currency-symbol-text">ZMW</span>
                                                    <input type="number" name="loan_amount" id="loan_amount" value="{{ old('loan_amount') }}" step="0.01" class="w-full pl-12 compact-input border border-gray-300 rounded-lg" oninput="calculateInstallment()" required>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 6: FEES AND PENALTY TERMS --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-clipboard-list mr-3 text-zedbank-blue"></i>
                                        Fees & Penalty Terms
                                    </h2>
                                </div>
                                <div class="p-4 grid grid-cols-1 md:grid-cols-3 gap-4">
                                    <div>
                                        <label class="block text-gray-700 compact-label font-medium mb-2">Processing Fee</label>
                                        <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg">
                                            <span id="fee_processing_rate">0.00%</span> on <span id="fee_processing_basis">Initial Amount</span>
                                        </div>
                                        <input type="hidden" name="processing_fee" id="hidden_processing_fee" value="{{ old('processing_fee', 0) }}">
                                        <input type="hidden" name="processing_fee_basis" id="hidden_processing_fee_basis" value="{{ old('processing_fee_basis', 'initial_amount') }}">
                                    </div>
                                    <div>
                                        <label class="block text-gray-700 compact-label font-medium mb-2">Admin Fee</label>
                                        <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg" id="fee_admin_rate">0.00</div>
                                        <input type="hidden" name="adminfee" id="hidden_admin_fee" value="{{ old('adminfee', 0) }}">
                                    </div>
                                    <div>
                                        <label class="block text-gray-700 compact-label font-medium mb-2">Insurance Fee</label>
                                        <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg" id="fee_insurance_rate">0.00</div>
                                        <input type="hidden" name="insurancefee" id="hidden_insurance_fee" value="{{ old('insurancefee', 0) }}">
                                    </div>
                                    <div class="md:col-span-3 border-t pt-4 mt-2">
                                        <div class="grid grid-cols-2 gap-4">
                                            <div><label class="compact-label font-medium mb-2">Penalty Rate</label>
                                                 <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg" id="penalty_rate_display">0.00%</div>
                                                 <input type="hidden" name="penalty_rate" id="hidden_penalty_rate" value="{{ old('penalty_rate', 0) }}"></div>
                                            <div><label class="compact-label font-medium mb-2">Penalty Basis</label>
                                                 <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg" id="penalty_basis_display">N/A</div>
                                                 <input type="hidden" name="penalty_basis" id="hidden_penalty_basis" value="{{ old('penalty_basis', 'percentage_of_due') }}"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 7: DOCUMENT REQUIREMENTS --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-file-alt mr-3 text-zedbank-blue"></i>
                                        Document Requirements
                                        <span class="ml-2 text-sm font-semibold px-2 py-1 bg-blue-100 text-blue-800 rounded-full" id="document-count-badge">
                                            0/0
                                        </span>
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="mb-6">
                                        <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-4">
                                            <div class="bg-blue-50 border border-blue-100 rounded-lg p-3 text-center">
                                                <div class="text-lg font-bold text-blue-700" id="mandatory-count">0/0</div>
                                                <div class="text-xs font-medium text-blue-600">Mandatory</div>
                                            </div>
                                            <div class="bg-gray-50 border border-gray-100 rounded-lg p-3 text-center">
                                                <div class="text-lg font-bold text-gray-700" id="optional-count">0</div>
                                                <div class="text-xs font-medium text-gray-600">Optional</div>
                                            </div>
                                            <div class="bg-green-50 border border-green-100 rounded-lg p-3 text-center">
                                                <div class="text-lg font-bold text-green-700" id="uploaded-count">0</div>
                                                <div class="text-xs font-medium text-green-600">Uploaded</div>
                                            </div>
                                        </div>
                                        
                                        <div class="mb-4">
                                            <div class="flex justify-between text-sm mb-1">
                                                <span class="font-medium">Mandatory Completion</span>
                                                <span class="font-bold" id="completion-percentage">0%</span>
                                            </div>
                                            <div class="w-full bg-gray-200 rounded-full h-2.5">
                                                <div class="bg-green-600 h-2.5 rounded-full" id="completion-bar" style="width: 0%"></div>
                                            </div>
                                            <p class="text-xs text-gray-500 mt-1" id="completion-text">
                                                Select a loan product to see document requirements
                                            </p>
                                        </div>
                                    </div>

                                    {{-- Document Requirements Container --}}
                                    <div id="document-requirements-container">
                                        <div class="text-center py-8">
                                            <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-blue-100">
                                                <i class="fas fa-info-circle text-blue-600 text-2xl"></i>
                                            </div>
                                            <h3 class="mt-4 text-lg font-medium text-gray-900">No Product Selected</h3>
                                            <p class="mt-2 text-sm text-gray-500">
                                                Please select a loan product to view document requirements.
                                            </p>
                                        </div>
                                    </div>

                                    {{-- Document Upload Progress --}}
                                    <div id="document-upload-progress" class="hidden">
                                        <div class="border-t border-gray-200 pt-4 mt-6">
                                            <h4 class="font-medium text-gray-900 mb-3">Upload Progress</h4>
                                            <div id="upload-progress-list" class="space-y-3"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 8: APPLICATION DATE (NEW CARD FOR BACKDATING) --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-calendar-alt mr-3 text-zedbank-blue"></i>
                                        Application Date
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="space-y-4">
                                        <div>
                                            <label class="block text-gray-700 compact-label font-medium mb-2" for="application_date">
                                                Application Date
                                            </label>
                                            <div class="flex flex-col space-y-2">
                                                <input type="date" name="application_date" id="application_date" 
                                                       value="{{ date('Y-m-d') }}"
                                                       max="{{ date('Y-m-d') }}"
                                                       class="w-full compact-input border border-gray-300 rounded-lg"
                                                       onchange="checkBackdateWarning(this)">
                                                <p class="text-xs text-gray-500 mt-1">
                                                    Defaults to today's date. Change only if backdating application.
                                                </p>
                                            </div>
                                            
                                            {{-- Backdate Warning --}}
                                            <div id="backdate-warning" class="backdate-warning mt-3 p-3 bg-yellow-50 border border-yellow-200 rounded-lg">
                                                <div class="flex items-start">
                                                    <i class="fas fa-exclamation-triangle text-yellow-600 mt-0.5 mr-2"></i>
                                                    <div>
                                                        <p class="text-sm font-medium text-yellow-800" id="backdate-warning-title"></p>
                                                        <p class="text-xs text-yellow-700 mt-1" id="backdate-warning-message"></p>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 9: ADDITIONAL INFORMATION --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-info-circle mr-3 text-zedbank-blue"></i>
                                        Additional Information
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div>
                                        <label class="block text-gray-700 compact-label font-medium mb-2" for="source">
                                            <span class="text-red-500">*</span> Application Source
                                        </label>
                                        <select name="source" id="source" 
                                                class="w-full compact-input border border-gray-300 rounded-lg focus:ring-2 focus:ring-blue-500 focus:border-blue-500 transition-colors @error('source') border-red-500 @enderror"
                                                required>
                                            <option value="online" {{ old('source') == 'online' ? 'selected' : '' }}>Online</option>
                                            <option value="branch" {{ old('source', 'branch') == 'branch' ? 'selected' : '' }}>Branch</option>
                                            <option value="agent" {{ old('source') == 'agent' ? 'selected' : '' }}>Agent</option>
                                            <option value="mobile" {{ old('source') == 'mobile' ? 'selected' : '' }}>Mobile</option>
                                        </select>
                                        @error('source')
                                            <p class="text-red-500 text-xs mt-2 flex items-center">
                                                <i class="fas fa-exclamation-circle mr-1"></i>
                                                {{ $message }}
                                            </p>
                                        @enderror
                                    </div>
                                </div>
                            </div>

                            {{-- CARD 10: DISBURSEMENT INFORMATION --}}
                            <div class="bg-white rounded-xl shadow-sm border mt-6">
                                <div class="p-4 border-b border-gray-200">
                                    <h2 class="text-lg font-bold text-gray-900 flex items-center">
                                        <i class="fas fa-exchange-alt mr-3 text-zedbank-blue"></i>
                                        Disbursement Information
                                    </h2>
                                </div>
                                <div class="p-4">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label class="block text-gray-700 compact-label font-medium mb-2">Calculated Installment</label>
                                            <div class="compact-input bg-gray-50 border border-gray-200 rounded-lg font-bold flex items-center justify-between">
                                                <div>
                                                    <span class="currency-symbol-text">ZMW</span> <span id="disbursement-installment">0.00</span>
                                                </div>
                                                <div id="installment-loading" class="hidden">
                                                    <div class="loading-spinner"></div>
                                                </div>
                                            </div>
                                            <input type="hidden" name="installment_amount" id="installment_amount_hidden" value="{{ old('installment_amount', 0) }}">
                                        </div>
                                        <div>
                                            <label class="block text-gray-700 compact-label font-medium mb-2" for="payment_method">
                                                <span class="text-red-500">*</span> Payment Method
                                            </label>
                                            <select name="suggestedpaymentmethod" id="payment_method" class="w-full compact-input border border-gray-300 rounded-lg" onchange="handlePaymentMethodChange()" required>
                                                <option value="">Select Method</option>
                                                <option value="Cash" {{ old('suggestedpaymentmethod') == 'Cash' ? 'selected' : '' }}>Cash</option>
                                                <option value="Bank" {{ old('suggestedpaymentmethod') == 'Bank' ? 'selected' : '' }}>Bank</option>
                                                <option value="Mobile" {{ old('suggestedpaymentmethod') == 'Mobile' ? 'selected' : '' }}>Mobile Money</option>
                                            </select>
                                        </div>
                                    </div>

                                    {{-- DYNAMIC BANK FIELDS --}}
                                    <div id="disbursement_bank_fields" class="mt-4 pt-4 border-t border-gray-200 hidden space-y-4">
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div class="md:col-span-2"><label class="compact-label font-medium mb-1">Bank Name</label>
                                                 <input type="text" name="bank_name" value="{{ old('bank_name') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                            <div><label class="compact-label font-medium mb-1">Account Holder</label>
                                                 <input type="text" name="account_holder" value="{{ old('account_holder') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                            <div><label class="compact-label font-medium mb-1">Account Number</label>
                                                 <input type="text" name="account_number" value="{{ old('account_number') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                            <div><label class="compact-label font-medium mb-1">Branch</label>
                                                 <input type="text" name="bank_branch" value="{{ old('bank_branch') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                            <div><label class="compact-label font-medium mb-1">SWIFT Code</label>
                                                 <input type="text" name="swift_code" value="{{ old('swift_code') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                        </div>
                                    </div>

                                    {{-- DYNAMIC MOBILE FIELDS --}}
                                    <div id="disbursement_mobile_fields" class="mt-4 pt-4 border-t border-gray-200 hidden">
                                        <div class="grid grid-cols-2 gap-4">
                                            <div><label class="compact-label font-medium mb-1">Payment Channel</label>
                                                 <select name="mobile_channel" class="w-full compact-input border border-gray-300 rounded-lg">
                                                     <option value="MTN Mobile" {{ old('mobile_channel') == 'MTN Mobile' ? 'selected' : '' }}>MTN Mobile</option>
                                                     <option value="Airtel Mobile" {{ old('mobile_channel') == 'Airtel Mobile' ? 'selected' : '' }}>Airtel Mobile</option>
                                                     <option value="Zamtel Mobile" {{ old('mobile_channel') == 'Zamtel Mobile' ? 'selected' : '' }}>Zamtel Mobile</option>
                                                 </select></div>
                                            <div><label class="compact-label font-medium mb-1">Mobile Phone Number</label>
                                                 <input type="text" name="mobile_number" value="{{ old('mobile_number') }}" class="w-full compact-input border border-gray-300 rounded-lg"></div>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>

                        {{-- SIDEBAR: ESTIMATES & ACTIONS --}}
                        <div class="space-y-6">
                            <div class="bg-white rounded-xl shadow-sm border p-4">
                                <h3 class="text-lg font-bold text-gray-900 mb-4">Actions</h3>
                                
                                {{-- Submit Button --}}
                                <button type="submit" name="action" value="submit" class="w-full px-4 py-3 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark font-medium flex items-center justify-center mb-2">
                                    <i class="fas fa-paper-plane mr-2"></i> Submit Application
                                </button>
                                
                                {{-- Save as Draft Button --}}
                                <button type="submit" name="action" value="draft" class="w-full px-4 py-3 bg-gray-500 text-white rounded-lg hover:bg-gray-600 font-medium flex items-center justify-center mb-2">
                                    <i class="fas fa-save mr-2"></i> Save as Draft
                                </button>
                                
                                {{-- Cancel Button --}}
                                <a href="{{ route('admin.applications.index') }}" class="block text-center w-full px-4 py-3 border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50">
                                    Cancel
                                </a>
                                
                                {{-- Help Text --}}
                                <div class="mt-3 text-xs text-gray-500">
                                    <p><strong>Submit:</strong> Application will be processed immediately.</p>
                                    <p><strong>Save as Draft:</strong> Application will be saved for later review and editing.</p>
                                </div>
                            </div>

                            <div class="bg-white rounded-xl shadow-sm border p-4">
                                <h3 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                                    <i class="fas fa-calculator mr-3 text-zedbank-blue"></i> Loan Estimates
                                </h3>
                                <div class="space-y-3">
                                    <div class="flex justify-between">
                                        <span class="text-gray-600">Principal</span>
                                        <span class="font-semibold"><span class="currency-symbol-text">ZMW</span> <span id="preview-amount">0.00</span></span>
                                    </div>
                                    <div class="flex justify-between border-t border-gray-100 pt-3">
                                        <span class="text-gray-600 font-medium">Est. Installment</span>
                                        <span class="font-bold text-green-600"><span class="currency-symbol-text">ZMW</span> <span id="calc-installment">0.00</span></span>
                                    </div>
                                </div>
                            </div>

                            {{-- Fee Breakdown Section --}}
                            <div class="bg-white rounded-xl shadow-sm border p-4">
                                <h3 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                                    <i class="fas fa-receipt mr-3 text-zedbank-blue"></i> Fee Breakdown
                                </h3>
                                <div id="fee-breakdown-container" class="text-sm">
                                    <div class="text-gray-500 italic text-center py-4">
                                        Enter loan details to see fee breakdown
                                    </div>
                                </div>
                            </div>

                            {{-- Document Status Summary --}}
                            <div class="bg-white rounded-xl shadow-sm border p-4">
                                <h3 class="text-lg font-bold text-gray-900 mb-4 flex items-center">
                                    <i class="fas fa-clipboard-check mr-3 text-zedbank-blue"></i> Document Status
                                </h3>
                                <div id="document-status-summary">
                                    <div class="text-center py-4">
                                        <i class="fas fa-info-circle text-gray-400 text-2xl mb-2"></i>
                                        <p class="text-sm text-gray-500">Select a loan product to view requirements</p>
                                    </div>
                                </div>
                            </div>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    {{-- Document Upload Modal --}}
    <div id="documentUploadModal" class="fixed inset-0 bg-gray-500 bg-opacity-75 flex items-center justify-center z-50 hidden">
        <div class="bg-white rounded-lg shadow-xl max-w-md w-full mx-4">
            <div class="p-6">
                <div class="flex justify-between items-center mb-4">
                    <h3 class="text-lg font-bold text-gray-900" id="modalDocumentName">Upload Document</h3>
                    <button type="button" onclick="closeDocumentModal()" class="text-gray-400 hover:text-gray-500">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
                
                <input type="hidden" id="modalRequirementId">
                <input type="hidden" id="modalRequirementName">
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">Description</label>
                    <p class="text-sm text-gray-600" id="modalDocumentDescription">No description available</p>
                </div>
                
                <div class="mb-4">
                    <label class="block text-sm font-medium text-gray-700 mb-2">File *</label>
                    <div id="dropZone" class="document-upload-zone">
                        <div class="space-y-2">
                            <i class="fas fa-cloud-upload-alt text-3xl text-gray-400"></i>
                            <div>
                                <p class="text-sm text-gray-600">
                                    <span class="font-medium text-blue-600 hover:text-blue-500 cursor-pointer">Click to browse</span> or drag and drop
                                </p>
                                <p class="text-xs text-gray-500 mt-1">
                                    PDF, JPG, PNG, DOC, DOCX, XLS, XLSX up to 10MB
                                </p>
                            </div>
                        </div>
                        <input type="file" id="documentFileInput" class="hidden" accept=".pdf,.jpg,.jpeg,.png,.doc,.docx,.xls,.xlsx">
                    </div>
                    
                    <div id="filePreview" class="hidden">
                        <div class="document-preview">
                            <div class="flex items-center">
                                <i class="fas fa-file text-blue-500 mr-3 text-xl"></i>
                                <div>
                                    <p id="fileName" class="font-medium text-gray-900"></p>
                                    <p id="fileSize" class="text-sm text-gray-500"></p>
                                </div>
                            </div>
                            <button type="button" onclick="clearFile()" class="text-red-500 hover:text-red-700">
                                <i class="fas fa-times"></i>
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="mb-4">
                    <label for="documentNotes" class="block text-sm font-medium text-gray-700 mb-2">Notes (Optional)</label>
                    <textarea id="documentNotes" rows="2" class="w-full px-3 py-2 border border-gray-300 rounded-md focus:outline-none focus:ring-2 focus:ring-blue-500 focus:border-transparent" placeholder="Add any notes about this document..."></textarea>
                </div>
                
                <div id="uploadProgress" class="hidden">
                    <div class="flex items-center space-x-3 mb-2">
                        <div class="flex-1">
                            <div class="document-progress">
                                <div id="progressBar" class="document-progress-bar" style="width: 0%"></div>
                            </div>
                            <p id="progressText" class="text-xs text-gray-600 mt-1">Uploading...</p>
                        </div>
                    </div>
                </div>
                
                <div class="flex justify-end space-x-3">
                    <button type="button" onclick="closeDocumentModal()" class="px-4 py-2 border border-gray-300 rounded-md text-gray-700 hover:bg-gray-50">
                        Cancel
                    </button>
                    <button type="button" id="uploadButton" onclick="uploadDocument()" class="px-4 py-2 bg-zedbank-blue text-white rounded-md hover:bg-zedbank-blue-dark">
                        <i class="fas fa-upload mr-2"></i> Upload
                    </button>
                </div>
            </div>
        </div>
    </div>

    <script>
        const productSelect = document.getElementById('product_id');
        const rateInput = document.getElementById('interest_rate');
        const frequencySelect = document.getElementById('payment_frequency');
        const tenureInput = document.getElementById('loan_tenure');
        const amountInput = document.getElementById('loan_amount');
        const tenureLabelText = document.getElementById('tenure-label-text');
        const interestMethodSelect = document.getElementById('interest_method');
        const applicationDateInput = document.getElementById('application_date');

        // Document tracking variables
        let uploadedDocuments = {};
        let currentRequirementId = null;
        let documentRequirements = {};

        // --- GLOBAL HELPER FUNCTIONS ---

        function updateCurrencySymbols(selectElement) {
            const selectedOption = selectElement.options[selectElement.selectedIndex];
            const symbol = selectedOption ? selectedOption.getAttribute('data-symbol') || 'ZMW' : 'ZMW'; 
            document.querySelectorAll('.currency-symbol-text').forEach(el => el.textContent = symbol);
        }

        function updateTenureLabel() {
            if (!frequencySelect || !tenureLabelText) return;
            const frequency = frequencySelect.value;
            if (frequency === 'one_off' || frequency === 'daily') {
                tenureLabelText.textContent = 'Loan Tenure (Days)';
                tenureInput.placeholder = "e.g., 30";
            } else {
                tenureLabelText.textContent = 'Loan Tenure (Months)';
                tenureInput.placeholder = "e.g., 12";
            }
        }

        function updateInterestRateFromProduct(selectElement) {
            if (!selectElement || !frequencySelect) return;
            const option = selectElement.options[selectElement.selectedIndex];
            if (!option) return;

            const frequency = frequencySelect.value;
            let rateKey = 'data-monthly-rate'; // Default

            if (frequency === 'daily') rateKey = 'data-daily-rate';
            else if (frequency === 'weekly') rateKey = 'data-weekly-rate';
            else if (frequency === 'quarterly') rateKey = 'data-yearly-rate';

            const rate = option.getAttribute(rateKey) || '0.00';
            if (rateInput) rateInput.value = parseFloat(rate).toFixed(2);

            // Set Interest Method from product
            const interestMethod = option.getAttribute('data-interest-method') || 'reducing_balance';
            if (interestMethodSelect) {
                interestMethodSelect.value = interestMethod;
            }

            // Populate Fees & Penalties
            const pFee = option.getAttribute('data-processing-fee') || '0.00';
            const pBasis = option.getAttribute('data-processing-fee-basis') || 'initial_amount';
            const adminFee = option.getAttribute('data-admin-fee') || '0.00';
            const insuranceFee = option.getAttribute('data-insurance-fee') || '0.00';
            const penaltyRate = option.getAttribute('data-penalty-rate') || '0.00';
            const penaltyBasis = option.getAttribute('data-penalty-basis') || 'percentage_of_due';

            document.getElementById('fee_processing_rate').textContent = pFee + '%';
            document.getElementById('fee_processing_basis').textContent = pBasis.replace('_', ' ');
            document.getElementById('fee_admin_rate').textContent = adminFee;
            document.getElementById('fee_insurance_rate').textContent = insuranceFee;
            document.getElementById('penalty_rate_display').textContent = penaltyRate + '%';
            document.getElementById('penalty_basis_display').textContent = penaltyBasis.replace(/_/g, ' ');

            document.getElementById('hidden_processing_fee').value = pFee;
            document.getElementById('hidden_processing_fee_basis').value = pBasis;
            document.getElementById('hidden_admin_fee').value = adminFee;
            document.getElementById('hidden_insurance_fee').value = insuranceFee;
            document.getElementById('hidden_penalty_rate').value = penaltyRate;
            document.getElementById('hidden_penalty_basis').value = penaltyBasis;
        }

        function calculateInstallment() {
            const amount = parseFloat(amountInput.value) || 0;
            const rate = parseFloat(rateInput.value) || 0;
            const tenure = parseInt(tenureInput.value) || 0;
            const frequency = frequencySelect.value;
            const interestMethod = interestMethodSelect.value;
            
            // Get fee values from hidden inputs
            const processingFee = parseFloat(document.getElementById('hidden_processing_fee').value) || 0;
            const processingFeeBasis = document.getElementById('hidden_processing_fee_basis').value || 'initial_amount';
            const adminFee = parseFloat(document.getElementById('hidden_admin_fee').value) || 0;
            const insuranceFee = parseFloat(document.getElementById('hidden_insurance_fee').value) || 0;

            // Show loading indicator
            document.getElementById('installment-loading').classList.remove('hidden');
            const installmentElements = [
                document.getElementById('calc-installment'),
                document.getElementById('disbursement-installment'),
                document.getElementById('preview-amount')
            ];
            
            installmentElements.forEach(el => {
                if (el) el.textContent = 'Calculating...';
            });

            // Only calculate if we have valid inputs
            if (amount > 0 && rate > 0 && tenure > 0 && frequency && interestMethod) {
                $.ajax({
                    url: '{{ route("admin.applications.calculate_installment") }}',
                    type: 'POST',
                    data: {
                        _token: '{{ csrf_token() }}',
                        loan_amount: amount,
                        interest_rate: rate,
                        loan_tenure: tenure,
                        payment_frequency: frequency,
                        interest_method: interestMethod,
                        processing_fee: processingFee,
                        processing_fee_basis: processingFeeBasis,
                        adminfee: adminFee,
                        insurancefee: insuranceFee
                    },
                    success: function(response) {
                        document.getElementById('installment-loading').classList.add('hidden');
                        if (response.success) {
                            const installment = response.installment;
                            const rawInstallment = response.installment_raw;
                            
                            document.getElementById('preview-amount').textContent = amount.toFixed(2);
                            document.getElementById('calc-installment').textContent = installment;
                            document.getElementById('disbursement-installment').textContent = installment;
                            document.getElementById('installment_amount_hidden').value = rawInstallment;
                            
                            // Update fee breakdown
                            updateFeeBreakdown(response.breakdown);
                        } else {
                            showCalculationError('Calculation failed: ' . response.message);
                        }
                    },
                    error: function(xhr) {
                        document.getElementById('installment-loading').classList.add('hidden');
                        showCalculationError('Server error during calculation');
                        console.error('Calculation error:', xhr.responseText);
                    }
                });
            } else {
                document.getElementById('installment-loading').classList.add('hidden');
                // Reset to default values
                const defaultValue = '0.00';
                installmentElements.forEach(el => {
                    if (el) el.textContent = defaultValue;
                });
                document.getElementById('installment_amount_hidden').value = 0;
                document.getElementById('fee-breakdown-container').innerHTML = `
                    <div class="text-gray-500 italic text-center py-4">
                        Enter loan details to see fee breakdown
                    </div>
                `;
            }
        }

        function showCalculationError(message) {
            const errorMsg = message || 'Error calculating installment';
            const elements = [
                document.getElementById('calc-installment'),
                document.getElementById('disbursement-installment')
            ];
            
            elements.forEach(el => {
                if (el) {
                    el.textContent = 'Error';
                    el.classList.add('text-red-500');
                }
            });
            
            // Show error alert
            const alertDiv = document.createElement('div');
            alertDiv.className = 'fixed top-4 right-4 bg-red-100 border border-red-400 text-red-700 px-4 py-3 rounded z-50';
            alertDiv.innerHTML = `<strong>Calculation Error:</strong> ${errorMsg}`;
            document.body.appendChild(alertDiv);
            
            setTimeout(() => {
                document.body.removeChild(alertDiv);
                elements.forEach(el => {
                    if (el) el.classList.remove('text-red-500');
                });
            }, 5000);
        }

        function updateFeeBreakdown(breakdown) {
            const breakdownContainer = document.getElementById('fee-breakdown-container');
            if (breakdownContainer && breakdown) {
                let basisText = breakdown.processing_fee_basis === 'initial_amount' 
                    ? 'on initial amount' 
                    : 'on outstanding balance';
                
                breakdownContainer.innerHTML = `
                    <div class="text-xs space-y-1.5 mt-2 p-3 bg-gray-50 rounded border">
                        <div class="flex justify-between">
                            <span class="text-gray-600">Principal Amount:</span>
                            <span class="font-medium">${breakdown.principal.toFixed(2)}</span>
                        </div>
                        
                        ${breakdown.processing_fee_amount > 0 ? `
                        <div class="flex justify-between">
                            <span class="text-gray-600">Processing Fee (${breakdown.processing_fee_rate}% ${basisText}):</span>
                            <span class="font-medium">${breakdown.processing_fee_amount.toFixed(2)}</span>
                        </div>
                        ` : ''}
                        
                        ${breakdown.admin_fee > 0 ? `
                        <div class="flex justify-between">
                            <span class="text-gray-600">Admin Fee:</span>
                            <span class="font-medium">${breakdown.admin_fee.toFixed(2)}</span>
                        </div>
                        ` : ''}
                        
                        ${breakdown.insurance_fee > 0 ? `
                        <div class="flex justify-between">
                            <span class="text-gray-600">Insurance Fee:</span>
                            <span class="font-medium">${breakdown.insurance_fee.toFixed(2)}</span>
                        </div>
                        ` : ''}
                        
                        <div class="flex justify-between border-t pt-2 mt-1">
                            <span class="text-gray-700 font-semibold">Total Fees:</span>
                            <span class="font-semibold">${breakdown.total_fees.toFixed(2)}</span>
                        </div>
                        
                        <div class="flex justify-between">
                            <span class="text-gray-700 font-semibold">Total Amount with Fees:</span>
                            <span class="font-bold text-zedbank-blue">${breakdown.amount_with_fees.toFixed(2)}</span>
                        </div>
                        
                        <div class="flex justify-between border-t pt-2 mt-1">
                            <span class="text-gray-800 font-bold">Monthly Installment:</span>
                            <span class="font-bold text-green-600">${breakdown.installment_amount.toFixed(2)}</span>
                        </div>
                        
                        ${breakdown.processing_fee_basis === 'outstanding_balance' ? `
                        <div class="mt-2 p-2 bg-blue-50 border border-blue-100 rounded text-blue-700 text-xs">
                            <i class="fas fa-info-circle mr-1"></i>
                            Processing fee (${breakdown.processing_fee_rate}%) is included in the interest calculation.
                        </div>
                        ` : ''}
                    </div>
                `;
            }
        }

        function handlePaymentMethodChange() {
            const method = document.getElementById('payment_method').value;
            document.getElementById('disbursement_bank_fields').classList.add('hidden');
            document.getElementById('disbursement_mobile_fields').classList.add('hidden');
            if (method === 'Bank') document.getElementById('disbursement_bank_fields').classList.remove('hidden');
            else if (method === 'Mobile') document.getElementById('disbursement_mobile_fields').classList.remove('hidden');
        }

        function checkBackdateWarning(dateInput) {
            const warningDiv = document.getElementById('backdate-warning');
            const warningTitle = document.getElementById('backdate-warning-title');
            const warningMessage = document.getElementById('backdate-warning-message');
            
            if (!dateInput.value) {
                warningDiv.style.display = 'none';
                return;
            }
            
            const selectedDate = new Date(dateInput.value);
            const today = new Date();
            today.setHours(0, 0, 0, 0);
            
            if (selectedDate < today) {
                const daysAgo = Math.floor((today - selectedDate) / (1000 * 60 * 60 * 24));
                
                if (daysAgo > 90) {
                    warningTitle.textContent = '⚠️ Extensive Backdating Warning';
                    warningMessage.textContent = `This application is being backdated ${daysAgo} days (over 3 months). Please ensure this is intentional and properly documented.`;
                } else if (daysAgo > 30) {
                    warningTitle.textContent = 'Backdating Notice';
                    warningMessage.textContent = `Application is being backdated ${daysAgo} days. This may affect reporting timelines.`;
                } else if (daysAgo > 7) {
                    warningTitle.textContent = 'Backdating Information';
                    warningMessage.textContent = `Application is being backdated ${daysAgo} days.`;
                } else {
                    warningTitle.textContent = 'Minor Backdating';
                    warningMessage.textContent = `Application is being backdated ${daysAgo} days.`;
                }
                
                warningDiv.style.display = 'block';
            } else {
                warningDiv.style.display = 'none';
            }
        }

        // --- DOCUMENT MANAGEMENT FUNCTIONS ---

        function loadDocumentRequirements(productId) {
            if (!productId) {
                // Clear document requirements if no product selected
                document.getElementById('document-requirements-container').innerHTML = `
                    <div class="text-center py-8">
                        <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-blue-100">
                            <i class="fas fa-info-circle text-blue-600 text-2xl"></i>
                        </div>
                        <h3 class="mt-4 text-lg font-medium text-gray-900">No Product Selected</h3>
                        <p class="mt-2 text-sm text-gray-500">
                            Please select a loan product to view document requirements.
                        </p>
                    </div>
                `;
                updateDocumentCounts(0, 0, 0);
                documentRequirements = {};
                return;
            }

            // Show loading state
            document.getElementById('document-requirements-container').innerHTML = `
                <div class="text-center py-8">
                    <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-blue-100">
                        <i class="fas fa-spinner fa-spin text-blue-600 text-2xl"></i>
                    </div>
                    <h3 class="mt-4 text-lg font-medium text-gray-900">Loading Requirements</h3>
                    <p class="mt-2 text-sm text-gray-500">
                        Loading document requirements for selected product...
                    </p>
                </div>
            `;

            $.ajax({
                url: '{{ route("admin.applications.get-product-requirements") }}',
                type: 'GET',
                data: { product_id: productId },
                success: function(response) {
                    if (response.success) {
                        documentRequirements = response.requirements;
                        renderDocumentRequirements(response.requirements);
                        updateDocumentCounts(
                            response.mandatory_count,
                            response.optional_count,
                            0 // Start with 0 uploaded
                        );
                    } else {
                        document.getElementById('document-requirements-container').innerHTML = `
                            <div class="text-center py-8">
                                <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-yellow-100">
                                    <i class="fas fa-exclamation-triangle text-yellow-600 text-2xl"></i>
                                </div>
                                <h3 class="mt-4 text-lg font-medium text-gray-900">No Requirements</h3>
                                <p class="mt-2 text-sm text-gray-500">
                                    This loan product has no document requirements.
                                </p>
                            </div>
                        `;
                        updateDocumentCounts(0, 0, 0);
                    }
                },
                error: function() {
                    document.getElementById('document-requirements-container').innerHTML = `
                        <div class="text-center py-8">
                            <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-red-100">
                                <i class="fas fa-times text-red-600 text-2xl"></i>
                            </div>
                            <h3 class="mt-4 text-lg font-medium text-gray-900">Error Loading Requirements</h3>
                            <p class="mt-2 text-sm text-gray-500">
                                Failed to load document requirements. Please try again.
                            </p>
                        </div>
                    `;
                    updateDocumentCounts(0, 0, 0);
                }
            });
        }

        function renderDocumentRequirements(requirements) {
            const container = document.getElementById('document-requirements-container');
            
            if (!requirements || requirements.length === 0) {
                container.innerHTML = `
                    <div class="text-center py-8">
                        <div class="mx-auto flex items-center justify-center h-16 w-16 rounded-full bg-green-100">
                            <i class="fas fa-check-circle text-green-600 text-2xl"></i>
                        </div>
                        <h3 class="mt-4 text-lg font-medium text-gray-900">No Requirements</h3>
                        <p class="mt-2 text-sm text-gray-500">
                            This loan product has no document requirements.
                        </p>
                    </div>
                `;
                return;
            }

            let html = '<div class="space-y-4">';
            
            requirements.forEach((req, index) => {
                const isUploaded = uploadedDocuments[req.id];
                const statusClass = isUploaded ? 'bg-green-50 border-green-200' : (req.is_mandatory ? 'bg-red-50 border-red-100' : 'bg-gray-50 border-gray-100');
                const statusIcon = isUploaded ? 'fa-check-circle text-green-500' : (req.is_mandatory ? 'fa-exclamation-circle text-red-500' : 'fa-circle text-gray-400');
                const statusText = isUploaded ? 'Uploaded' : (req.is_mandatory ? 'Required' : 'Optional');
                
                html += `
                    <div class="document-requirement-card ${req.is_mandatory ? 'mandatory' : 'optional'} p-4 rounded-lg border ${statusClass}">
                        <div class="flex justify-between items-start">
                            <div class="flex-1">
                                <div class="flex items-center mb-2">
                                    <i class="fas ${statusIcon} mr-2"></i>
                                    <h4 class="font-medium text-gray-900">${req.name}</h4>
                                    <span class="document-badge ${req.is_mandatory ? 'mandatory' : 'optional'} ml-2">
                                        ${req.is_mandatory ? 'Mandatory' : 'Optional'}
                                    </span>
                                </div>
                                ${req.description ? `<p class="text-sm text-gray-600 mb-3">${req.description}</p>` : ''}
                            </div>
                            <div class="ml-4">
                                ${isUploaded ? `
                                    <span class="inline-flex items-center px-3 py-1 rounded-full text-xs font-medium bg-green-100 text-green-800">
                                        <i class="fas fa-check mr-1"></i> Uploaded
                                    </span>
                                ` : `
                                    <button type="button" onclick="openDocumentModal(${req.id}, '${req.name.replace(/'/g, "\\'")}', '${req.description ? req.description.replace(/'/g, "\\'") : 'No description'}')"
                                            class="inline-flex items-center px-3 py-2 border border-transparent text-sm leading-4 font-medium rounded-md text-white bg-zedbank-blue hover:bg-zedbank-blue-dark">
                                        <i class="fas fa-upload mr-2"></i> Upload
                                    </button>
                                `}
                            </div>
                        </div>
                        
                        ${isUploaded ? `
                        <div class="mt-3 p-3 bg-white rounded border border-green-200">
                            <div class="flex justify-between items-center">
                                <div class="flex items-center">
                                    <i class="fas fa-file text-blue-500 mr-2"></i>
                                    <span class="text-sm font-medium text-gray-900">${uploadedDocuments[req.id].name}</span>
                                    <span class="text-xs text-gray-500 ml-2">${formatFileSize(uploadedDocuments[req.id].size)}</span>
                                </div>
                                <div class="flex space-x-2">
                                    <button type="button" onclick="removeDocument(${req.id})" 
                                            class="text-red-600 hover:text-red-800 text-sm">
                                        <i class="fas fa-trash"></i> Remove
                                    </button>
                                    <button type="button" onclick="openDocumentModal(${req.id}, '${req.name.replace(/'/g, "\\'")}', '${req.description ? req.description.replace(/'/g, "\\'") : 'No description'}')"
                                            class="text-blue-600 hover:text-blue-800 text-sm">
                                        <i class="fas fa-sync-alt"></i> Replace
                                    </button>
                                </div>
                            </div>
                        </div>
                        ` : ''}
                    </div>
                `;
            });
            
            html += '</div>';
            container.innerHTML = html;
        }

        function openDocumentModal(requirementId, requirementName, description) {
            currentRequirementId = requirementId;
            document.getElementById('modalRequirementId').value = requirementId;
            document.getElementById('modalRequirementName').value = requirementName;
            document.getElementById('modalDocumentName').textContent = requirementName;
            document.getElementById('modalDocumentDescription').textContent = description || 'No description available';
            
            // Reset file input and preview
            document.getElementById('documentFileInput').value = '';
            document.getElementById('filePreview').classList.add('hidden');
            document.getElementById('uploadProgress').classList.add('hidden');
            document.getElementById('uploadButton').disabled = false;
            
            // Show modal
            document.getElementById('documentUploadModal').classList.remove('hidden');
        }

        function closeDocumentModal() {
            document.getElementById('documentUploadModal').classList.add('hidden');
            currentRequirementId = null;
        }

        function uploadDocument() {
            const fileInput = document.getElementById('documentFileInput');
            const file = fileInput.files[0];
            
            if (!file) {
                alert('Please select a file to upload');
                return;
            }

            // Validate file size (10MB max)
            const maxSize = 10 * 1024 * 1024; // 10MB in bytes
            if (file.size > maxSize) {
                alert('File size exceeds 10MB limit. Please choose a smaller file.');
                return;
            }

            // Validate file type
            const validTypes = ['application/pdf', 'image/jpeg', 'image/jpg', 'image/png', 
                              'application/msword', 'application/vnd.openxmlformats-officedocument.wordprocessingml.document',
                              'application/vnd.ms-excel', 'application/vnd.openxmlformats-officedocument.spreadsheetml.sheet'];
            
            if (!validTypes.includes(file.type)) {
                alert('Please upload a valid document file (PDF, JPG, PNG, DOC, DOCX, XLS, XLSX).');
                return;
            }

            // Show progress
            document.getElementById('uploadProgress').classList.remove('hidden');
            document.getElementById('uploadButton').disabled = true;
            
            // Simulate upload progress (in real app, this would be actual AJAX upload)
            simulateUploadProgress(file);
        }

        function simulateUploadProgress(file) {
            let progress = 0;
            const progressBar = document.getElementById('progressBar');
            const progressText = document.getElementById('progressText');
            
            const interval = setInterval(() => {
                progress += 10;
                progressBar.style.width = progress + '%';
                progressText.textContent = `Uploading: ${progress}%`;
                
                if (progress >= 100) {
                    clearInterval(interval);
                    
                    // Simulate successful upload
                    setTimeout(() => {
                        // Store uploaded document info
                        uploadedDocuments[currentRequirementId] = {
                            name: file.name,
                            size: file.size,
                            type: file.type,
                            requirementId: currentRequirementId
                        };
                        
                        // Hide progress and close modal
                        document.getElementById('uploadProgress').classList.add('hidden');
                        closeDocumentModal();
                        
                        // Re-render requirements with updated status
                        renderDocumentRequirements(Object.values(documentRequirements));
                        
                        // Update counts
                        const mandatoryCount = Object.values(documentRequirements).filter(r => r.is_mandatory).length;
                        const optionalCount = Object.values(documentRequirements).filter(r => !r.is_mandatory).length;
                        const uploadedCount = Object.keys(uploadedDocuments).length;
                        
                        updateDocumentCounts(mandatoryCount, optionalCount, uploadedCount);
                        
                        // Add hidden input for the uploaded document
                        addDocumentInput(currentRequirementId, file);
                        
                        // Show success message
                        showNotification('Document uploaded successfully', 'success');
                    }, 500);
                }
            }, 100);
        }

        function addDocumentInput(requirementId, file) {
            // Remove existing input for this requirement if any
            const existingInput = document.querySelector(`input[name="documents[${requirementId}]"]`);
            if (existingInput) {
                existingInput.remove();
            }
            
            // Create hidden input for the document
            const input = document.createElement('input');
            input.type = 'hidden';
            input.name = `documents[${requirementId}]`;
            input.value = JSON.stringify({
                name: file.name,
                size: file.size,
                type: file.type,
                requirementId: requirementId,
                notes: document.getElementById('documentNotes').value
            });
            
            document.getElementById('application-form').appendChild(input);
        }

        function removeDocument(requirementId) {
            if (confirm('Are you sure you want to remove this document?')) {
                delete uploadedDocuments[requirementId];
                
                // Remove hidden input
                const input = document.querySelector(`input[name="documents[${requirementId}]"]`);
                if (input) {
                    input.remove();
                }
                
                // Re-render requirements
                renderDocumentRequirements(Object.values(documentRequirements));
                
                // Update counts
                const mandatoryCount = Object.values(documentRequirements).filter(r => r.is_mandatory).length;
                const optionalCount = Object.values(documentRequirements).filter(r => !r.is_mandatory).length;
                const uploadedCount = Object.keys(uploadedDocuments).length;
                
                updateDocumentCounts(mandatoryCount, optionalCount, uploadedCount);
                
                showNotification('Document removed', 'info');
            }
        }

        function updateDocumentCounts(mandatoryCount, optionalCount, uploadedCount) {
            const totalCount = mandatoryCount + optionalCount;
            
            // Update badge
            document.getElementById('document-count-badge').textContent = `${uploadedCount}/${totalCount}`;
            
            // Update counts
            document.getElementById('mandatory-count').textContent = `${uploadedCount}/${mandatoryCount}`;
            document.getElementById('optional-count').textContent = optionalCount;
            document.getElementById('uploaded-count').textContent = uploadedCount;
            
            // Update progress bar
            const completionPercentage = mandatoryCount > 0 ? Math.round((uploadedCount / mandatoryCount) * 100) : 100;
            document.getElementById('completion-percentage').textContent = `${completionPercentage}%`;
            document.getElementById('completion-bar').style.width = `${completionPercentage}%`;
            
            // Update completion text
            if (mandatoryCount > 0) {
                document.getElementById('completion-text').textContent = 
                    `${uploadedCount} of ${mandatoryCount} mandatory documents uploaded`;
            } else {
                document.getElementById('completion-text').textContent = 
                    'No mandatory document requirements';
            }
            
            // Update status summary
            updateDocumentStatusSummary(mandatoryCount, optionalCount, uploadedCount);
        }

        function updateDocumentStatusSummary(mandatoryCount, optionalCount, uploadedCount) {
            const summaryContainer = document.getElementById('document-status-summary');
            const completionPercentage = mandatoryCount > 0 ? Math.round((uploadedCount / mandatoryCount) * 100) : 100;
            
            if (mandatoryCount === 0) {
                summaryContainer.innerHTML = `
                    <div class="text-center py-2">
                        <i class="fas fa-check-circle text-green-500 text-xl mb-2"></i>
                        <p class="text-sm font-medium text-green-600">No document requirements</p>
                    </div>
                `;
            } else if (uploadedCount === mandatoryCount) {
                summaryContainer.innerHTML = `
                    <div class="text-center py-2">
                        <i class="fas fa-check-circle text-green-500 text-xl mb-2"></i>
                        <p class="text-sm font-medium text-green-600">All mandatory documents uploaded</p>
                        <p class="text-xs text-gray-500">${uploadedCount}/${mandatoryCount} documents</p>
                    </div>
                `;
            } else if (uploadedCount > 0) {
                summaryContainer.innerHTML = `
                    <div class="text-center py-2">
                        <i class="fas fa-exclamation-circle text-yellow-500 text-xl mb-2"></i>
                        <p class="text-sm font-medium text-yellow-600">Incomplete</p>
                        <p class="text-xs text-gray-500">${uploadedCount}/${mandatoryCount} mandatory documents</p>
                        <div class="mt-2 w-full bg-gray-200 rounded-full h-1.5">
                            <div class="bg-yellow-500 h-1.5 rounded-full" style="width: ${completionPercentage}%"></div>
                        </div>
                    </div>
                `;
            } else {
                summaryContainer.innerHTML = `
                    <div class="text-center py-2">
                        <i class="fas fa-times-circle text-red-500 text-xl mb-2"></i>
                        <p class="text-sm font-medium text-red-600">No documents uploaded</p>
                        <p class="text-xs text-gray-500">${mandatoryCount} mandatory documents required</p>
                    </div>
                `;
            }
        }

        function formatFileSize(bytes) {
            if (bytes === 0) return '0 Bytes';
            const k = 1024;
            const sizes = ['Bytes', 'KB', 'MB', 'GB'];
            const i = Math.floor(Math.log(bytes) / Math.log(k));
            return parseFloat((bytes / Math.pow(k, i)).toFixed(2)) + ' ' + sizes[i];
        }

        function showNotification(message, type = 'info') {
            const notification = document.createElement('div');
            notification.className = `fixed top-4 right-4 px-4 py-3 rounded shadow-lg z-50 ${
                type === 'success' ? 'bg-green-100 text-green-700 border-green-400' :
                type === 'error' ? 'bg-red-100 text-red-700 border-red-400' :
                'bg-blue-100 text-blue-700 border-blue-400'
            }`;
            notification.innerHTML = `
                <div class="flex">
                    <div class="py-1">
                        <i class="fas ${
                            type === 'success' ? 'fa-check-circle' :
                            type === 'error' ? 'fa-exclamation-circle' :
                            'fa-info-circle'
                        } mr-3"></i>
                    </div>
                    <div>
                        <p class="font-medium">${message}</p>
                    </div>
                    <button onclick="this.parentElement.parentElement.remove()" class="ml-4">
                        <i class="fas fa-times"></i>
                    </button>
                </div>
            `;
            document.body.appendChild(notification);
            
            setTimeout(() => {
                if (notification.parentElement) {
                    notification.remove();
                }
            }, 3000);
        }

        // --- DRAG AND DROP FUNCTIONALITY ---
        document.getElementById('dropZone').addEventListener('click', () => {
            document.getElementById('documentFileInput').click();
        });

        document.getElementById('documentFileInput').addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                updateFilePreview(file);
            }
        });

        function updateFilePreview(file) {
            document.getElementById('fileName').textContent = file.name;
            document.getElementById('fileSize').textContent = formatFileSize(file.size);
            document.getElementById('filePreview').classList.remove('hidden');
        }

        function clearFile() {
            document.getElementById('documentFileInput').value = '';
            document.getElementById('filePreview').classList.add('hidden');
        }

        // Drag and drop events
        const dropZone = document.getElementById('dropZone');
        
        dropZone.addEventListener('dragover', (e) => {
            e.preventDefault();
            dropZone.classList.add('drag-over');
        });

        dropZone.addEventListener('dragleave', () => {
            dropZone.classList.remove('drag-over');
        });

        dropZone.addEventListener('drop', (e) => {
            e.preventDefault();
            dropZone.classList.remove('drag-over');
            
            const file = e.dataTransfer.files[0];
            if (file) {
                document.getElementById('documentFileInput').files = e.dataTransfer.files;
                updateFilePreview(file);
            }
        });

        // --- HANDLE FORM SUBMISSION FOR DRAFT VS SUBMIT ---
        document.addEventListener('DOMContentLoaded', function() {
            const submitBtn = document.querySelector('button[name="action"][value="submit"]');
            const draftBtn = document.querySelector('button[name="action"][value="draft"]');
            const formActionInput = document.getElementById('form_action');
            
            if (submitBtn) {
                submitBtn.addEventListener('click', function(e) {
                    formActionInput.value = 'submit';
                });
            }
            
            if (draftBtn) {
                draftBtn.addEventListener('click', function(e) {
                    formActionInput.value = 'draft';
                });
            }
            
            // Also handle the form submission to show confirmation
            const form = document.getElementById('application-form');
            if (form) {
                form.addEventListener('submit', function(e) {
                    const action = formActionInput.value;
                    
                    if (action === 'draft') {
                        // Optional: Add confirmation for draft
                        const confirmed = confirm('Save this application as draft? You can edit and submit it later from the Draft Applications section.');
                        if (!confirmed) {
                            e.preventDefault();
                            return false;
                        }
                    } else {
                        // Optional: Add confirmation for submit
                        const confirmed = confirm('Submit this application for processing? Once submitted, it will be reviewed for approval.');
                        if (!confirmed) {
                            e.preventDefault();
                            return false;
                        }
                    }
                    
                    // Show loading state
                    const submitBtn = form.querySelector('button[name="action"][value="submit"]');
                    const draftBtn = form.querySelector('button[name="action"][value="draft"]');
                    
                    if (submitBtn && action === 'submit') {
                        submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Submitting...';
                        submitBtn.disabled = true;
                    } else if (draftBtn && action === 'draft') {
                        draftBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Saving...';
                        draftBtn.disabled = true;
                    }
                    
                    return true;
                });
            }
        });

        // --- INITIALIZATION ---
        $(document).ready(function() {
            // Load customer details when selected
            $(document).on('change', '#loan_customer_select', function() {
                const id = $(this).val();
                if (!id) return;
                $.ajax({
                    url: '{{ route("admin.applications.get_customer", ["id" => "TEMP_ID"]) }}'.replace('TEMP_ID', id),
                    type: 'GET',
                    success: function(res) {
                        // Personal Details
                        $('#title').val(res.title);
                        $('#first_name').val(res.first_name);
                        $('#surname').val(res.surname);
                        $('#nrc_no').val(res.nrc_no);
                        $('#phone').val(res.phone);
                        $('#email').val(res.email);
                        $('#physical_address').val(res.physical_address);
                        $('#town').val(res.town);
                        
                        // If customer has existing employment/kin data, it would come here
                    }
                });
            });
            
            // Initialize form
            updateTenureLabel();
            calculateInstallment();
            
            // Monitor fee fields for changes
            const feeFields = ['hidden_processing_fee', 'hidden_processing_fee_basis', 'hidden_admin_fee', 'hidden_insurance_fee'];
            
            feeFields.forEach(fieldId => {
                const field = document.getElementById(fieldId);
                if (field) {
                    field.addEventListener('change', calculateInstallment);
                }
            });
            
            // Check initial backdate warning if date is pre-filled from old data
            if (applicationDateInput && applicationDateInput.value) {
                checkBackdateWarning(applicationDateInput);
            }
            
            // Load document requirements if product is pre-selected (from validation errors)
            const selectedProductId = document.getElementById('product_id').value;
            if (selectedProductId) {
                loadDocumentRequirements(selectedProductId);
            }
        });
    </script>
</body>
</html>