@php
    // Define page variables
    $pageTitle = 'Loan Applications';
    $pageSubtitle = 'Manage and review loan applications';

    // This count now correctly excludes 'disbursed' applications due to controller change
    $pendingCount = $applications->where('status', 'submitted')->count() ?? 0;
    
    // Dummy stats for header/sidebar (Pass to partial if used)
    $stats_sidebar = [
        'pending_applications' => $pendingCount, 
    ];
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | ZedBankOS</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; 
            --zedbank-blue-dark: #005A94; 
        }

        /* Custom Tailwind-like classes */
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Local Overrides */
        .focus\:ring-zedbank-blue:focus { --tw-ring-color: var(--zedbank-blue); }
        .focus\:border-zedbank-blue:focus { border-color: var(--zedbank-blue); }
        .bg-primary { background-color: var(--zedbank-blue); }
        
        .action-btn { width: 30px; height: 30px; border-radius: 4px; display: inline-flex; align-items: center; justify-content: center; margin: 0 1px; transition: all 0.2s; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    {{-- Assuming 'accounting.partials.sidebar' exists with the new style --}}
    @include('accounting.partials.sidebar', ['stats' => $stats_sidebar])

    <div class="lg:ml-64 min-h-screen">
        
        {{-- Pass page variables to the header partial --}}
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
                    {{ session('success') }}
                </div>
            @endif
            
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-4 mb-6">
                <div class="flex flex-col md:flex-row justify-between items-start md:items-center gap-4">
                    <div class="w-full md:w-auto">
                        <div class="relative">
                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                <i class="fas fa-search text-gray-400"></i>
                            </div>
                            <input 
                                type="text" 
                                id="searchInput" 
                                class="block w-full md:w-80 pl-10 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-zedbank-blue focus:border-zedbank-blue focus:ring-1 text-sm" 
                                placeholder="Search by application no, customer, product..."
                            >
                        </div>
                    </div>
                    <div class="flex flex-wrap gap-2 w-full md:w-auto">
                        <a href="{{ route('admin.applications.create') }}" class="px-4 py-2 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark transition flex items-center gap-2 text-sm font-semibold">
                            <i class="fas fa-plus"></i>
                            <span>New Application</span>
                        </a>
                        <button class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2 text-sm">
                            <i class="fas fa-filter"></i>
                            <span>Filter</span>
                        </button>
                        <button class="px-4 py-2 bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2 text-sm">
                            <i class="fas fa-download"></i>
                            <span>Export</span>
                        </button>
                    </div>
                </div>
            </div>
            
            <div class="overflow-x-auto bg-white shadow-sm rounded-lg border border-gray-200">
                <table class="min-w-full text-sm" id="dataTable">
                    <thead class="bg-gray-50 border-b border-gray-200">
                        <tr>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Loan ID</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Application No</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Customer</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Product</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Submission Date</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Amount</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Interest Rate</th>
                            
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Status</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Tenure</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Frequency</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Installment</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Source</th>
                            <th class="px-4 py-3 text-left text-gray-600 font-medium whitespace-nowrap">Actions</th>
                        </tr>
                    </thead>

                    <tbody>
                        @forelse($applications as $application)
                            <tr class="border-b hover:bg-gray-50 transition">
                                <td class="px-4 py-3 text-gray-700 font-medium">{{ $application->id }}</td>
                                <td class="px-4 py-3 text-gray-700 font-medium">{{ $application->application_number }}</td>
                                <td class="px-4 py-3 text-gray-700">
                                    {{ $application->customer ? $application->customer->first_name . ' ' . $application->customer->surname : '' }}
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    {{ $application->product->product ?? '' }}
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    {{ $application->submitted_at ? $application->submitted_at->format('M d, Y') : '' }}
                                </td>
                                <td class="px-4 py-3 text-gray-800 font-bold">
                                    ZMW {{ number_format($application->loan_amount ?? 0, 2) }}
                                </td>
                                <td class="px-4 py-3 text-gray-700">
    {{ number_format($application->interest_rate ?? 0, 2) }}% 
    <span class="text-xs text-gray-500">
        @if($application->payment_frequency == 'one_off')
            (Daily)
        @else
            (Monthly)
        @endif
    </span>
</td>
                                
                                {{-- MODIFICATION: ADDED DATA CELLS --}}
                                <td class="px-4 py-3 text-gray-700">
                                @php
                                   $statusColors = [
                                      'draft' => 'bg-gray-100 text-gray-800',
                                      'submitted' => 'bg-blue-100 text-blue-800',
                                      'approved' => 'bg-green-100 text-green-800',
                                      'rejected' => 'bg-red-100 text-red-800',
                                      'disbursed' => 'bg-purple-100 text-purple-800',
                                    ];
        
                                    $status = $application->status ?? 'draft';
                                    $color = $statusColors[$status] ?? 'bg-gray-100 text-gray-800';
                                 @endphp
    
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium {{ $color }} capitalize">
                                     {{ ucfirst($status) }}
                                   </span>
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    {{ $application->loan_tenure }} {{ $application->payment_frequency == 'one_off' ? 'Days' : 'Months' }}
                                </td>
                                
                                <td class="px-4 py-3 text-gray-700">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-blue-100 text-blue-800 capitalize">
                                        {{ str_replace('_', ' ', $application->payment_frequency) }}
                                    </span>
                                </td>
                                <td class="px-4 py-3 text-gray-800 font-bold">
                                    ZMW {{ number_format($application->installment_amount ?? 0, 2) }}
                                </td>
                                <td class="px-4 py-3 text-gray-700">
                                    <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium bg-green-100 text-green-800 capitalize">
                                        {{ $application->source ?? 'online' }}
                                    </span>
                                </td>
                                
                                <td class="px-4 py-3 whitespace-nowrap">
                                    <div class="flex space-x-1 justify-center">
                                        <a href="{{ route('admin.applications.show', $application->id) }}" 
                                            class="action-btn bg-gray-100 text-zedbank-blue hover:bg-zedbank-blue-dark hover:text-white" 
                                            title="View Details">
                                            <i class="fas fa-eye text-xs"></i>
                                        </a>
                                        <a href="{{ route('admin.applications.schedule.view', $application->id) }}" 
                                            class="action-btn bg-yellow-100 text-yellow-800 hover:bg-yellow-800 hover:text-white" 
                                            title="View Amortization Schedule">
                                            <i class="fas fa-calendar-alt text-xs"></i>
                                        </a>
                                        <a href="{{ route('admin.applications.edit', $application->id) }}" 
                                            class="action-btn bg-green-50 text-green-600 hover:bg-green-800 hover:text-white" 
                                            title="Edit Application">
                                            <i class="fas fa-edit text-xs"></i>
                                        </a>
                                        
                                        @if(auth()->user()->hasPermission('applications.approve') && $application->status !== 'approved' && $application->status !== 'rejected')
                                            <form method="POST" action="{{ route('admin.applications.approve', $application->id) }}" class="inline">
                                                @csrf
                                                <button type="submit" 
                                                        class="action-btn bg-blue-100 text-blue-600 hover:bg-blue-600 hover:text-white" 
                                                        title="Approve"
                                                        onclick="return confirm('Are you sure you want to approve this application?')">
                                                    <i class="fas fa-check text-xs"></i>
                                                </button>
                                            </form>
                                            <form method="POST" action="{{ route('admin.applications.reject', $application->id) }}" class="inline">
                                                @csrf
                                                <button type="submit" 
                                                        class="action-btn bg-red-100 text-red-600 hover:bg-red-600 hover:text-white" 
                                                        title="Reject"
                                                        onclick="return confirm('Are you sure you want to reject this application?')">
                                                    <i class="fas fa-times text-xs"></i>
                                                </button>
                                            </form>
                                        @endif
                                    </div>
                                </td>
                            </tr>
                        @empty
                            <tr>
                                <td colspan="12" class="px-4 py-8 text-center text-gray-500">
                                    <div class="flex flex-col items-center justify-center">
                                        <i class="fas fa-file-invoice-dollar text-4xl text-gray-300 mb-4"></i>
                                        <p class="text-lg">No non-disbursed applications found</p>
                                        <p class="text-sm">New applications will appear here for review.</p>
                                    </div>
                                </td>
                            </tr>
                        @endforelse
                    </tbody>
                </table>
            </div>
            
            <div class="mt-6 flex flex-col md:flex-row justify-between items-center gap-4">
                <div class="text-sm text-gray-600">
                    Showing <span class="font-medium">{{ $applications->firstItem() ?? 0 }}</span> to 
                    <span class="font-medium">{{ $applications->lastItem() ?? 0 }}</span> of 
                    <span class="font-medium">{{ $applications->total() }}</span> entries
                </div>
                <div class="flex space-x-2">
                    @if($applications->onFirstPage())
                        <button class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg text-gray-400 cursor-not-allowed flex items-center gap-2" disabled>
                            <i class="fas fa-chevron-left"></i>
                            <span>Previous</span>
                        </button>
                    @else
                        <a href="{{ $applications->previousPageUrl() }}" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2">
                            <i class="fas fa-chevron-left"></i>
                            <span>Previous</span>
                        </a>
                    @endif

                    {{-- Simple Pagination Links --}}
                    @foreach($applications->getUrlRange(1, $applications->lastPage()) as $page => $url)
                        @if($page == $applications->currentPage())
                            <span class="px-4 py-2 text-sm bg-zedbank-blue text-white rounded-lg">{{ $page }}</span>
                        @else
                            <a href="{{ $url }}" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition">{{ $page }}</a>
                        @endif
                    @endforeach

                    @if($applications->hasMorePages())
                        <a href="{{ $applications->nextPageUrl() }}" class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg hover:bg-gray-50 transition flex items-center gap-2">
                            <span>Next</span>
                            <i class="fas fa-chevron-right"></i>
                        </a>
                    @else
                        <button class="px-4 py-2 text-sm bg-white border border-gray-300 rounded-lg text-gray-400 cursor-not-allowed flex items-center gap-2" disabled>
                            <span>Next</span>
                            <i class="fas fa-chevron-right"></i>
                        </button>
                    @endif
                </div>
            </div>
        </main>
    </div>

    <script>
        // Real-time Search functionality
        document.getElementById('searchInput')?.addEventListener('input', function() {
            const searchTerm = this.value.toLowerCase();
            const rows = document.querySelectorAll('#dataTable tbody tr');
            
            rows.forEach(row => {
                const rowText = row.textContent.toLowerCase();
                row.style.display = rowText.includes(searchTerm) ? '' : 'none';
            });
        });
    </script>
</body>
</html>