@php
    // Define page variables
    $pageTitle = 'Close Loan';
    $pageSubtitle = 'Manually close this loan account';
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>{{ $pageTitle }} | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; /* ThemeBlue */
            --zedbank-blue-dark: #005A94; /* Darker shade for hover/active background */
        }

        /* Custom Tailwind-like classes */
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Local Overrides */
        .focus\:ring-primary:focus { --tw-ring-color: var(--zedbank-blue); }
        .focus\:border-primary:focus { border-color: var(--zedbank-blue); }
        .bg-primary { background-color: var(--zedbank-blue); }
        
        /* Custom color for status badges */
        .status-badge {
            @apply inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium;
        }
        .status-active { @apply bg-green-100 text-green-800; }
        .status-closed { @apply bg-gray-100 text-gray-800; }
        .status-default { @apply bg-red-100 text-red-800; }
        
        /* Custom color for balances */
        .balance-positive { color: #059669; }
        .balance-negative { color: #DC2626; }
        .balance-neutral { color: #4B5563; }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    {{-- Including sidebar --}}
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64 min-h-screen">
        
        {{-- Including header with page variables --}}
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            {{-- Breadcrumb Navigation --}}
            <div class="mb-6">
                <nav class="flex" aria-label="Breadcrumb">
                    <ol class="inline-flex items-center space-x-1 md:space-x-3">
                        <li class="inline-flex items-center">
                            <a href="{{ route('admin.dashboard') }}" class="inline-flex items-center text-sm font-medium text-gray-700 hover:text-zedbank-blue">
                                <i class="fas fa-home mr-2"></i>
                                Dashboard
                            </a>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400"></i>
                                <a href="{{ route('admin.loans.active') }}" class="ml-1 text-sm font-medium text-gray-700 hover:text-zedbank-blue md:ml-2">
                                    Loans
                                </a>
                            </div>
                        </li>
                        <li>
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400"></i>
                                <a href="{{ route('admin.loans.show', $loan->loanid) }}" class="ml-1 text-sm font-medium text-gray-700 hover:text-zedbank-blue md:ml-2">
                                    Loan {{ $loan->loannumber }}
                                </a>
                            </div>
                        </li>
                        <li aria-current="page">
                            <div class="flex items-center">
                                <i class="fas fa-chevron-right text-gray-400"></i>
                                <span class="ml-1 text-sm font-medium text-gray-500 md:ml-2">Close Loan</span>
                            </div>
                        </li>
                    </ol>
                </nav>
            </div>

            {{-- Page Header with Action Buttons --}}
            <div class="mb-8">
                <div class="flex flex-col md:flex-row md:items-center justify-between gap-4">
                    <div>
                        <div class="flex items-center mb-2">
                            <a href="{{ route('admin.loans.show', $loan->loanid) }}" 
                               class="inline-flex items-center text-zedbank-blue hover:text-zedbank-blue-dark mr-3 transition">
                                <i class="fas fa-arrow-left mr-1"></i>
                                Back to Loan
                            </a>
                            <h1 class="text-3xl font-bold text-gray-800">Close Loan: {{ $loan->loannumber }}</h1>
                        </div>
                        <p class="text-gray-600">
                            Permanently close this loan account. This action should only be performed when the loan has been fully settled or under special circumstances.
                        </p>
                    </div>
                    
                    <div class="flex flex-wrap gap-2">
                        <a href="{{ route('admin.loans.payment-form', $loan->loanid) }}" 
                           class="px-4 py-2 bg-green-600 text-white rounded-lg hover:bg-green-700 transition font-medium flex items-center">
                            <i class="fas fa-dollar-sign mr-2"></i>
                            Record Payment
                        </a>
                        <a href="{{ route('admin.loans.npl-form', $loan->loanid) }}" 
                           class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition font-medium flex items-center">
                            <i class="fas fa-exclamation-triangle mr-2"></i>
                            Classify as NPL
                        </a>
                    </div>
                </div>
            </div>
            
            {{-- Session Messages --}}
            @if(session('success'))
                <div class="mb-6 bg-green-50 border-l-4 border-green-500 p-4">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-check-circle text-green-500 text-xl"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-green-700 font-medium">{{ session('success') }}</p>
                        </div>
                    </div>
                </div>
            @endif
            
            @if(session('error'))
                <div class="mb-6 bg-red-50 border-l-4 border-red-500 p-4">
                    <div class="flex">
                        <div class="flex-shrink-0">
                            <i class="fas fa-exclamation-circle text-red-500 text-xl"></i>
                        </div>
                        <div class="ml-3">
                            <p class="text-red-700 font-medium">{{ session('error') }}</p>
                        </div>
                    </div>
                </div>
            @endif
            
            {{-- Quick Loan Overview --}}
            <div class="grid grid-cols-1 md:grid-cols-3 gap-6 mb-6">
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-5">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-sm text-gray-600">Customer</p>
                            <p class="text-lg font-bold text-gray-800">
                                @if($loan->customer)
                                    {{ $loan->customer->first_name }} {{ $loan->customer->surname }}
                                @else
                                    <span class="text-gray-500">Not Found</span>
                                @endif
                            </p>
                            @if($loan->customer)
                                <p class="text-xs text-gray-500 mt-1">{{ $loan->customer->customer_number }}</p>
                            @endif
                        </div>
                        <div class="w-12 h-12 bg-blue-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-user text-zedbank-blue text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-5">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-sm text-gray-600">Loan Product</p>
                            <p class="text-lg font-bold text-gray-800">
                                {{ $loan->loanType->product ?? 'N/A' }}
                            </p>
                            <p class="text-xs text-gray-500 mt-1">
                                {{ number_format($loan->interestrate, 2) }}% Interest
                            </p>
                        </div>
                        <div class="w-12 h-12 bg-green-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-file-contract text-green-600 text-xl"></i>
                        </div>
                    </div>
                </div>
                
                <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-5">
                    <div class="flex items-center justify-between mb-3">
                        <div>
                            <p class="text-sm text-gray-600">Current Status</p>
                            <p class="text-lg font-bold">
                                @php
                                    $statusClass = '';
                                    if ($loan->status === 'active') $statusClass = 'status-active';
                                    elseif ($loan->status === 'closed') $statusClass = 'status-closed';
                                    elseif ($loan->status === 'default') $statusClass = 'status-default';
                                    else $statusClass = 'status-active';
                                @endphp
                                <span class="status-badge {{ $statusClass }}">
                                    <i class="fas fa-{{ $loan->status === 'active' ? 'check' : ($loan->status === 'closed' ? 'lock' : 'exclamation-triangle') }} mr-1"></i>
                                    {{ ucfirst($loan->status) }}
                                </span>
                            </p>
                            @if($loan->isOverdue())
                                <p class="text-xs text-red-600 mt-1">
                                    <i class="fas fa-clock mr-1"></i>
                                    {{ $loan->days_overdue }} days overdue
                                </p>
                            @endif
                        </div>
                        <div class="w-12 h-12 bg-purple-100 rounded-lg flex items-center justify-center">
                            <i class="fas fa-chart-line text-purple-600 text-xl"></i>
                        </div>
                    </div>
                </div>
            </div>
            
            {{-- Main Content Grid --}}
            <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                {{-- Left Column: Loan Details --}}
                <div class="lg:col-span-2 space-y-6">
                    {{-- Balance Summary Card --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Balance Summary</h2>
                            <p class="text-sm text-gray-600 mt-1">Current outstanding balances</p>
                        </div>
                        <div class="p-6">
                            <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                {{-- Principal Section --}}
                                <div class="{{ $loan->principalbalance > 0 ? 'bg-red-50 border border-red-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                                    <div class="flex items-start justify-between">
                                        <div>
                                            <p class="text-sm text-gray-600">Principal Balance</p>
                                            <p class="text-2xl font-bold {{ $loan->principalbalance > 0 ? 'text-red-600' : 'text-green-600' }}">
                                                K {{ number_format($loan->principalbalance, 2) }}
                                            </p>
                                            <div class="flex items-center mt-2">
                                                <div class="w-full bg-gray-200 rounded-full h-2">
                                                    @php
                                                        $principalPercentage = $loan->amount > 0 ? ($loan->principalbalance / $loan->amount) * 100 : 0;
                                                    @endphp
                                                    <div class="bg-red-600 h-2 rounded-full" style="width: {{ min($principalPercentage, 100) }}%"></div>
                                                </div>
                                                <span class="ml-2 text-xs text-gray-600">{{ number_format($principalPercentage, 1) }}%</span>
                                            </div>
                                        </div>
                                        <div class="w-10 h-10 {{ $loan->principalbalance > 0 ? 'bg-red-100' : 'bg-green-100' }} rounded-lg flex items-center justify-center">
                                            <i class="fas fa-home {{ $loan->principalbalance > 0 ? 'text-red-600' : 'text-green-600' }}"></i>
                                        </div>
                                    </div>
                                    <p class="text-xs {{ $loan->principalbalance > 0 ? 'text-red-500' : 'text-green-500' }} mt-2">
                                        @if($loan->principalbalance > 0)
                                            <i class="fas fa-exclamation-circle mr-1"></i>
                                            Outstanding principal amount
                                        @else
                                            <i class="fas fa-check-circle mr-1"></i>
                                            Principal fully paid
                                        @endif
                                    </p>
                                </div>
                                
                                {{-- Interest Section --}}
                                <div class="{{ $loan->interestbalance > 0 ? 'bg-orange-50 border border-orange-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                                    <div class="flex items-start justify-between">
                                        <div>
                                            <p class="text-sm text-gray-600">Interest Balance</p>
                                            <p class="text-2xl font-bold {{ $loan->interestbalance > 0 ? 'text-orange-600' : 'text-green-600' }}">
                                                K {{ number_format($loan->interestbalance, 2) }}
                                            </p>
                                            <div class="mt-2">
                                                <span class="inline-block px-2 py-1 text-xs font-semibold rounded {{ $loan->interestbalance > 0 ? 'bg-orange-100 text-orange-800' : 'bg-green-100 text-green-800' }}">
                                                    {{ number_format($loan->interestrate, 2) }}% Rate
                                                </span>
                                            </div>
                                        </div>
                                        <div class="w-10 h-10 {{ $loan->interestbalance > 0 ? 'bg-orange-100' : 'bg-green-100' }} rounded-lg flex items-center justify-center">
                                            <i class="fas fa-percentage {{ $loan->interestbalance > 0 ? 'text-orange-600' : 'text-green-600' }}"></i>
                                        </div>
                                    </div>
                                    <p class="text-xs {{ $loan->interestbalance > 0 ? 'text-orange-500' : 'text-green-500' }} mt-2">
                                        @if($loan->interestbalance > 0)
                                            <i class="fas fa-exclamation-circle mr-1"></i>
                                            Accrued interest due
                                        @else
                                            <i class="fas fa-check-circle mr-1"></i>
                                            No interest due
                                        @endif
                                    </p>
                                </div>
                                
                                {{-- Processing Fees Section --}}
                                <div class="{{ $loan->processing_fee_balance > 0 ? 'bg-purple-50 border border-purple-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                                    <div class="flex items-start justify-between">
                                        <div>
                                            <p class="text-sm text-gray-600">Processing Fees</p>
                                            <p class="text-2xl font-bold {{ $loan->processing_fee_balance > 0 ? 'text-purple-600' : 'text-green-600' }}">
                                                K {{ number_format($loan->processing_fee_balance, 2) }}
                                            </p>
                                            <div class="mt-2">
                                                <span class="inline-block px-2 py-1 text-xs font-semibold rounded {{ $loan->processing_fee_balance > 0 ? 'bg-purple-100 text-purple-800' : 'bg-green-100 text-green-800' }}">
                                                    {{ $loan->processing_fee ?? 0 }}% Fee
                                                </span>
                                            </div>
                                        </div>
                                        <div class="w-10 h-10 {{ $loan->processing_fee_balance > 0 ? 'bg-purple-100' : 'bg-green-100' }} rounded-lg flex items-center justify-center">
                                            <i class="fas fa-file-invoice-dollar {{ $loan->processing_fee_balance > 0 ? 'text-purple-600' : 'text-green-600' }}"></i>
                                        </div>
                                    </div>
                                    <p class="text-xs {{ $loan->processing_fee_balance > 0 ? 'text-purple-500' : 'text-green-500' }} mt-2">
                                        @if($loan->processing_fee_balance > 0)
                                            <i class="fas fa-exclamation-circle mr-1"></i>
                                            Outstanding fees
                                        @else
                                            <i class="fas fa-check-circle mr-1"></i>
                                            Fees fully paid
                                        @endif
                                    </p>
                                </div>
                                
                                {{-- Total Outstanding Section --}}
                                <div class="{{ $loan->totalbalance > 0 ? 'bg-blue-50 border border-blue-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                                    <div class="flex items-start justify-between">
                                        <div>
                                            <p class="text-sm text-gray-600">Total Outstanding</p>
                                            <p class="text-2xl font-bold {{ $loan->totalbalance > 0 ? 'text-blue-600' : 'text-green-600' }}">
                                                K {{ number_format($loan->totalbalance, 2) }}
                                            </p>
                                            @if($loan->totalbalance > 0)
                                                <div class="flex items-center mt-2">
                                                    <div class="w-full bg-gray-200 rounded-full h-2">
                                                        @php
                                                            $totalPercentage = $loan->amount > 0 ? ($loan->totalbalance / $loan->amount) * 100 : 0;
                                                        @endphp
                                                        <div class="bg-blue-600 h-2 rounded-full" style="width: {{ min($totalPercentage, 100) }}%"></div>
                                                    </div>
                                                    <span class="ml-2 text-xs text-gray-600">{{ number_format($totalPercentage, 1) }}%</span>
                                                </div>
                                            @endif
                                        </div>
                                        <div class="w-10 h-10 {{ $loan->totalbalance > 0 ? 'bg-blue-100' : 'bg-green-100' }} rounded-lg flex items-center justify-center">
                                            <i class="fas fa-balance-scale {{ $loan->totalbalance > 0 ? 'text-blue-600' : 'text-green-600' }}"></i>
                                        </div>
                                    </div>
                                    <p class="text-xs {{ $loan->totalbalance > 0 ? 'text-blue-500' : 'text-green-500' }} mt-2">
                                        @if($loan->totalbalance > 0)
                                            <i class="fas fa-exclamation-circle mr-1"></i>
                                            Total amount due
                                        @else
                                            <i class="fas fa-check-circle mr-1"></i>
                                            Loan fully settled
                                        @endif
                                    </p>
                                </div>
                            </div>
                            
                            {{-- Warning Alert for Outstanding Balances --}}
                            @if($loan->totalbalance > 0)
                                <div class="mt-6 bg-yellow-50 border-l-4 border-yellow-400 p-4">
                                    <div class="flex">
                                        <div class="flex-shrink-0">
                                            <i class="fas fa-exclamation-triangle text-yellow-400 text-xl"></i>
                                        </div>
                                        <div class="ml-3">
                                            <h3 class="text-sm font-medium text-yellow-800">
                                                Outstanding Balances Detected
                                            </h3>
                                            <div class="mt-2 text-sm text-yellow-700">
                                                <p>
                                                    This loan has outstanding balances totaling <strong>K {{ number_format($loan->totalbalance, 2) }}</strong>.
                                                    Closing the loan with outstanding balances may require special authorization.
                                                </p>
                                                <p class="mt-1">
                                                    <strong>Recommendation:</strong> Consider processing a final payment before closing.
                                                </p>
                                            </div>
                                            <div class="mt-3">
                                                <div class="-mx-2 -my-1.5 flex">
                                                    <a href="{{ route('admin.loans.payment-form', $loan->loanid) }}" 
                                                       class="bg-yellow-100 px-3 py-1.5 rounded-md text-sm font-medium text-yellow-800 hover:bg-yellow-200 focus:outline-none focus:ring-2 focus:ring-offset-2 focus:ring-yellow-600">
                                                        Record Payment
                                                    </a>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @else
                                <div class="mt-6 bg-green-50 border-l-4 border-green-400 p-4">
                                    <div class="flex">
                                        <div class="flex-shrink-0">
                                            <i class="fas fa-check-circle text-green-400 text-xl"></i>
                                        </div>
                                        <div class="ml-3">
                                            <h3 class="text-sm font-medium text-green-800">
                                                Loan Fully Settled
                                            </h3>
                                            <div class="mt-2 text-sm text-green-700">
                                                <p>
                                                    This loan has been fully paid with no outstanding balances. You can safely close this loan account.
                                                </p>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            @endif
                        </div>
                    </div>
                    
                    {{-- Payment History Summary --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Payment Summary</h2>
                            <p class="text-sm text-gray-600 mt-1">Total payments received to date</p>
                        </div>
                        <div class="p-6">
                            <div class="grid grid-cols-2 md:grid-cols-4 gap-4">
                                <div class="text-center p-4 bg-gray-50 rounded-lg">
                                    <p class="text-sm text-gray-600">Principal Paid</p>
                                    <p class="text-2xl font-bold text-gray-800">K {{ number_format($loan->totalprincipalpaid, 2) }}</p>
                                    @if($loan->amount > 0)
                                        <p class="text-xs text-gray-500 mt-1">
                                            {{ number_format(($loan->totalprincipalpaid / $loan->amount) * 100, 1) }}% of principal
                                        </p>
                                    @endif
                                </div>
                                
                                <div class="text-center p-4 bg-gray-50 rounded-lg">
                                    <p class="text-sm text-gray-600">Interest Paid</p>
                                    <p class="text-2xl font-bold text-gray-800">K {{ number_format($loan->totalinterestpaid, 2) }}</p>
                                    <p class="text-xs text-gray-500 mt-1">
                                        @if($loan->totalprincipalpaid > 0)
                                            {{ number_format(($loan->totalinterestpaid / $loan->totalprincipalpaid) * 100, 1) }}% of principal paid
                                        @endif
                                    </p>
                                </div>
                                
                                <div class="text-center p-4 bg-gray-50 rounded-lg">
                                    <p class="text-sm text-gray-600">Total Paid</p>
                                    <p class="text-2xl font-bold text-gray-800">K {{ number_format($loan->totalpaid, 2) }}</p>
                                    @if($loan->amount > 0)
                                        <p class="text-xs text-gray-500 mt-1">
                                            {{ number_format(($loan->totalpaid / $loan->amount) * 100, 1) }}% of loan amount
                                        </p>
                                    @endif
                                </div>
                                
                                <div class="text-center p-4 bg-gray-50 rounded-lg">
                                    <p class="text-sm text-gray-600">Last Payment</p>
                                    <p class="text-lg font-bold text-gray-800">
                                        @if($loan->lastpaymentdate)
                                            {{ \Carbon\Carbon::parse($loan->lastpaymentdate)->format('M d, Y') }}
                                        @else
                                            Never
                                        @endif
                                    </p>
                                    @if($loan->lastpaymentdate && $loan->lastpaymentamount > 0)
                                        <p class="text-xs text-gray-500 mt-1">
                                            K {{ number_format($loan->lastpaymentamount, 2) }}
                                        </p>
                                    @endif
                                </div>
                            </div>
                            
                            {{-- Repayment Progress Bar --}}
                            @if($loan->amount > 0)
                                <div class="mt-6">
                                    <div class="flex justify-between items-center mb-2">
                                        <span class="text-sm font-medium text-gray-700">Repayment Progress</span>
                                        <span class="text-sm font-medium text-gray-700">{{ number_format(($loan->totalpaid / $loan->amount) * 100, 1) }}%</span>
                                    </div>
                                    <div class="w-full bg-gray-200 rounded-full h-3">
                                        @php
                                            $repaymentPercentage = min(($loan->totalpaid / $loan->amount) * 100, 100);
                                        @endphp
                                        <div class="bg-green-600 h-3 rounded-full" style="width: {{ $repaymentPercentage }}%"></div>
                                    </div>
                                    <div class="flex justify-between text-xs text-gray-500 mt-1">
                                        <span>K 0</span>
                                        <span>K {{ number_format($loan->amount, 2) }}</span>
                                    </div>
                                </div>
                            @endif
                        </div>
                    </div>
                </div>
                
                {{-- Right Column: Close Loan Form --}}
                <div class="space-y-6">
                    {{-- Close Loan Form --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Close Loan</h2>
                            <p class="text-sm text-gray-600 mt-1">Complete the form below to close this loan</p>
                        </div>
                        <div class="p-6">
                            <form action="{{ route('admin.loans.close', $loan->loanid) }}" method="POST" id="closeLoanForm">
                                @csrf
                                
                                <div class="space-y-5">
                                    {{-- Closure Date --}}
                                    <div>
                                        <label for="closure_date" class="block text-sm font-medium text-gray-700 mb-1">
                                            Closure Date <span class="text-red-500">*</span>
                                        </label>
                                        <div class="relative">
                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <i class="fas fa-calendar text-gray-400"></i>
                                            </div>
                                            <input type="date" id="closure_date" name="closure_date" required
                                                value="{{ old('closure_date', date('Y-m-d')) }}"
                                                max="{{ date('Y-m-d') }}"
                                                class="pl-10 block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary transition">
                                        </div>
                                        <p class="mt-1 text-xs text-gray-500">Date when the loan should be considered closed.</p>
                                        @error('closure_date')
                                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                    
                                    {{-- Closure Reason --}}
                                    <div>
                                        <label for="closing_reason" class="block text-sm font-medium text-gray-700 mb-1">
                                            Closure Reason <span class="text-red-500">*</span>
                                        </label>
                                        <select id="closing_reason" name="closing_reason" required
                                            class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary transition">
                                            <option value="">Select a reason...</option>
                                            <option value="fully_settled" {{ old('closing_reason') == 'fully_settled' ? 'selected' : '' }}>Fully Settled (Normal Closure)</option>
                                            <option value="loan_forgiveness" {{ old('closing_reason') == 'loan_forgiveness' ? 'selected' : '' }}>Loan Forgiveness</option>
                                            <option value="settlement_agreement" {{ old('closing_reason') == 'settlement_agreement' ? 'selected' : '' }}>Settlement Agreement</option>
                                            <option value="write_off_approved" {{ old('closing_reason') == 'write_off_approved' ? 'selected' : '' }}>Write-off Approved</option>
                                            <option value="customer_bankruptcy" {{ old('closing_reason') == 'customer_bankruptcy' ? 'selected' : '' }}>Customer Bankruptcy</option>
                                            <option value="deceased_customer" {{ old('closing_reason') == 'deceased_customer' ? 'selected' : '' }}>Deceased Customer</option>
                                            <option value="system_error" {{ old('closing_reason') == 'system_error' ? 'selected' : '' }}>System Error Correction</option>
                                            <option value="other" {{ old('closing_reason') == 'other' ? 'selected' : '' }}>Other (Specify in Notes)</option>
                                        </select>
                                        @error('closing_reason')
                                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                    
                                    {{-- Closure Notes --}}
                                    <div>
                                        <label for="closure_notes" class="block text-sm font-medium text-gray-700 mb-1">
                                            Closure Notes
                                        </label>
                                        <textarea id="closure_notes" name="closure_notes" rows="3"
                                            placeholder="Provide additional details about why this loan is being closed..."
                                            class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary transition">{{ old('closure_notes') }}</textarea>
                                        <p class="mt-1 text-xs text-gray-500">These notes will be saved with the loan record for audit purposes.</p>
                                        @error('closure_notes')
                                            <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                                        @enderror
                                    </div>
                                    
                                    {{-- Final Settlement Amount (Conditional) --}}
                                    @if($loan->totalbalance > 0)
                                        <div id="settlementSection">
                                            <label for="final_settlement_amount" class="block text-sm font-medium text-gray-700 mb-1">
                                                Final Settlement Amount
                                            </label>
                                            <div class="relative">
                                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                    <span class="text-gray-500">K</span>
                                                </div>
                                                <input type="number" id="final_settlement_amount" name="final_settlement_amount"
                                                    value="{{ old('final_settlement_amount', $loan->totalbalance) }}"
                                                    min="0" max="{{ $loan->totalbalance }}"
                                                    step="0.01"
                                                    class="pl-10 block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary transition">
                                            </div>
                                            <div class="flex justify-between mt-1">
                                                <p class="text-xs text-gray-500">Amount accepted as final settlement.</p>
                                                <p class="text-xs text-gray-500">Outstanding: K {{ number_format($loan->totalbalance, 2) }}</p>
                                            </div>
                                            <div id="writeoffInfo" class="mt-2 p-2 bg-gray-50 rounded border border-gray-200 hidden">
                                                <p class="text-xs">
                                                    <span class="font-medium">Amount to write off:</span>
                                                    <span id="writeoffAmount" class="text-red-600 font-bold ml-1"></span>
                                                </p>
                                            </div>
                                            @error('final_settlement_amount')
                                                <p class="mt-1 text-xs text-red-600">{{ $message }}</p>
                                            @enderror
                                        </div>
                                    @endif
                                    
                                    {{-- Confirmation Checkboxes --}}
                                    <div class="space-y-3 pt-4 border-t border-gray-200">
                                        <div class="flex items-start">
                                            <div class="flex items-center h-5">
                                                <input id="confirm_closure" name="confirm_closure" type="checkbox" required
                                                    class="w-4 h-4 text-zedbank-blue border-gray-300 rounded focus:ring-zedbank-blue">
                                            </div>
                                            <div class="ml-3 text-sm">
                                                <label for="confirm_closure" class="font-medium text-gray-700">
                                                    Confirm Loan Closure
                                                </label>
                                                <p class="text-gray-500">
                                                    I understand that closing this loan will change its status to "closed" and prevent further transactions.
                                                </p>
                                            </div>
                                        </div>
                                        
                                        <div class="flex items-start">
                                            <div class="flex items-center h-5">
                                                <input id="confirm_audit" name="confirm_audit" type="checkbox" required
                                                    class="w-4 h-4 text-zedbank-blue border-gray-300 rounded focus:ring-zedbank-blue">
                                            </div>
                                            <div class="ml-3 text-sm">
                                                <label for="confirm_audit" class="font-medium text-gray-700">
                                                    Confirm Audit Trail
                                                </label>
                                                <p class="text-gray-500">
                                                    I confirm that all necessary approvals and documentation are in place for audit purposes.
                                                </p>
                                            </div>
                                        </div>
                                        
                                        @if($loan->totalbalance > 0)
                                            <div class="flex items-start">
                                                <div class="flex items-center h-5">
                                                    <input id="confirm_writeoff" name="confirm_writeoff" type="checkbox" required
                                                        class="w-4 h-4 text-zedbank-blue border-gray-300 rounded focus:ring-zedbank-blue">
                                                </div>
                                                <div class="ml-3 text-sm">
                                                    <label for="confirm_writeoff" class="font-medium text-gray-700">
                                                        Confirm Write-off Authorization
                                                    </label>
                                                    <p class="text-gray-500">
                                                        I confirm that I have authorization to write off any outstanding balances.
                                                    </p>
                                                </div>
                                            </div>
                                        @endif
                                    </div>
                                    
                                    {{-- Form Actions --}}
                                    <div class="pt-6 border-t border-gray-200">
                                        <div class="flex justify-end space-x-3">
                                            <a href="{{ route('admin.loans.show', $loan->loanid) }}" 
                                               class="px-4 py-2 border border-gray-300 rounded-lg text-gray-700 hover:bg-gray-50 transition font-medium">
                                                Cancel
                                            </a>
                                            <button type="button" 
                                                    onclick="confirmClosure()"
                                                    class="px-4 py-2 bg-red-600 text-white rounded-lg hover:bg-red-700 transition font-medium flex items-center">
                                                <i class="fas fa-lock mr-2"></i>
                                                Close Loan
                                            </button>
                                        </div>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    {{-- Quick Statistics --}}
                    <div class="bg-white rounded-lg shadow-sm border border-gray-200 overflow-hidden">
                        <div class="border-b border-gray-200 px-6 py-4">
                            <h2 class="text-xl font-semibold text-gray-800">Quick Stats</h2>
                        </div>
                        <div class="p-6">
                            <dl class="grid grid-cols-1 gap-4">
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Original Amount</dt>
                                    <dd class="text-sm font-medium text-gray-900">K {{ number_format($loan->amount, 2) }}</dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Days Since Disbursement</dt>
                                    <dd class="text-sm font-medium text-gray-900">
                                        {{ $loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->diffInDays(now()) : 'N/A' }}
                                    </dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Maturity Date</dt>
                                    <dd class="text-sm font-medium text-gray-900">
                                        {{ $loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('M d, Y') : 'N/A' }}
                                    </dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">First Payment Date</dt>
                                    <dd class="text-sm font-medium text-gray-900">
                                        {{ $loan->firstpaymentdate ? \Carbon\Carbon::parse($loan->firstpaymentdate)->format('M d, Y') : 'N/A' }}
                                    </dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Installment Amount</dt>
                                    <dd class="text-sm font-medium text-gray-900">K {{ number_format($loan->installmentamount, 2) }}</dd>
                                </div>
                                <div class="flex justify-between items-center">
                                    <dt class="text-sm text-gray-600">Payment Frequency</dt>
                                    <dd class="text-sm font-medium text-gray-900">{{ ucfirst($loan->paymentfrequency) }}</dd>
                                </div>
                            </dl>
                        </div>
                    </div>
                </div>
            </div>
        </main>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Set max date for closure date to today
            const closureDateInput = document.getElementById('closure_date');
            if (closureDateInput) {
                const today = new Date().toISOString().split('T')[0];
                closureDateInput.max = today;
            }
            
            // Handle settlement amount and write-off calculation
            const settlementInput = document.getElementById('final_settlement_amount');
            const writeoffInfo = document.getElementById('writeoffInfo');
            const writeoffAmount = document.getElementById('writeoffAmount');
            
            if (settlementInput && writeoffInfo && writeoffAmount) {
                const outstandingBalance = parseFloat('{{ $loan->totalbalance }}');
                
                function updateWriteoffInfo() {
                    const settlementAmount = parseFloat(settlementInput.value) || 0;
                    const writeoff = outstandingBalance - settlementAmount;
                    
                    if (writeoff > 0) {
                        writeoffAmount.textContent = 'K ' + writeoff.toFixed(2);
                        writeoffInfo.classList.remove('hidden');
                    } else {
                        writeoffInfo.classList.add('hidden');
                    }
                }
                
                settlementInput.addEventListener('input', updateWriteoffInfo);
                updateWriteoffInfo(); // Initial calculation
            }
            
            // Show/hide settlement section based on closure reason
            const closingReasonSelect = document.getElementById('closing_reason');
            const settlementSection = document.getElementById('settlementSection');
            
            if (closingReasonSelect && settlementSection) {
                function toggleSettlementSection() {
                    const reason = closingReasonSelect.value;
                    const writeoffReasons = ['loan_forgiveness', 'write_off_approved', 'customer_bankruptcy', 'deceased_customer'];
                    
                    if (writeoffReasons.includes(reason)) {
                        settlementSection.classList.remove('hidden');
                    } else {
                        settlementSection.classList.add('hidden');
                    }
                }
                
                closingReasonSelect.addEventListener('change', toggleSettlementSection);
                toggleSettlementSection(); // Initial state
            }
        });
        
        function confirmClosure() {
            const loanNumber = '{{ $loan->loannumber }}';
            const outstandingBalance = parseFloat('{{ $loan->totalbalance }}');
            const closingReasonSelect = document.getElementById('closing_reason');
            const closingReason = closingReasonSelect ? closingReasonSelect.options[closingReasonSelect.selectedIndex].text : '';
            
            let message = `You are about to close loan ${loanNumber}.\n\n`;
            message += `Closure Details:\n`;
            message += `- Reason: ${closingReason}\n`;
            
            if (outstandingBalance > 0) {
                const settlementInput = document.getElementById('final_settlement_amount');
                const settlementAmount = settlementInput ? parseFloat(settlementInput.value) || 0 : 0;
                const writeoffAmount = outstandingBalance - settlementAmount;
                
                message += `- Outstanding Balance: K${outstandingBalance.toFixed(2)}\n`;
                message += `- Settlement Amount: K${settlementAmount.toFixed(2)}\n`;
                if (writeoffAmount > 0) {
                    message += `- Amount to Write-off: K${writeoffAmount.toFixed(2)}\n`;
                }
                message += `\n⚠️  WARNING: Closing with outstanding balances requires special authorization.\n\n`;
            } else {
                message += `- Outstanding Balance: K0.00 (Fully Settled)\n\n`;
                message += `✅  Loan is fully settled and can be safely closed.\n\n`;
            }
            
            message += `Are you sure you want to proceed with closing this loan?\n\n`;
            message += `This action cannot be undone.`;
            
            if (confirm(message)) {
                // Validate checkboxes
                const confirmClosureCheckbox = document.getElementById('confirm_closure');
                const confirmAuditCheckbox = document.getElementById('confirm_audit');
                const confirmWriteoffCheckbox = document.getElementById('confirm_writeoff');
                
                let missingConfirmations = [];
                
                if (confirmClosureCheckbox && !confirmClosureCheckbox.checked) {
                    missingConfirmations.push('Confirm Loan Closure');
                    confirmClosureCheckbox.focus();
                }
                
                if (confirmAuditCheckbox && !confirmAuditCheckbox.checked) {
                    missingConfirmations.push('Confirm Audit Trail');
                    if (missingConfirmations.length === 1) confirmAuditCheckbox.focus();
                }
                
                if (confirmWriteoffCheckbox && !confirmWriteoffCheckbox.checked && outstandingBalance > 0) {
                    missingConfirmations.push('Confirm Write-off Authorization');
                    if (missingConfirmations.length === 1) confirmWriteoffCheckbox.focus();
                }
                
                if (missingConfirmations.length > 0) {
                    alert('Please confirm the following:\n\n- ' + missingConfirmations.join('\n- '));
                    return;
                }
                
                document.getElementById('closeLoanForm').submit();
            }
        }
    </script>
</body>
</html>