@php
    // Define page variables
    $pageTitle = 'Classify as Non-Performing Loan';
    $pageSubtitle = 'Transfer loan to non-performing status';
@endphp

<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <meta name="csrf-token" content="{{ csrf_token() }}">
    <title>{{ $pageTitle }} | IMMIA FINANCE</title>
    
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <style>
        /* --- ZEDBANKOS THEME COLORS --- */
        :root {
            --zedbank-blue: #0077C5; /* ThemeBlue */
            --zedbank-blue-dark: #005A94; /* Darker shade for hover/active background */
            --npl-red: #DC2626; /* Red for NPL warnings */
            --npl-red-dark: #B91C1C;
        }

        /* Custom Tailwind-like classes */
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .bg-zedbank-blue-dark { background-color: var(--zedbank-blue-dark); }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .hover\:bg-zedbank-blue-dark:hover { background-color: var(--zedbank-blue-dark); }
        
        /* NPL Colors */
        .bg-npl-red { background-color: var(--npl-red); }
        .bg-npl-red-dark { background-color: var(--npl-red-dark); }
        .text-npl-red { color: var(--npl-red); }
        .hover\:bg-npl-red-dark:hover { background-color: var(--npl-red-dark); }
        .border-npl-red { border-color: var(--npl-red); }

        /* Sidebar/Main Content Layout */
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { 
            .lg\:ml-64 { margin-left: 0 !important; }
        }

        /* Local Overrides */
        .focus\:ring-primary:focus { --tw-ring-color: var(--zedbank-blue); }
        .focus\:border-primary:focus { border-color: var(--zedbank-blue); }
        .bg-primary { background-color: var(--zedbank-blue); }
        
        /* Custom color for warnings */
        .text-warning { color: #B45309; }
        .bg-warning-light { background-color: #FEF3C7; }
        
        /* Provision calculation styles */
        .provision-breakdown {
            border-left: 4px solid var(--npl-red);
            background-color: rgba(220, 38, 38, 0.05);
        }
        
        /* Loading spinner */
        .spinner {
            animation: spin 1s linear infinite;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    
    {{-- Assuming 'accounting.partials.sidebar' exists --}}
    @include('accounting.partials.sidebar')

    <div class="lg:ml-64 min-h-screen">
        
        {{-- Pass page variables to the header partial --}}
        @include('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ])

        <main class="p-6">
            <div class="mb-8">
                <div class="flex items-center mb-2">
                    <a href="{{ route('admin.loans.show', $loan) }}" class="text-zedbank-blue hover:text-zedbank-blue-dark mr-2">
                        <i class="fas fa-arrow-left"></i>
                    </a>
                    <h1 class="text-3xl font-bold text-gray-800">Classify as NPL: {{ $loan->loannumber }}</h1>
                </div>
                <p class="text-gray-600">Transfer this loan to non-performing status. This action should be taken when a loan shows signs of credit impairment.</p>
            </div>
            
            @if(session('success'))
                <div class="mb-6 bg-green-50 border border-green-200 text-green-700 px-4 py-3 rounded-lg">
                    {{ session('success') }}
                </div>
            @endif
            
            @if(session('error'))
                <div class="mb-6 bg-red-50 border border-red-200 text-red-700 px-4 py-3 rounded-lg">
                    {{ session('error') }}
                </div>
            @endif
            
            {{-- LOAN DETAILS CARD --}}
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Loan Information</h2>
                
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                    <div>
                        <p class="text-sm text-gray-600">Loan Number</p>
                        <p class="font-bold text-gray-800">{{ $loan->loannumber }}</p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Customer</p>
                        <p class="font-bold text-gray-800">
                            @if($loan->customer)
                                {{ $loan->customer->first_name }} {{ $loan->customer->surname }}
                                <span class="text-xs text-gray-500 block">{{ $loan->customer->customer_number }}</span>
                            @else
                                <span class="text-gray-500">—</span>
                            @endif
                        </p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Loan Product</p>
                        <p class="font-bold text-gray-800">
                            {{ $loan->loanType->product ?? 'N/A' }}
                        </p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Disbursement Date</p>
                        <p class="font-bold text-gray-800">
                            {{ $loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('M d, Y') : 'N/A' }}
                        </p>
                    </div>
                </div>
                
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4">
                    <div>
                        <p class="text-sm text-gray-600">Original Amount</p>
                        <p class="font-bold text-gray-800">{{ number_format($loan->amount, 2) }}</p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Days Delinquent</p>
                        <p class="font-bold {{ $loan->daysdelinquent > 90 ? 'text-npl-red' : ($loan->daysdelinquent > 30 ? 'text-orange-600' : 'text-gray-800') }}">
                            {{ $loan->daysdelinquent }} days
                        </p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Maturity Date</p>
                        <p class="font-bold text-gray-800">
                            {{ $loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('M d, Y') : 'N/A' }}
                        </p>
                    </div>
                    
                    <div>
                        <p class="text-sm text-gray-600">Current Status</p>
                        <span class="inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium 
                            {{ $loan->status === 'active' ? 'bg-green-100 text-green-800' : 
                               ($loan->status === 'closed' ? 'bg-gray-100 text-gray-800' : 
                               ($loan->status === 'default' ? 'bg-npl-red text-white' : 'bg-blue-100 text-blue-800')) }}">
                            {{ ucfirst($loan->status) }}
                        </span>
                    </div>
                </div>
            </div>
            
            {{-- BALANCE SUMMARY CARD --}}
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6 mb-6">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Outstanding Balances</h2>
                
                <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-4 mb-6">
                    <div class="{{ $loan->principalbalance > 0 ? 'bg-red-50 border border-red-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                        <p class="text-sm text-gray-600">Principal Balance</p>
                        <p class="text-2xl font-bold {{ $loan->principalbalance > 0 ? 'text-red-600' : 'text-green-600' }}">
                            {{ number_format($loan->principalbalance, 2) }}
                        </p>
                        <p class="text-xs text-gray-500 mt-1">{{ number_format(($loan->principalbalance / $loan->amount) * 100, 1) }}% of original</p>
                    </div>
                    
                    <div class="{{ $loan->interestbalance > 0 ? 'bg-orange-50 border border-orange-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                        <p class="text-sm text-gray-600">Interest Balance</p>
                        <p class="text-2xl font-bold {{ $loan->interestbalance > 0 ? 'text-orange-600' : 'text-green-600' }}">
                            {{ number_format($loan->interestbalance, 2) }}
                        </p>
                        @if($loan->interestbalance > 0)
                            <p class="text-xs text-orange-500 mt-1">Accrued interest</p>
                        @else
                            <p class="text-xs text-green-500 mt-1">No interest due</p>
                        @endif
                    </div>
                    
                    <div class="{{ $loan->processing_fee_balance > 0 ? 'bg-purple-50 border border-purple-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                        <p class="text-sm text-gray-600">Processing Fees</p>
                        <p class="text-2xl font-bold {{ $loan->processing_fee_balance > 0 ? 'text-purple-600' : 'text-green-600' }}">
                            {{ number_format($loan->processing_fee_balance, 2) }}
                        </p>
                        @if($loan->processing_fee_balance > 0)
                            <p class="text-xs text-purple-500 mt-1">Outstanding fees</p>
                        @else
                            <p class="text-xs text-green-500 mt-1">No fees due</p>
                        @endif
                    </div>
                    
                    <div class="{{ $loan->totalbalance > 0 ? 'bg-blue-50 border border-blue-100' : 'bg-green-50 border border-green-100' }} p-4 rounded-lg">
                        <p class="text-sm text-gray-600">Total Outstanding</p>
                        <p class="text-2xl font-bold {{ $loan->totalbalance > 0 ? 'text-blue-600' : 'text-green-600' }}">
                            {{ number_format($loan->totalbalance, 2) }}
                        </p>
                        @if($loan->totalbalance > 0)
                            <p class="text-xs text-blue-500 mt-1">Total amount due</p>
                        @else
                            <p class="text-xs text-green-500 mt-1">Fully settled</p>
                        @endif
                    </div>
                </div>
                
                {{-- PROVISION CALCULATION PREVIEW --}}
                <div class="provision-breakdown p-4 rounded-lg mb-4">
                    <div class="flex items-center mb-3">
                        <i class="fas fa-calculator text-npl-red text-xl mr-3"></i>
                        <h3 class="text-lg font-semibold text-npl-red">Provision Calculation Preview</h3>
                    </div>
                    
                    <div class="grid grid-cols-1 md:grid-cols-3 gap-4 mb-3">
                        <div class="text-center">
                            <p class="text-sm text-gray-600">Substandard (20%)</p>
                            <p class="text-xl font-bold text-npl-red">K {{ number_format($loan->totalbalance * 0.2, 2) }}</p>
                        </div>
                        <div class="text-center">
                            <p class="text-sm text-gray-600">Doubtful (50%)</p>
                            <p class="text-xl font-bold text-npl-red">K {{ number_format($loan->totalbalance * 0.5, 2) }}</p>
                        </div>
                        <div class="text-center">
                            <p class="text-sm text-gray-600">Loss (100%)</p>
                            <p class="text-xl font-bold text-npl-red">K {{ number_format($loan->totalbalance * 1.0, 2) }}</p>
                        </div>
                    </div>
                    
                    <div class="text-sm text-gray-600">
                        <p><i class="fas fa-info-circle mr-2"></i>Provision will be created as: <strong>Dr. Loan Loss Expense / Cr. Allowance for Loan Losses</strong></p>
                    </div>
                </div>
                
                {{-- NPL CLASSIFICATION CRITERIA --}}
                <div class="bg-npl-red bg-opacity-10 border border-npl-red rounded-lg p-4 mb-4">
                    <div class="flex items-start">
                        <div class="flex-shrink-0">
                            <i class="fas fa-exclamation-triangle text-npl-red text-xl"></i>
                        </div>
                        <div class="ml-3">
                            <h3 class="text-lg font-semibold text-npl-red mb-1">NPL Classification Criteria</h3>
                            <p class="text-npl-red">
                                A loan should be classified as non-performing when:
                            </p>
                            <ul class="list-disc pl-5 mt-2 text-npl-red space-y-1">
                                <li>Payments are more than 90 days overdue</li>
                                <li>There is evidence of credit impairment</li>
                                <li>The borrower is experiencing financial difficulties</li>
                                <li>There is a breach of loan covenants</li>
                                <li>The loan is unlikely to be repaid in full</li>
                            </ul>
                        </div>
                    </div>
                </div>
                
                {{-- PAYMENT HISTORY SUMMARY --}}
                <div class="mt-6">
                    <h3 class="text-lg font-semibold text-gray-800 mb-3">Payment History</h3>
                    <div class="grid grid-cols-1 md:grid-cols-4 gap-4">
                        <div class="bg-gray-50 p-3 rounded-lg">
                            <p class="text-sm text-gray-600">Principal Paid</p>
                            <p class="text-lg font-bold text-gray-800">{{ number_format($loan->totalprincipalpaid, 2) }}</p>
                        </div>
                        
                        <div class="bg-gray-50 p-3 rounded-lg">
                            <p class="text-sm text-gray-600">Interest Paid</p>
                            <p class="text-lg font-bold text-gray-800">{{ number_format($loan->totalinterestpaid, 2) }}</p>
                        </div>
                        
                        <div class="bg-gray-50 p-3 rounded-lg">
                            <p class="text-sm text-gray-600">Total Paid</p>
                            <p class="text-lg font-bold text-gray-800">{{ number_format($loan->totalpaid, 2) }}</p>
                        </div>
                        
                        <div class="bg-gray-50 p-3 rounded-lg">
                            <p class="text-sm text-gray-600">Last Payment</p>
                            <p class="text-lg font-bold text-gray-800">
                                {{ $loan->lastpaymentdate ? \Carbon\Carbon::parse($loan->lastpaymentdate)->format('M d, Y') : 'Never' }}
                            </p>
                        </div>
                    </div>
                </div>
            </div>
            
            {{-- NPL CLASSIFICATION FORM --}}
            <div class="bg-white rounded-lg shadow-sm border border-gray-200 p-6">
                <h2 class="text-xl font-semibold text-gray-800 mb-4">Non-Performing Loan Classification</h2>
                
                <!-- DEBUG INFO -->
                <div style="display: none; background: #f0f0f0; padding: 10px; margin-bottom: 10px; border: 1px solid #ccc;">
                    <strong>Debug Information:</strong><br>
                    Route: {{ route('admin.loans.transferToNPL', $loan) }}<br>
                    Loan ID: {{ $loan->loanid }}<br>
                    Loan Number: {{ $loan->loannumber }}<br>
                    CSRF Token Exists: {{ csrf_token() ? 'Yes' : 'No' }}
                </div>
                
                <form action="{{ route('admin.loans.transferToNPL', $loan) }}" method="POST" id="nplForm" novalidate>
                    @csrf
                    
                    <div class="space-y-6">
                        {{-- NPL REASON --}}
                        <div>
                            <label for="npl_reason" class="block text-sm font-medium text-gray-700 mb-2">
                                NPL Classification Reason <span class="text-red-500">*</span>
                            </label>
                            <select id="npl_reason" name="npl_reason" required
                                class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">
                                <option value="">Select classification reason...</option>
                                <option value="payment_default">Payment Default (90+ days overdue)</option>
                                <option value="financial_difficulties">Borrower Financial Difficulties</option>
                                <option value="bankruptcy">Borrower Bankruptcy/Insolvency</option>
                                <option value="breach_of_covenants">Breach of Loan Covenants</option>
                                <option value="significant_doubt">Significant Doubt About Repayment</option>
                                <option value="legal_issues">Legal/Regulatory Issues</option>
                                <option value="death_disability">Death or Permanent Disability</option>
                                <option value="other">Other (Specify in Details)</option>
                            </select>
                            <p class="mt-1 text-sm text-gray-500">Select the primary reason for NPL classification.</p>
                            @error('npl_reason')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- CLASSIFICATION DATE --}}
                        <div>
                            <label for="classification_date" class="block text-sm font-medium text-gray-700 mb-2">
                                Classification Date <span class="text-red-500">*</span>
                            </label>
                            <input type="date" id="classification_date" name="classification_date" required
                                value="{{ old('classification_date', date('Y-m-d')) }}"
                                class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">
                            <p class="mt-1 text-sm text-gray-500">Date when the loan should be classified as non-performing.</p>
                            @error('classification_date')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- NPL CATEGORY --}}
                        <div>
                            <label for="npl_category" class="block text-sm font-medium text-gray-700 mb-2">
                                NPL Category <span class="text-red-500">*</span>
                            </label>
                            <select id="npl_category" name="npl_category" required
                                class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">
                                <option value="">Select NPL category...</option>
                                <option value="substandard">Substandard (90-180 days overdue)</option>
                                <option value="doubtful">Doubtful (181-365 days overdue)</option>
                                <option value="loss">Loss (365+ days overdue)</option>
                                <option value="watch">Watch List (Early warning)</option>
                            </select>
                            <p class="mt-1 text-sm text-gray-500">Select the appropriate NPL category based on days overdue and risk assessment.</p>
                            @error('npl_category')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- PROVISIONING PERCENTAGE --}}
                        <div>
                            <label for="provision_percentage" class="block text-sm font-medium text-gray-700 mb-2">
                                Provisioning Percentage <span class="text-red-500">*</span>
                            </label>
                            <div class="relative">
                                <input type="number" id="provision_percentage" name="provision_percentage" required
                                    value="{{ old('provision_percentage', 20) }}"
                                    min="1" max="100" step="0.01"
                                    class="block w-full pl-12 pr-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">
                                <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                    <span class="text-gray-500">%</span>
                                </div>
                            </div>
                            <div class="flex justify-between mt-1">
                                <p class="text-sm text-gray-500">Recommended provisioning for this NPL category.</p>
                                <p class="text-sm font-semibold text-npl-red">Provision Amount: <span id="provisionAmount">K {{ number_format($loan->totalbalance * 0.2, 2) }}</span></p>
                            </div>
                            
                            {{-- PROVISION BREAKDOWN --}}
                            <div id="provisionBreakdown" class="mt-3 p-3 bg-gray-50 rounded-lg border border-gray-200">
                                <h4 class="text-sm font-semibold text-gray-700 mb-2">Provision Breakdown:</h4>
                                <div class="grid grid-cols-3 gap-2 text-sm">
                                    <div>
                                        <p class="text-gray-600">Principal:</p>
                                        <p class="font-semibold" id="provisionPrincipal">K {{ number_format($loan->principalbalance * 0.2, 2) }}</p>
                                    </div>
                                    <div>
                                        <p class="text-gray-600">Interest:</p>
                                        <p class="font-semibold" id="provisionInterest">K {{ number_format($loan->interestbalance * 0.2, 2) }}</p>
                                    </div>
                                    <div>
                                        <p class="text-gray-600">Fees:</p>
                                        <p class="font-semibold" id="provisionFees">K {{ number_format($loan->processing_fee_balance * 0.2, 2) }}</p>
                                    </div>
                                </div>
                            </div>
                            
                            @error('provision_percentage')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- DETAILED ASSESSMENT --}}
                        <div>
                            <label for="assessment_details" class="block text-sm font-medium text-gray-700 mb-2">
                                Detailed Assessment <span class="text-red-500">*</span>
                            </label>
                            <textarea id="assessment_details" name="assessment_details" rows="4" required
                                placeholder="Provide detailed assessment of why this loan is being classified as NPL. Include: 1) Payment history, 2) Communication with borrower, 3) Financial analysis, 4) Recovery prospects..."
                                class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">{{ old('assessment_details') }}</textarea>
                            <div class="flex justify-between mt-1">
                                <p class="text-sm text-gray-500">This detailed assessment will be saved with the loan record for audit and regulatory purposes.</p>
                                <p class="text-sm text-gray-500" id="charCount">Minimum 50 characters required</p>
                            </div>
                            @error('assessment_details')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- RECOVERY ACTION PLAN --}}
                        <div>
                            <label for="recovery_plan" class="block text-sm font-medium text-gray-700 mb-2">
                                Recovery Action Plan
                            </label>
                            <textarea id="recovery_plan" name="recovery_plan" rows="3"
                                placeholder="Outline the recovery action plan. Include: 1) Next steps, 2) Responsible party, 3) Timeline, 4) Expected outcomes..."
                                class="block w-full px-4 py-2 border border-gray-300 rounded-lg focus:ring-primary focus:border-primary">{{ old('recovery_plan') }}</textarea>
                            <p class="mt-1 text-sm text-gray-500">Optional: Document the planned recovery actions for this NPL.</p>
                            @error('recovery_plan')
                                <p class="mt-1 text-sm text-red-600">{{ $message }}</p>
                            @enderror
                        </div>
                        
                        {{-- SUPPORTING DOCUMENTS --}}
                        <div>
                            <label class="block text-sm font-medium text-gray-700 mb-2">
                                Supporting Documents
                            </label>
                            <div class="space-y-2">
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="supporting_docs[]" value="payment_history"
                                        class="rounded border-gray-300 text-zedbank-blue focus:ring-zedbank-blue">
                                    <span class="ml-2 text-gray-700">Payment History Report</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="supporting_docs[]" value="communication_log"
                                        class="rounded border-gray-300 text-zedbank-blue focus:ring-zedbank-blue">
                                    <span class="ml-2 text-gray-700">Communication Log</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="supporting_docs[]" value="financial_analysis"
                                        class="rounded border-gray-300 text-zedbank-blue focus:ring-zedbank-blue">
                                    <span class="ml-2 text-gray-700">Financial Analysis</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="supporting_docs[]" value="credit_report"
                                        class="rounded border-gray-300 text-zedbank-blue focus:ring-zedbank-blue">
                                    <span class="ml-2 text-gray-700">Credit Bureau Report</span>
                                </label>
                                <label class="inline-flex items-center">
                                    <input type="checkbox" name="supporting_docs[]" value="legal_documentation"
                                        class="rounded border-gray-300 text-zedbank-blue focus:ring-zedbank-blue">
                                    <span class="ml-2 text-gray-700">Legal Documentation</span>
                                </label>
                            </div>
                            <p class="mt-1 text-sm text-gray-500">Check all supporting documents that are available for this NPL classification.</p>
                        </div>
                        
                        {{-- ACCOUNTING IMPACT SUMMARY --}}
                        <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4">
                            <div class="flex items-start">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-book text-yellow-600 text-xl"></i>
                                </div>
                                <div class="ml-3">
                                    <h3 class="text-lg font-semibold text-yellow-800 mb-2">Accounting Impact Summary</h3>
                                    <div class="space-y-2 text-yellow-700">
                                        <div class="flex justify-between items-center">
                                            <span>Loan Loss Expense:</span>
                                            <span class="font-bold" id="accountingExpense">K {{ number_format($loan->totalbalance * 0.2, 2) }}</span>
                                        </div>
                                        <div class="flex justify-between items-center">
                                            <span>Allowance for Loan Losses:</span>
                                            <span class="font-bold" id="accountingAllowance">K {{ number_format($loan->totalbalance * 0.2, 2) }}</span>
                                        </div>
                                        <div class="pt-2 border-t border-yellow-300">
                                            <p class="text-sm"><strong>Journal Entry:</strong> Dr. Loan Loss Expense / Cr. Allowance for Loan Losses</p>
                                            <p class="text-sm mt-1"><i class="fas fa-info-circle"></i> This provision will be reflected in the financial statements.</p>
                                        </div>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        {{-- SERIOUS WARNING --}}
                        <div class="bg-npl-red border border-npl-red rounded-lg p-4">
                            <div class="flex items-start">
                                <div class="flex-shrink-0">
                                    <i class="fas fa-exclamation-circle text-white bg-npl-red rounded-full p-1 text-xl"></i>
                                </div>
                                <div class="ml-3">
                                    <h3 class="text-lg font-semibold text-white mb-1">Important Regulatory Warning</h3>
                                    <p class="text-white opacity-90">
                                        Classifying a loan as non-performing has significant regulatory and financial implications:
                                    </p>
                                    <ul class="list-disc pl-5 mt-2 text-white opacity-90 space-y-1">
                                        <li>Requires additional capital provisions</li>
                                        <li>Affects the institution's financial ratios</li>
                                        <li>May trigger regulatory reporting requirements</li>
                                        <li>Impacts the borrower's credit rating</li>
                                        <li>May require Board/Committee approval</li>
                                    </ul>
                                    <div class="mt-4 space-y-2">
                                        <label class="inline-flex items-center">
                                            <input type="checkbox" id="confirm_regulatory" name="confirm_regulatory" 
                                                class="rounded border-gray-300 text-white focus:ring-white bg-transparent"
                                                required>
                                            <span class="ml-2 text-white">
                                                I confirm that this NPL classification complies with regulatory requirements
                                            </span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="checkbox" id="confirm_approval" name="confirm_approval" 
                                                class="rounded border-gray-300 text-white focus:ring-white bg-transparent"
                                                required>
                                            <span class="ml-2 text-white">
                                                I confirm that I have obtained necessary approvals for this classification
                                            </span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="checkbox" id="confirm_impact" name="confirm_impact" 
                                                class="rounded border-gray-300 text-white focus:ring-white bg-transparent"
                                                required>
                                            <span class="ml-2 text-white">
                                                I understand the financial and regulatory impact of this classification
                                            </span>
                                        </label>
                                        <label class="inline-flex items-center">
                                            <input type="checkbox" id="confirm_accounting" name="confirm_accounting" 
                                                class="rounded border-gray-300 text-white focus:ring-white bg-transparent"
                                                required>
                                            <span class="ml-2 text-white">
                                                I understand that accounting entries will be created for loan loss provisions
                                            </span>
                                        </label>
                                    </div>
                                </div>
                            </div>
                        </div>
                        
                        {{-- ACTION BUTTONS --}}
                        <div class="flex justify-between pt-6 border-t border-gray-200">
                            <a href="{{ route('admin.loans.show', $loan) }}" 
                               class="px-6 py-3 bg-white border border-gray-300 text-gray-700 rounded-lg hover:bg-gray-50 transition font-medium">
                                <i class="fas fa-times mr-2"></i> Cancel
                            </a>
                            
                            <button type="button" 
                                    id="nplSubmitBtn"
                                    onclick="confirmNPLClassification()"
                                    class="px-6 py-3 bg-npl-red text-white rounded-lg hover:bg-npl-red-dark transition font-medium flex items-center">
                                <i class="fas fa-exclamation-triangle mr-2"></i> Classify as NPL & Create Provision
                            </button>
                        </div>
                        
                        <!-- TEMPORARY TEST BUTTON -->
                        <div class="mt-4 p-4 bg-yellow-50 border border-yellow-200 rounded-lg">
                            <p class="text-sm text-yellow-800 mb-2">Debug: Test form submission</p>
                            <button type="button" 
                                    onclick="testFormSubmission()"
                                    class="px-4 py-2 bg-yellow-500 text-white rounded hover:bg-yellow-600 text-sm">
                                Test Form Submission
                            </button>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', function() {
            // Get loan balance data
            const principalBalance = parseFloat('{{ $loan->principalbalance }}');
            const interestBalance = parseFloat('{{ $loan->interestbalance }}');
            const feeBalance = parseFloat('{{ $loan->processing_fee_balance }}');
            const totalBalance = parseFloat('{{ $loan->totalbalance }}');
            const daysDelinquent = parseInt('{{ $loan->daysdelinquent }}');
            
            // DOM elements
            const provisionInput = document.getElementById('provision_percentage');
            const provisionAmount = document.getElementById('provisionAmount');
            const provisionPrincipal = document.getElementById('provisionPrincipal');
            const provisionInterest = document.getElementById('provisionInterest');
            const provisionFees = document.getElementById('provisionFees');
            const accountingExpense = document.getElementById('accountingExpense');
            const accountingAllowance = document.getElementById('accountingAllowance');
            const categorySelect = document.getElementById('npl_category');
            const reasonSelect = document.getElementById('npl_reason');
            const assessmentTextarea = document.getElementById('assessment_details');
            const charCount = document.getElementById('charCount');
            
            // Update all provision calculations
            function updateProvisionCalculations() {
                const percentage = parseFloat(provisionInput.value) || 0;
                
                // Calculate amounts
                const totalProvision = totalBalance * (percentage / 100);
                const principalProvision = principalBalance * (percentage / 100);
                const interestProvision = interestBalance * (percentage / 100);
                const feeProvision = feeBalance * (percentage / 100);
                
                // Update display
                provisionAmount.textContent = 'K ' + totalProvision.toFixed(2);
                provisionPrincipal.textContent = 'K ' + principalProvision.toFixed(2);
                provisionInterest.textContent = 'K ' + interestProvision.toFixed(2);
                provisionFees.textContent = 'K ' + feeProvision.toFixed(2);
                accountingExpense.textContent = 'K ' + totalProvision.toFixed(2);
                accountingAllowance.textContent = 'K ' + totalProvision.toFixed(2);
            }
            
            // Update provision percentage based on category
            function updateProvisionByCategory() {
                const category = categorySelect.value;
                let percentage = 20; // Default
                
                switch(category) {
                    case 'substandard':
                        percentage = 20;
                        break;
                    case 'doubtful':
                        percentage = 50;
                        break;
                    case 'loss':
                        percentage = 100;
                        break;
                    case 'watch':
                        percentage = 10;
                        break;
                }
                
                provisionInput.value = percentage;
                updateProvisionCalculations();
            }
            
            // Set default NPL category based on days delinquent
            if (daysDelinquent > 0) {
                if (daysDelinquent >= 365) {
                    categorySelect.value = 'loss';
                } else if (daysDelinquent >= 181) {
                    categorySelect.value = 'doubtful';
                } else if (daysDelinquent >= 90) {
                    categorySelect.value = 'substandard';
                } else {
                    categorySelect.value = 'watch';
                }
                updateProvisionByCategory();
            }
            
            // Auto-populate NPL reason based on days delinquent
            if (daysDelinquent >= 90) {
                reasonSelect.value = 'payment_default';
            }
            
            // Update character count for assessment
            function updateCharCount() {
                const text = assessmentTextarea.value;
                const length = text.trim().length;
                const minChars = 50;
                
                if (length >= minChars) {
                    charCount.textContent = length + ' characters ✓';
                    charCount.className = 'text-sm text-green-600';
                } else {
                    charCount.textContent = length + ' characters (minimum ' + minChars + ' required)';
                    charCount.className = 'text-sm text-red-600';
                }
            }
            
            // Auto-populate assessment details based on selected options
            function updateAssessmentDetails() {
                const reason = reasonSelect ? reasonSelect.value : '';
                const category = categorySelect ? categorySelect.value : '';
                const percentage = provisionInput ? parseFloat(provisionInput.value) || 0 : 0;
                
                if (reason && category && assessmentTextarea && !assessmentTextarea.value) {
                    let assessment = `NPL CLASSIFICATION ASSESSMENT\n`;
                    assessment += `================================\n\n`;
                    assessment += `1. LOAN DETAILS:\n`;
                    assessment += `   - Loan Number: {{ $loan->loannumber }}\n`;
                    assessment += `   - Customer: {{ $loan->customer->first_name ?? '' }} {{ $loan->customer->surname ?? '' }}\n`;
                    assessment += `   - Original Amount: K {{ number_format($loan->amount, 2) }}\n`;
                    assessment += `   - Outstanding Balance: K ${totalBalance.toFixed(2)}\n`;
                    assessment += `   - Days Delinquent: ${daysDelinquent} days\n\n`;
                    
                    assessment += `2. CLASSIFICATION DETAILS:\n`;
                    assessment += `   - Classification Reason: ${reasonSelect.options[reasonSelect.selectedIndex].text}\n`;
                    assessment += `   - NPL Category: ${categorySelect.options[categorySelect.selectedIndex].text}\n`;
                    assessment += `   - Provision Percentage: ${percentage}%\n`;
                    assessment += `   - Provision Amount: K ${(totalBalance * percentage / 100).toFixed(2)}\n\n`;
                    
                    assessment += `3. BALANCE BREAKDOWN:\n`;
                    assessment += `   - Principal Balance: K ${principalBalance.toFixed(2)}\n`;
                    assessment += `   - Interest Balance: K ${interestBalance.toFixed(2)}\n`;
                    assessment += `   - Processing Fee Balance: K ${feeBalance.toFixed(2)}\n`;
                    assessment += `   - Total Outstanding: K ${totalBalance.toFixed(2)}\n\n`;
                    
                    assessment += `4. ASSESSMENT:\n`;
                    assessment += `   - Payment History: [Describe payment pattern]\n`;
                    assessment += `   - Communication: [Detail communication attempts]\n`;
                    assessment += `   - Financial Analysis: [Analyze borrower's financial situation]\n`;
                    assessment += `   - Recovery Prospects: [Assess likelihood of recovery]\n`;
                    assessment += `   - Supporting Evidence: [List supporting documents]\n\n`;
                    
                    assessment += `5. RECOMMENDATION:\n`;
                    assessment += `   Based on the above assessment, this loan should be classified as non-performing under the ${categorySelect.options[categorySelect.selectedIndex].text} category with a ${percentage}% provision.`;
                    
                    assessmentTextarea.value = assessment;
                    updateCharCount();
                }
            }
            
            // Event listeners
            provisionInput.addEventListener('input', updateProvisionCalculations);
            categorySelect.addEventListener('change', function() {
                updateProvisionByCategory();
                updateAssessmentDetails();
            });
            reasonSelect.addEventListener('change', updateAssessmentDetails);
            provisionInput.addEventListener('change', updateAssessmentDetails);
            assessmentTextarea.addEventListener('input', updateCharCount);
            
            // Set max date for classification date to today
            const classificationDateInput = document.getElementById('classification_date');
            if (classificationDateInput) {
                const today = new Date().toISOString().split('T')[0];
                classificationDateInput.max = today;
            }
            
            // Initial calculations
            updateProvisionCalculations();
            updateAssessmentDetails();
            updateCharCount();
        });
        
        // Main validation and submission function
        function confirmNPLClassification() {
            console.log('=== NPL Classification Started ===');
            
            try {
                // 1. Get all required elements
                const form = document.getElementById('nplForm');
                if (!form) {
                    console.error('Form not found!');
                    alert('Form not found. Please refresh the page.');
                    return false;
                }
                
                console.log('1. Form found');
                console.log('Form action:', form.action);
                
                // 2. Simple validation - with detailed logging
                console.log('2. Starting validation...');
                
                const requiredFields = [
                    { id: 'npl_reason', name: 'NPL Reason', type: 'select' },
                    { id: 'npl_category', name: 'NPL Category', type: 'select' },
                    { id: 'provision_percentage', name: 'Provision Percentage', type: 'number' },
                    { id: 'assessment_details', name: 'Assessment Details', type: 'textarea' }
                ];
                
                for (const field of requiredFields) {
                    console.log(`Checking field: ${field.id} (${field.name})`);
                    const element = document.getElementById(field.id);
                    
                    if (!element) {
                        console.error(`Field not found: ${field.id}`);
                        alert(`Field ${field.name} not found. Please refresh the page.`);
                        return false;
                    }
                    
                    console.log(`Field value: "${element.value}"`);
                    
                    if (field.type === 'textarea') {
                        if (!element.value || element.value.trim().length < 50) {
                            console.log(`Field validation failed: ${field.name} - text too short`);
                            alert(`Please provide a detailed ${field.name} (minimum 50 characters).`);
                            element.focus();
                            return false;
                        }
                    } else if (field.type === 'number') {
                        const value = parseFloat(element.value);
                        console.log(`Parsed number value: ${value}`);
                        if (isNaN(value) || value <= 0 || value > 100) {
                            console.log(`Field validation failed: ${field.name} - invalid number`);
                            alert(`Please enter a valid ${field.name} between 1 and 100.`);
                            element.focus();
                            return false;
                        }
                    } else if (field.type === 'select') {
                        if (!element.value) {
                            console.log(`Field validation failed: ${field.name} - no selection`);
                            alert(`Please select ${field.name}.`);
                            element.focus();
                            return false;
                        }
                    }
                    
                    console.log(`Field ${field.id} passed validation`);
                }
                
                console.log('3. All field validations passed');
                
                // 3. Check confirmations
                console.log('4. Checking confirmations...');
                
                const confirmations = [
                    { id: 'confirm_regulatory', message: 'Please confirm regulatory compliance.' },
                    { id: 'confirm_approval', message: 'Please confirm necessary approvals.' },
                    { id: 'confirm_impact', message: 'Please confirm understanding of financial impact.' },
                    { id: 'confirm_accounting', message: 'Please confirm understanding of accounting impact.' }
                ];
                
                for (const confirmation of confirmations) {
                    console.log(`Checking confirmation: ${confirmation.id}`);
                    const element = document.getElementById(confirmation.id);
                    console.log(`Checkbox found: ${!!element}, Checked: ${element ? element.checked : 'N/A'}`);
                    
                    if (element && !element.checked) {
                        console.log(`Confirmation failed: ${confirmation.id}`);
                        alert(confirmation.message);
                        element.focus();
                        return false;
                    }
                    console.log(`Confirmation ${confirmation.id} passed`);
                }
                
                console.log('5. All confirmations passed');
                
                // 4. Simple confirmation
                const loanNumber = '{{ $loan->loannumber }}';
                const customerName = '{{ $loan->customer->first_name ?? "" }} {{ $loan->customer->surname ?? "" }}';
                
                const confirmMsg = `Are you sure you want to classify loan ${loanNumber} (${customerName}) as Non-Performing?\n\nThis action cannot be undone and will create accounting entries.`;
                
                console.log('6. Showing confirmation dialog...');
                
                if (!confirm(confirmMsg)) {
                    console.log('7. User cancelled');
                    return false;
                }
                
                console.log('8. User confirmed, preparing to submit...');
                
                // 5. Create hidden confirmation field
                const confirmField = document.createElement('input');
                confirmField.type = 'hidden';
                confirmField.name = 'confirmed_submission';
                confirmField.value = 'true';
                form.appendChild(confirmField);
                
                console.log('9. Hidden field added');
                
                // 6. Show loading state
                const submitBtn = document.getElementById('nplSubmitBtn');
                if (submitBtn) {
                    console.log('10. Button found, setting loading state');
                    const originalText = submitBtn.innerHTML;
                    submitBtn.innerHTML = '<i class="fas fa-spinner fa-spin mr-2"></i> Processing...';
                    submitBtn.disabled = true;
                    
                    // Restore button after 10 seconds (in case submission fails)
                    setTimeout(() => {
                        submitBtn.innerHTML = originalText;
                        submitBtn.disabled = false;
                        console.log('Button state restored (timeout)');
                    }, 10000);
                } else {
                    console.log('10. Button not found');
                }
                
                // 7. Submit the form
                console.log('11. Form submitting to:', form.action);
                
                // Force form submission
                setTimeout(() => {
                    console.log('12. Actually submitting form now...');
                    form.submit();
                }, 100);
                
                console.log('13. Form submission initiated');
                
                return true;
                
            } catch (error) {
                console.error('Error in confirmNPLClassification:', error);
                alert('An error occurred: ' + error.message);
                return false;
            }
        }
        
        // Test function for debugging
        function testFormSubmission() {
            console.log('=== Test Form Submission ===');
            const form = document.getElementById('nplForm');
            
            if (!form) {
                alert('Form not found!');
                return;
            }
            
            // Fill in required fields for testing
            document.getElementById('npl_reason').value = 'payment_default';
            document.getElementById('npl_category').value = 'substandard';
            document.getElementById('provision_percentage').value = '20';
            document.getElementById('assessment_details').value = 'Test assessment details. This is a test to check if the form can submit properly. Testing form submission functionality. Minimum 50 characters requirement met.';
            document.getElementById('classification_date').value = '{{ date("Y-m-d") }}';
            
            // Check all confirmations
            document.getElementById('confirm_regulatory').checked = true;
            document.getElementById('confirm_approval').checked = true;
            document.getElementById('confirm_impact').checked = true;
            document.getElementById('confirm_accounting').checked = true;
            
            console.log('Form filled with test data');
            console.log('Form action:', form.action);
            console.log('CSRF token exists:', document.querySelector('input[name="_token"]') !== null);
            
            // Try to submit
            if (confirm('Test: Submit form now?')) {
                console.log('Submitting test form...');
                form.submit();
            }
        }
        
        // Add global error handler
        window.addEventListener('error', function(event) {
            console.error('Global JavaScript Error:', event.error);
            console.error('Error message:', event.message);
            console.error('Error at:', event.filename, 'line', event.lineno);
        });
        
        // Also catch promise rejections
        window.addEventListener('unhandledrejection', function(event) {
            console.error('Unhandled Promise Rejection:', event.reason);
        });
        
        // Test that the function is accessible globally
        console.log('confirmNPLClassification function available:', typeof confirmNPLClassification === 'function');
        console.log('testFormSubmission function available:', typeof testFormSubmission === 'function');
        
    </script>
</body>
</html>