<?php
    $pageTitle = 'Record Payment';
    $pageSubtitle = 'Processing Payment for Loan: ' . $loan->loannumber;
    
    // Calculate available balances
    $principalBalance = $loan->principalbalance ?? 0;
    $interestBalance = $loan->interestbalance ?? 0;
    $processingBalance = $loan->processing_fee_balance ?? 0;
    $penaltyBalance = $loan->penalty_balance ?? 0;
    $totalBalance = $principalBalance + $interestBalance + $processingBalance + $penaltyBalance;
    
    // Get customer info
    $customer = $loan->customer;
    $customerName = $customer ? $customer->first_name . ' ' . $customer->surname : 'Unknown Customer';
    $customerNumber = $customer ? $customer->customer_number : 'N/A';
    
    // Payment methods (matching your pmethod field)
    $paymentMethods = [
        'Cash' => 'Cash',
        'Bank' => 'Bank Transfer',
        'Mobile Money' => 'Mobile Money',
        'Cheque' => 'Cheque',
        'Card' => 'Card Payment',
        'Other' => 'Other'
    ];
    
    // Get cash/bank accounts for paccount field
    $cashBankAccounts = \App\Models\CashBankAccount::where('isActive', 1)->get();
    
    // Get payment schedule info
    $nextPayment = $loan->paymentSchedules()
        ->where('status', 'scheduled')
        ->orderBy('paymentdate', 'asc')
        ->first();
        
    // Calculate loan tenure from loanterm field
    $loanTenure = '';
    $loanTerm = $loan->loanterm ?? 0;
    $frequency = $loan->paymentfrequency ?? 'monthly';
    
    if ($loanTerm > 0) {
        if ($frequency === 'weekly') {
            $loanTenure = $loanTerm . ' week' . ($loanTerm > 1 ? 's' : '');
        } elseif ($frequency === 'daily') {
            $loanTenure = $loanTerm . ' day' . ($loanTerm > 1 ? 's' : '');
        } else {
            // monthly
            $loanTenure = $loanTerm . ' month' . ($loanTerm > 1 ? 's' : '');
        }
    } else {
        $loanTenure = 'N/A';
    }
    
    // Calculate maturity date display
    $maturityDateDisplay = $loan->maturitydate ? \Carbon\Carbon::parse($loan->maturitydate)->format('M d, Y') : 'N/A';
    
    // Add CSRF token for API calls
    $csrfToken = csrf_token();
    
    // Add base URL for API endpoints
    $baseUrl = url('/'); // This gets the application base URL
    $apiBaseUrl = $baseUrl . '/api'; // API base URL
    
    // Specific payment gateway endpoints
    $collectEndpoint = $apiBaseUrl . '/payments/collect';
    $statusEndpoint = $apiBaseUrl . '/payments/status';
    
    // Current user and company info for gateway
    $currentUserId = auth()->id();
    $currentCompanyId = auth()->user()->companyid ?? 1;
    
    // Find Mobile Money cash account for gateway
    $mobileMoneyAccount = $cashBankAccounts->first(function($account) {
        $accountName = strtolower($account->accountName);
        return str_contains($accountName, 'mobile money') || 
               str_contains($accountName, 'mobile') || 
               str_contains($accountName, 'mtn') || 
               str_contains($accountName, 'airtel') ||
               str_contains($accountName, 'zamtel');
    });
    
    // Default account ID for gateway
    $defaultAccountId = $mobileMoneyAccount ? $mobileMoneyAccount->cashBankId : ($cashBankAccounts->first()->cashBankId ?? 1);
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title><?php echo e($pageTitle); ?> | IMMIA FINANCE</title>
    
    <!-- Replace CDN with local Tailwind for production -->
    <script src="https://cdn.tailwindcss.com"></script>
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    <script src="https://code.jquery.com/jquery-3.6.0.min.js"></script>
    
    <style>
        :root { --zedbank-blue: #0077C5; }
        .text-zedbank-blue { color: var(--zedbank-blue); }
        .bg-zedbank-blue { background-color: var(--zedbank-blue); }
        .lg\:ml-64 { margin-left: 16rem; }
        @media (max-width: 1024px) { .lg\:ml-64 { margin-left: 0 !important; } }
        
        /* Custom styles */
        .auto-allocation-indicator {
            padding: 15px;
            border-radius: 8px;
            margin-bottom: 15px;
        }
        
        /* Read-only input styling */
        .readonly-input {
            background-color: #f9fafb;
            cursor: not-allowed;
            user-select: none;
        }
        .readonly-input:focus {
            box-shadow: none;
            border-color: #d1d5db;
        }
        
        /* Currency indicator */
        .currency-indicator {
            position: absolute;
            right: 8px;
            top: 50%;
            transform: translateY(-50%);
            color: #6b7280;
            font-size: 0.75rem;
            pointer-events: none;
        }
        
        /* Loan summary readonly inputs */
        .loan-summary-input {
            background-color: #f9fafb;
            border: 1px solid #e5e7eb;
            border-radius: 0.375rem;
            padding: 0.5rem 0.75rem;
            width: 100%;
            font-weight: 600;
            text-align: left;
            color: #1f2937;
        }
        .loan-summary-input:focus {
            outline: none;
            ring: 0;
            border-color: #d1d5db;
        }
        
        /* Balance display styling */
        .balance-label {
            color: #6b7280;
            font-size: 0.875rem;
            font-weight: 500;
        }
        
        .balance-value {
            font-size: 1.125rem;
            font-weight: 700;
            color: #1f2937;
        }
        
        .balance-row {
            padding: 0.75rem 0;
            border-bottom: 1px solid #f3f4f6;
        }
        .balance-row:last-child {
            border-bottom: none;
        }
        
        /* Custom checkbox styling */
        .custom-checkbox {
            position: relative;
            width: 1.25rem;
            height: 1.25rem;
            border-radius: 0.25rem;
            border: 2px solid #d1d5db;
            background-color: white;
            cursor: pointer;
            transition: all 0.2s;
        }
        
        .custom-checkbox.checked {
            background-color: var(--zedbank-blue);
            border-color: var(--zedbank-blue);
        }
        
        .custom-checkbox.checked:after {
            content: '✓';
            position: absolute;
            color: white;
            font-size: 0.75rem;
            font-weight: bold;
            top: 50%;
            left: 50%;
            transform: translate(-50%, -50%);
        }
        
        /* Allocation input styling */
        .allocation-input {
            transition: all 0.2s;
        }
        
        .allocation-input:disabled {
            opacity: 0.5;
            cursor: not-allowed;
        }
        
        /* Toggle switch */
        .toggle-switch {
            position: relative;
            display: inline-block;
            width: 44px;
            height: 24px;
        }
        
        .toggle-switch input {
            opacity: 0;
            width: 0;
            height: 0;
        }
        
        .toggle-slider {
            position: absolute;
            cursor: pointer;
            top: 0;
            left: 0;
            right: 0;
            bottom: 0;
            background-color: #ccc;
            transition: .4s;
            border-radius: 24px;
        }
        
        .toggle-slider:before {
            position: absolute;
            content: "";
            height: 18px;
            width: 18px;
            left: 3px;
            bottom: 3px;
            background-color: white;
            transition: .4s;
            border-radius: 50%;
        }
        
        input:checked + .toggle-slider {
            background-color: var(--zedbank-blue);
        }
        
        input:checked + .toggle-slider:before {
            transform: translateX(20px);
        }
        
        /* Manual allocation section styling */
        .manual-allocation-item {
            padding: 12px;
            border: 1px solid #e5e7eb;
            border-radius: 8px;
            margin-bottom: 12px;
            transition: all 0.3s;
        }
        
        .manual-allocation-item.active {
            border-color: var(--zedbank-blue);
            background-color: #f0f9ff;
        }
        
        .allocation-amount {
            font-weight: 600;
            color: #1f2937;
        }
        
        .balance-remaining {
            font-size: 0.875rem;
            color: #6b7280;
        }
        
        /* Compact input styling (from create.blade.php) */
        .compact-input { 
            padding: 10px 14px; 
            font-size: 0.875rem; 
        } 
        .compact-label { 
            font-size: 0.875rem; 
        } 
        
        /* Loading spinner */
        .loading-spinner {
            border: 3px solid #f3f3f3;
            border-top: 3px solid #0077C5;
            border-radius: 50%;
            width: 20px;
            height: 20px;
            animation: spin 1s linear infinite;
            display: inline-block;
        }
        
        @keyframes spin {
            0% { transform: rotate(0deg); }
            100% { transform: rotate(360deg); }
        }
        
        /* Fade in animation */
        @keyframes fadeIn {
            from { opacity: 0; transform: translateY(-10px); }
            to { opacity: 1; transform: translateY(0); }
        }
        
        /* Success/Error message styling */
        .message-container {
            animation: fadeIn 0.3s ease-out;
        }
        
        /* API processing indicator */
        .api-processing {
            border-left: 4px solid #0077C5;
            background-color: #f0f9ff;
        }
        
        /* Gateway processing indicator */
        .gateway-processing {
            border-left: 4px solid #10b981;
            background-color: #f0fdf4;
        }
        
        /* Gateway status colors */
        .gateway-status-pending {
            border-left: 4px solid #0077C5;
            background-color: #f0f9ff;
        }
        
        .gateway-status-success {
            border-left: 4px solid #10b981;
            background-color: #f0fdf4;
        }
        
        .gateway-status-failed {
            border-left: 4px solid #ef4444;
            background-color: #fef2f2;
        }
        
        .gateway-status-timeout {
            border-left: 4px solid #f59e0b;
            background-color: #fffbeb;
        }
        
        /* Validation error styling */
        .border-red-500 {
            border-color: #ef4444 !important;
        }
        
        /* Remove browser validation styling */
        input:invalid, select:invalid {
            box-shadow: none;
        }
        
        /* Custom validation message */
        .validation-error {
            color: #ef4444;
            font-size: 0.875rem;
            margin-top: 0.25rem;
        }
        
        /* Highlight for gateway auto-selected account */
        .gateway-auto-selected {
            border-color: #10b981 !important;
            background-color: #f0fdf4 !important;
        }
    </style>
</head>
<body class="bg-gray-50 min-h-screen">
    <?php echo $__env->make('accounting.partials.sidebar', array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

    <div class="lg:ml-64 min-h-screen">
        <?php echo $__env->make('accounting.partials.header', [
            'title' => $pageTitle,
            'subtitle' => $pageSubtitle
        ], array_diff_key(get_defined_vars(), ['__data' => 1, '__path' => 1]))->render(); ?>

        <main class="p-6">
            <div class="max-w-7xl mx-auto">
                
                <div class="mb-6">
                    <a href="<?php echo e(route('admin.loans.active')); ?>" 
                       class="inline-flex items-center text-zedbank-blue font-medium hover:text-zedbank-blue-dark">
                        <i class="fas fa-arrow-left mr-2"></i> Back to Active Loans
                    </a>
                </div>

                
                <div id="messageContainer"></div>
                
                
                <div id="completionDetails" class="hidden"></div>
                
                
                <div id="manualCompletionOption" class="hidden"></div>

                <form id="paymentForm" method="POST">
                    <input type="hidden" name="_token" value="<?php echo e($csrfToken); ?>">
                    
                    
                    <input type="hidden" id="rprincipal" name="rprincipal" value="0">
                    <input type="hidden" id="rinterest" name="rinterest" value="0">
                    <input type="hidden" id="processing_fees_amount_paid" name="processing_fees_amount_paid" value="0">
                    <input type="hidden" id="penalties" name="penalties" value="0">
                    
                    
                    <input type="hidden" id="allocation_mode" name="allocation_mode" value="auto">
                    
                    <div class="grid grid-cols-1 lg:grid-cols-3 gap-6">
                        
                        <div class="lg:col-span-2 space-y-6">
                            
                            
                            <div class="bg-white rounded-xl shadow-sm border overflow-hidden">
                                <div class="p-4 border-b bg-gray-50">
                                    <h3 class="font-bold text-gray-800">
                                        <i class="fas fa-file-invoice-dollar mr-2 text-zedbank-blue"></i> Loan Summary
                                    </h3>
                                </div>
                                <div class="p-5">
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-6 mb-6">
                                        
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Loan Number</label>
                                            <input type="text" 
                                                   value="<?php echo e($loan->loannumber); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Customer</label>
                                            <input type="text" 
                                                   value="<?php echo e($customerName); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                            <div class="mt-1">
                                                <input type="text" 
                                                       value="<?php echo e($customerNumber); ?>"
                                                       readonly
                                                       class="loan-summary-input readonly-input text-sm font-normal text-gray-500">
                                            </div>
                                        </div>
                                        
                                        
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Disbursed Date</label>
                                            <input type="text" 
                                                   value="<?php echo e($loan->disburseddate ? \Carbon\Carbon::parse($loan->disburseddate)->format('M d, Y') : 'N/A'); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Maturity Date</label>
                                            <input type="text" 
                                                   value="<?php echo e($maturityDateDisplay); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        
                                        
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Loan Tenure</label>
                                            <input type="text" 
                                                   value="<?php echo e($loanTenure); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Payment Frequency</label>
                                            <input type="text" 
                                                   value="<?php echo e($loan->paymentfrequency === 'weekly' ? 'Weekly' : ($loan->paymentfrequency === 'monthly' ? 'Monthly' : 'Daily')); ?>"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        
                                        
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Interest Rate</label>
                                            <input type="text" 
                                                   value="<?php echo e(number_format($loan->interestrate, 2)); ?>%"
                                                   readonly
                                                   class="loan-summary-input readonly-input">
                                        </div>
                                        <div>
                                            <label class="block text-sm text-gray-600 font-medium mb-1">Installment Amount</label>
                                            <div class="relative">
                                                <input type="text" 
                                                       value="<?php echo e(number_format($loan->installmentamount, 2)); ?>"
                                                       readonly
                                                       class="loan-summary-input readonly-input pl-12">
                                                <span class="absolute left-3 top-1/2 transform -translate-y-1/2 text-gray-500 text-sm">ZMW</span>
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div class="bg-gray-50 border border-gray-200 rounded-lg p-4">
                                        <h4 class="font-medium text-gray-700 mb-3">Loan Balances</h4>
                                        <div class="space-y-2">
                                            <div class="balance-row">
                                                <div class="flex justify-between items-center">
                                                    <span class="balance-label">Principal Balance</span>
                                                    <div class="relative">
                                                        <input type="text" 
                                                               id="principalBalanceDisplay" 
                                                               value="<?php echo e(number_format($principalBalance, 2)); ?>"
                                                               readonly
                                                               class="readonly-input text-right font-bold text-gray-800 w-40 pl-3 pr-10 py-1 border border-gray-300 rounded text-sm">
                                                        <span class="currency-indicator">ZMW</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="balance-row">
                                                <div class="flex justify-between items-center">
                                                    <span class="balance-label">Interest Balance</span>
                                                    <div class="relative">
                                                        <input type="text" 
                                                               id="interestBalanceDisplay" 
                                                               value="<?php echo e(number_format($interestBalance, 2)); ?>"
                                                               readonly
                                                               class="readonly-input text-right font-bold text-gray-800 w-40 pl-3 pr-10 py-1 border border-gray-300 rounded text-sm">
                                                        <span class="currency-indicator">ZMW</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="balance-row">
                                                <div class="flex justify-between items-center">
                                                    <span class="balance-label">Processing Fee Balance</span>
                                                    <div class="relative">
                                                        <input type="text" 
                                                               id="processingBalanceDisplay" 
                                                               value="<?php echo e(number_format($processingBalance, 2)); ?>"
                                                               readonly
                                                               class="readonly-input text-right font-bold text-gray-800 w-40 pl-3 pr-10 py-1 border border-gray-300 rounded text-sm">
                                                        <span class="currency-indicator">ZMW</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="balance-row">
                                                <div class="flex justify-between items-center">
                                                    <span class="balance-label">Penalty Balance</span>
                                                    <div class="relative">
                                                        <input type="text" 
                                                               id="penaltyBalanceDisplay" 
                                                               value="<?php echo e(number_format($penaltyBalance, 2)); ?>"
                                                               readonly
                                                               class="readonly-input text-right font-bold text-gray-800 w-40 pl-3 pr-10 py-1 border border-gray-300 rounded text-sm">
                                                        <span class="currency-indicator">ZMW</span>
                                                    </div>
                                                </div>
                                            </div>
                                            
                                            <div class="pt-3 border-t border-gray-300 mt-2">
                                                <div class="flex justify-between items-center">
                                                    <span class="balance-label font-bold">Total Due</span>
                                                    <div class="relative">
                                                        <input type="text" 
                                                               id="totalBalanceDisplay" 
                                                               value="<?php echo e(number_format($totalBalance, 2)); ?>"
                                                               readonly
                                                               class="readonly-input text-right font-bold text-blue-800 w-40 pl-3 pr-10 py-1 border border-blue-300 rounded text-sm">
                                                        <span class="currency-indicator">ZMW</span>
                                                    </div>
                                                </div>
                                            </div>
                                        </div>
                                    </div>
                                    
                                    <?php if($nextPayment): ?>
                                        <div class="mt-4 pt-4 border-t border-gray-100">
                                            <div class="flex items-center justify-between">
                                                <div class="flex-1">
                                                    <label class="block text-sm text-gray-600 font-medium mb-1">Next Scheduled Payment</label>
                                                    <div class="grid grid-cols-2 gap-4">
                                                        <div class="relative">
                                                            <input type="text" 
                                                                   value="<?php echo e(number_format($nextPayment->amountdue, 2)); ?>"
                                                                   readonly
                                                                   class="loan-summary-input readonly-input font-bold">
                                                            <span class="absolute right-2 top-1/2 transform -translate-y-1/2 text-xs text-gray-500">ZMW</span>
                                                        </div>
                                                        <div>
                                                            <input type="text" 
                                                                   value="Due on <?php echo e(\Carbon\Carbon::parse($nextPayment->paymentdate)->format('M d, Y')); ?>"
                                                                   readonly
                                                                   class="loan-summary-input readonly-input text-sm font-normal text-gray-500">
                                                        </div>
                                                    </div>
                                                </div>
                                                <button type="button" 
                                                        onclick="useScheduledAmount(<?php echo e($nextPayment->amountdue); ?>)"
                                                        class="ml-4 px-3 py-2 bg-blue-100 text-blue-700 rounded-lg text-sm font-medium hover:bg-blue-200 whitespace-nowrap">
                                                    Use This Amount
                                                </button>
                                            </div>
                                        </div>
                                    <?php endif; ?>
                                </div>
                            </div>

                            
                            <div class="bg-white rounded-xl shadow-sm border overflow-hidden">
                                <div class="p-4 border-b bg-gray-50">
                                    <h3 class="font-bold text-gray-800">
                                        <i class="fas fa-money-check-alt mr-2 text-zedbank-blue"></i> Payment Details
                                    </h3>
                                </div>
                                <div class="p-5 space-y-6">
                                    
                                    
                                    <div>
                                        <label for="processing_method" class="block text-sm font-medium text-gray-700 mb-2">
                                            Payment Processing <span class="text-red-500">*</span>
                                        </label>
                                        <select id="processing_method" 
                                                name="processing_method"
                                                class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                       focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                onchange="handleProcessingMethodChange()"
                                                required>
                                            <option value="">Select Processing Method</option>
                                            <option value="manual" selected>Manual Payment (Record Only)</option>
                                            <option value="gateway">Payment Gateway (Process Now)</option>
                                        </select>
                                        <div class="mt-1 text-xs text-gray-500">
                                            <div id="manual_description">
                                                <i class="fas fa-info-circle text-blue-500"></i> 
                                                <strong>Manual Payment:</strong> Record a payment that was already made outside the system (cash, bank deposit, etc.). 
                                                The system will only record the transaction without processing any actual payment.
                                            </div>
                                            <div id="gateway_description" class="hidden">
                                                <i class="fas fa-credit-card text-green-500"></i> 
                                                <strong>Payment Gateway:</strong> Process payment immediately through an integrated payment gateway. 
                                                Customer will be redirected to make payment online.
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div id="gateway_message" class="hidden">
                                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-4">
                                            <div class="flex items-center">
                                                <i class="fas fa-external-link-alt text-blue-500 mr-3 text-lg"></i>
                                                <div>
                                                    <h4 class="font-medium text-blue-800">Payment Gateway Processing</h4>
                                                    <p class="text-sm text-blue-700 mt-1">
                                                        The customer will receive a mobile money prompt to approve the payment.
                                                        Payment confirmation will be automatic once processed.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div id="gateway_info" class="hidden mb-4">
                                        <div class="bg-green-50 border border-green-200 rounded-lg p-4">
                                            <div class="flex items-center">
                                                <i class="fas fa-shield-alt text-green-500 mr-3 text-lg"></i>
                                                <div>
                                                    <h4 class="font-medium text-green-800">Secure Online Payment</h4>
                                                    <p class="text-sm text-green-700 mt-1">
                                                        <i class="fas fa-lock mr-1"></i> All transactions are encrypted
                                                        <br>
                                                        <i class="fas fa-check-circle mr-1"></i> Instant payment confirmation
                                                        <br>
                                                        <i class="fas fa-receipt mr-1"></i> Automatic receipt generation
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div>
                                        <label for="rtotal" class="block text-sm font-medium text-gray-700 mb-2">
                                            Payment Amount <span class="text-red-500">*</span>
                                        </label>
                                        <div class="relative">
                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                <span class="text-gray-500">ZMW</span>
                                            </div>
                                            <input type="number" 
                                                   id="rtotal" 
                                                   name="rtotal"
                                                   step="0.01"
                                                   min="0.01"
                                                   max="<?php echo e($totalBalance); ?>"
                                                   value="<?php echo e(old('rtotal')); ?>"
                                                   class="block w-full pl-16 pr-4 py-3 border border-gray-300 rounded-lg 
                                                          focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                   placeholder="0.00"
                                                   oninput="calculateAllocations()"
                                                   required>
                                        </div>
                                        <p class="mt-1 text-xs text-gray-500">
                                            Maximum payable amount: <?php echo e(number_format($totalBalance, 2)); ?>

                                        </p>
                                    </div>

                                    
                                    <div class="border border-gray-200 rounded-lg p-4">
                                        <div class="flex items-center justify-between mb-4">
                                            <div>
                                                <h4 class="font-medium text-gray-800">Payment Allocation</h4>
                                                <p class="text-sm text-gray-600">Choose how to allocate the payment</p>
                                            </div>
                                            <div class="flex items-center space-x-3">
                                                <span class="text-sm font-medium text-gray-700">Auto</span>
                                                <label class="toggle-switch">
                                                    <input type="checkbox" id="manualAllocationToggle" onchange="toggleAllocationMode()">
                                                    <span class="toggle-slider"></span>
                                                </label>
                                                <span class="text-sm font-medium text-gray-700">Manual</span>
                                            </div>
                                        </div>
                                        
                                        <div id="autoAllocationDisplay">
                                            <div class="bg-blue-50 border border-blue-200 auto-allocation-indicator rounded-lg">
                                                <div class="flex items-center justify-between mb-4">
                                                    <h4 class="font-medium text-blue-800">
                                                        <i class="fas fa-robot mr-2"></i> Auto-Allocation Breakdown
                                                    </h4>
                                                    <span class="text-xs bg-blue-100 text-blue-800 px-2 py-1 rounded-full">
                                                        Automated
                                                    </span>
                                                </div>
                                                
                                                <div class="space-y-4">
                                                    
                                                    <div class="flex items-center justify-between">
                                                        <label class="text-sm font-medium text-gray-700">Penalty:</label>
                                                        <div class="relative w-48">
                                                            <input type="text" 
                                                                   id="allocPenalty" 
                                                                   value="0.00"
                                                                   readonly
                                                                   class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-gray-300 rounded text-sm font-bold text-red-700 text-right">
                                                            <span class="currency-indicator">ZMW</span>
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="flex items-center justify-between">
                                                        <label class="text-sm font-medium text-gray-700">Processing Fee:</label>
                                                        <div class="relative w-48">
                                                            <input type="text" 
                                                                   id="allocProcessing" 
                                                                   value="0.00"
                                                                   readonly
                                                                   class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-gray-300 rounded text-sm font-bold text-purple-600 text-right">
                                                            <span class="currency-indicator">ZMW</span>
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="flex items-center justify-between">
                                                        <label class="text-sm font-medium text-gray-700">Interest:</label>
                                                        <div class="relative w-48">
                                                            <input type="text" 
                                                                   id="allocInterest" 
                                                                   value="0.00"
                                                                   readonly
                                                                   class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-gray-300 rounded text-sm font-bold text-yellow-700 text-right">
                                                            <span class="currency-indicator">ZMW</span>
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="flex items-center justify-between">
                                                        <label class="text-sm font-medium text-gray-700">Principal:</label>
                                                        <div class="relative w-48">
                                                            <input type="text" 
                                                                   id="allocPrincipal" 
                                                                   value="0.00"
                                                                   readonly
                                                                   class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-gray-300 rounded text-sm font-bold text-red-600 text-right">
                                                            <span class="currency-indicator">ZMW</span>
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="flex items-center justify-between pt-4 border-t border-gray-200">
                                                        <label class="text-sm font-medium text-gray-700">Total Allocated:</label>
                                                        <div class="relative w-48">
                                                            <input type="text" 
                                                                   id="allocTotal" 
                                                                   value="0.00"
                                                                   readonly
                                                                   class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-300 rounded text-sm font-bold text-green-600 text-right">
                                                            <span class="currency-indicator">ZMW</span>
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mt-4 pt-4 border-t border-blue-200">
                                                    <p class="text-xs text-blue-700">
                                                        <i class="fas fa-lightbulb mr-1"></i>
                                                        <span class="font-medium">Allocation Priority:</span> Penalties → Processing Fees → Interest → Principal
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                        
                                        <div id="manualAllocationDisplay" class="hidden">
                                            <div class="bg-amber-50 border border-amber-200 auto-allocation-indicator rounded-lg">
                                                <div class="flex items-center justify-between mb-4">
                                                    <h4 class="font-medium text-amber-800">
                                                        <i class="fas fa-hand-pointer mr-2"></i> Manual Allocation
                                                    </h4>
                                                    <span class="text-xs bg-amber-100 text-amber-800 px-2 py-1 rounded-full">
                                                        Manual Selection
                                                    </span>
                                                </div>
                                                
                                                <p class="text-sm text-amber-700 mb-4">
                                                    <i class="fas fa-info-circle mr-1"></i>
                                                    Check the components you want this payment to cover:
                                                </p>
                                                
                                                <div class="space-y-3">
                                                    
                                                    <div class="manual-allocation-item" id="penaltyAllocationItem">
                                                        <div class="flex items-center justify-between mb-2">
                                                            <div class="flex items-center space-x-3">
                                                                <div class="custom-checkbox" onclick="toggleAllocation('penalty')"></div>
                                                                <div>
                                                                    <label class="text-sm font-medium text-gray-700 cursor-pointer" onclick="toggleAllocation('penalty')">Penalty</label>
                                                                    <div class="text-xs text-gray-500">Apply to penalty balance</div>
                                                                </div>
                                                            </div>
                                                            <div class="text-right">
                                                                <div class="allocation-amount" id="penaltyAmount">0.00</div>
                                                                <div class="balance-remaining">
                                                                    Balance: <span id="penaltyBalanceRemaining"><?php echo e(number_format($penaltyBalance, 2)); ?></span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="relative mt-2">
                                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                                <span class="text-gray-500">ZMW</span>
                                                            </div>
                                                            <input type="number" 
                                                                   id="manualPenalty" 
                                                                   step="0.01"
                                                                   min="0"
                                                                   max="<?php echo e($penaltyBalance); ?>"
                                                                   value="0"
                                                                   disabled
                                                                   oninput="updateManualAllocation('penalty')"
                                                                   class="allocation-input block w-full pl-16 pr-4 py-2 border border-gray-300 rounded-lg 
                                                                          focus:ring-zedbank-blue focus:border-zedbank-blue">
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="manual-allocation-item" id="processingAllocationItem">
                                                        <div class="flex items-center justify-between mb-2">
                                                            <div class="flex items-center space-x-3">
                                                                <div class="custom-checkbox" onclick="toggleAllocation('processing')"></div>
                                                                <div>
                                                                    <label class="text-sm font-medium text-gray-700 cursor-pointer" onclick="toggleAllocation('processing')">Processing Fee</label>
                                                                    <div class="text-xs text-gray-500">Apply to processing fee balance</div>
                                                                </div>
                                                            </div>
                                                            <div class="text-right">
                                                                <div class="allocation-amount" id="processingAmount">0.00</div>
                                                                <div class="balance-remaining">
                                                                    Balance: <span id="processingBalanceRemaining"><?php echo e(number_format($processingBalance, 2)); ?></span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="relative mt-2">
                                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                                <span class="text-gray-500">ZMW</span>
                                                            </div>
                                                            <input type="number" 
                                                                   id="manualProcessing" 
                                                                   step="0.01"
                                                                   min="0"
                                                                   max="<?php echo e($processingBalance); ?>"
                                                                   value="0"
                                                                   disabled
                                                                   oninput="updateManualAllocation('processing')"
                                                                   class="allocation-input block w-full pl-16 pr-4 py-2 border border-gray-300 rounded-lg 
                                                                          focus:ring-zedbank-blue focus:border-zedbank-blue">
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="manual-allocation-item" id="interestAllocationItem">
                                                        <div class="flex items-center justify-between mb-2">
                                                            <div class="flex items-center space-x-3">
                                                                <div class="custom-checkbox" onclick="toggleAllocation('interest')"></div>
                                                                <div>
                                                                    <label class="text-sm font-medium text-gray-700 cursor-pointer" onclick="toggleAllocation('interest')">Interest</label>
                                                                    <div class="text-xs text-gray-500">Apply to interest balance</div>
                                                                </div>
                                                            </div>
                                                            <div class="text-right">
                                                                <div class="allocation-amount" id="interestAmount">0.00</div>
                                                                <div class="balance-remaining">
                                                                    Balance: <span id="interestBalanceRemaining"><?php echo e(number_format($interestBalance, 2)); ?></span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="relative mt-2">
                                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                                <span class="text-gray-500">ZMW</span>
                                                            </div>
                                                            <input type="number" 
                                                                   id="manualInterest" 
                                                                   step="0.01"
                                                                   min="0"
                                                                   max="<?php echo e($interestBalance); ?>"
                                                                   value="0"
                                                                   disabled
                                                                   oninput="updateManualAllocation('interest')"
                                                                   class="allocation-input block w-full pl-16 pr-4 py-2 border border-gray-300 rounded-lg 
                                                                          focus:ring-zedbank-blue focus:border-zedbank-blue">
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="manual-allocation-item" id="principalAllocationItem">
                                                        <div class="flex items-center justify-between mb-2">
                                                            <div class="flex items-center space-x-3">
                                                                <div class="custom-checkbox" onclick="toggleAllocation('principal')"></div>
                                                                <div>
                                                                    <label class="text-sm font-medium text-gray-700 cursor-pointer" onclick="toggleAllocation('principal')">Principal</label>
                                                                    <div class="text-xs text-gray-500">Apply to principal balance</div>
                                                                </div>
                                                            </div>
                                                            <div class="text-right">
                                                                <div class="allocation-amount" id="principalAmount">0.00</div>
                                                                <div class="balance-remaining">
                                                                    Balance: <span id="principalBalanceRemaining"><?php echo e(number_format($principalBalance, 2)); ?></span>
                                                                </div>
                                                            </div>
                                                        </div>
                                                        <div class="relative mt-2">
                                                            <div class="absolute inset-y-0 left-0 pl-3 flex items-center pointer-events-none">
                                                                <span class="text-gray-500">ZMW</span>
                                                            </div>
                                                            <input type="number" 
                                                                   id="manualPrincipal" 
                                                                   step="0.01"
                                                                   min="0"
                                                                   max="<?php echo e($principalBalance); ?>"
                                                                   value="0"
                                                                   disabled
                                                                   oninput="updateManualAllocation('principal')"
                                                                   class="allocation-input block w-full pl-16 pr-4 py-2 border border-gray-300 rounded-lg 
                                                                          focus:ring-zedbank-blue focus:border-zedbank-blue">
                                                        </div>
                                                    </div>
                                                    
                                                    
                                                    <div class="pt-4 border-t border-amber-200">
                                                        <div class="flex items-center justify-between mb-2">
                                                            <div>
                                                                <span class="text-sm font-medium text-gray-700">Total Allocated:</span>
                                                                <div class="text-xs text-gray-500">Payment allocation breakdown</div>
                                                            </div>
                                                            <div class="text-right">
                                                                <div class="text-lg font-bold text-green-600" id="manualTotalAllocated">0.00</div>
                                                                <div class="text-xs text-gray-500">ZMW</div>
                                                            </div>
                                                        </div>
                                                        
                                                        <div class="flex items-center justify-between">
                                                            <span class="text-sm text-gray-600">Payment Amount:</span>
                                                            <span class="text-sm font-medium text-gray-800" id="manualPaymentAmount">0.00</span>
                                                        </div>
                                                        
                                                        <div class="flex items-center justify-between">
                                                            <span class="text-sm text-gray-600">Remaining to Allocate:</span>
                                                            <span class="text-sm font-medium" id="manualRemainingAmount">0.00</span>
                                                        </div>
                                                        
                                                        <div id="manualAllocationError" class="mt-2 text-sm text-red-600 hidden">
                                                            <i class="fas fa-exclamation-triangle mr-1"></i>
                                                            <span id="manualAllocationErrorMessage"></span>
                                                        </div>
                                                        
                                                        <div id="manualAllocationSuccess" class="mt-2 text-sm text-green-600 hidden">
                                                            <i class="fas fa-check-circle mr-1"></i>
                                                            <span>Allocation matches payment amount</span>
                                                        </div>
                                                    </div>
                                                </div>
                                                
                                                <div class="mt-4 pt-4 border-t border-amber-200">
                                                    <p class="text-xs text-amber-700">
                                                        <i class="fas fa-lightbulb mr-1"></i>
                                                        <span class="font-medium">Tip:</span> Select components and specify amounts. Ensure total allocation matches payment amount.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div>
                                        <label for="pmethod" class="block text-sm font-medium text-gray-700 mb-2">
                                            Payment Method <span class="text-red-500">*</span>
                                        </label>
                                        <select id="pmethod" 
                                                name="pmethod"
                                                class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                       focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                onchange="handlePaymentMethodChange()"
                                                required>
                                            <option value="">Select Payment Method</option>
                                            <option value="Cash" selected>Cash</option>
                                            <option value="Bank">Bank Transfer</option>
                                            <option value="Mobile Money">Mobile Money</option>
                                            <option value="Cheque">Cheque</option>
                                            <option value="Card">Card Payment</option>
                                            <option value="Other">Other</option>
                                        </select>
                                    </div>

                                    
                                    <div id="payment_bank_fields" class="mt-4 pt-4 border-t border-gray-200 hidden space-y-4">
                                        <h4 class="font-medium text-gray-800 mb-3">Bank Transfer Details</h4>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div class="md:col-span-2">
                                                <label class="block compact-label font-medium mb-1">Bank Name <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="bank_name" 
                                                       value="<?php echo e(old('bank_name')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Account Holder <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="account_holder" 
                                                       value="<?php echo e(old('account_holder')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Account Number <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="account_number" 
                                                       value="<?php echo e(old('account_number')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Branch <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="bank_branch" 
                                                       value="<?php echo e(old('bank_branch')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">SWIFT Code</label>
                                                <input type="text" 
                                                       name="swift_code" 
                                                       value="<?php echo e(old('swift_code')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg">
                                            </div>
                                            <div class="md:col-span-2">
                                                <label class="block compact-label font-medium mb-1">Transaction Reference <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="bank_reference" 
                                                       id="bank_reference"
                                                       value="<?php echo e(old('bank_reference')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg"
                                                       placeholder="Bank transaction ID or reference">
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div id="payment_mobile_fields" class="mt-4 pt-4 border-t border-gray-200 hidden space-y-4">
                                        <h4 class="font-medium text-gray-800 mb-3">Mobile Money Details</h4>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Payment Channel <span class="text-red-500">*</span></label>
                                                <select name="mobile_channel" class="w-full compact-input border border-gray-300 rounded-lg">
                                                    <option value="">Select Channel</option>
                                                    <option value="MTN Mobile">MTN Mobile Money</option>
                                                    <option value="Airtel Mobile">Airtel Mobile Money</option>
                                                    <option value="Zamtel Mobile">Zamtel Mobile Money</option>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Mobile Phone Number <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       name="mobile_number" 
                                                       id="mobile_number"
                                                       value="<?php echo e(old('mobile_number')); ?>"
                                                       class="w-full compact-input border border-gray-300 rounded-lg"
                                                       placeholder="e.g., 0961234567">
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div id="gateway_mobile_info" class="mt-4 pt-4 border-t border-gray-200 hidden space-y-4">
                                        <h4 class="font-medium text-gray-800 mb-3">Mobile Money Gateway Details</h4>
                                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-3">
                                            <div class="flex items-center">
                                                <i class="fas fa-mobile-alt text-blue-500 mr-3 text-lg"></i>
                                                <div>
                                                    <h4 class="font-medium text-blue-800">Mobile Money Payment</h4>
                                                    <p class="text-sm text-blue-700 mt-1">
                                                        The customer will receive a mobile money prompt to approve the payment.
                                                        Ensure the mobile number is correct for the selected network.
                                                    </p>
                                                </div>
                                            </div>
                                        </div>
                                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Mobile Network <span class="text-red-500">*</span></label>
                                                <select id="gateway_mobile_channel" 
                                                        name="gateway_mobile_channel" 
                                                        class="w-full compact-input border border-gray-300 rounded-lg">
                                                    <option value="">Select Network</option>
                                                    <option value="mtn">MTN Mobile Money</option>
                                                    <option value="airtel">Airtel Money</option>
                                                    <option value="zamtel">Zamtel Kwacha</option>
                                                </select>
                                            </div>
                                            <div>
                                                <label class="block compact-label font-medium mb-1">Mobile Phone Number <span class="text-red-500">*</span></label>
                                                <input type="text" 
                                                       id="gateway_mobile_number" 
                                                       name="gateway_mobile_number"
                                                       class="w-full compact-input border border-gray-300 rounded-lg"
                                                       placeholder="e.g., 0976379025">
                                                <p class="text-xs text-gray-500 mt-1">Enter the customer's mobile number for payment</p>
                                            </div>
                                        </div>
                                    </div>

                                    
                                    <div>
                                        <label for="paccount" class="block text-sm font-medium text-gray-700 mb-2">
                                            Payment Account <span class="text-red-500">*</span>
                                        </label>
                                        <select id="paccount" 
                                                name="paccount"
                                                class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                       focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                required>
                                            <option value="">Select Account</option>
                                            <?php $__currentLoopData = $cashBankAccounts; $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $account): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                                                <option value="<?php echo e($account->cashBankId); ?>" 
                                                        <?php if(old('paccount') == $account->cashBankId): echo 'selected'; endif; ?>
                                                        data-account-name="<?php echo e(strtolower($account->accountName)); ?>"
                                                        <?php if($mobileMoneyAccount && $account->cashBankId == $mobileMoneyAccount->cashBankId): ?> 
                                                            class="font-medium text-green-600" 
                                                        <?php endif; ?>>
                                                    <?php echo e($account->accountName); ?> 
                                                    <?php if($account->accountNumber): ?>
                                                        (<?php echo e($account->accountNumber); ?>)
                                                    <?php endif; ?>
                                                    - Balance: <?php echo e(number_format($account->current_balance ?? $account->balance ?? 0, 2)); ?>

                                                </option>
                                            <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?>
                                        </select>
                                        <div id="gateway_account_note" class="hidden mt-1 text-xs text-green-600">
                                            <i class="fas fa-info-circle"></i> This account has been auto-selected for gateway payments
                                        </div>
                                    </div>

                                    
                                    <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                                        <div>
                                            <label for="paymentreference" class="block text-sm font-medium text-gray-700 mb-2">
                                                Payment Reference
                                            </label>
                                            <input type="text" 
                                                   id="paymentreference" 
                                                   name="paymentreference"
                                                   value="<?php echo e(old('paymentreference')); ?>"
                                                   class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                          focus:ring-zedbank-blue focus:border-zedbank-blue readonly-input"
                                                   placeholder="Will be auto-generated for gateway payments"
                                                   readonly>
                                            <p class="text-xs text-gray-500 mt-1">
                                                <span id="paymentReferenceHelp">
                                                    For gateway payments, this will be auto-filled with the transaction reference
                                                </span>
                                            </p>
                                        </div>
                                        
                                        <div>
                                            <label for="rdate" class="block text-sm font-medium text-gray-700 mb-2">
                                                Payment Date <span class="text-red-500">*</span>
                                            </label>
                                            <input type="date" 
                                                   id="rdate" 
                                                   name="rdate"
                                                   value="<?php echo e(old('rdate', date('Y-m-d'))); ?>"
                                                   class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                          focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                   required>
                                        </div>
                                    </div>

                                    
                                    <div>
                                        <label for="notes" class="block text-sm font-medium text-gray-700 mb-2">
                                            Payment Notes
                                        </label>
                                        <textarea id="notes" 
                                                  name="notes"
                                                  rows="3"
                                                  class="block w-full px-4 py-3 border border-gray-300 rounded-lg 
                                                         focus:ring-zedbank-blue focus:border-zedbank-blue"
                                                  placeholder="Any additional notes about this payment..."><?php echo e(old('notes')); ?></textarea>
                                    </div>
                                </div>
                            </div>
                            
                            
                            <div id="gateway_status" class="hidden"></div>
                        </div>

                        
                        <div class="space-y-6">
                            
                            <div class="bg-white rounded-xl shadow-sm border p-5">
                                <h3 class="font-bold text-gray-800 mb-4 border-b pb-2">Payment Summary</h3>
                                <div class="space-y-3">
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Payment Amount:</span>
                                        <div class="relative">
                                            <input type="text" 
                                                   id="summaryAmount" 
                                                   value="0.00"
                                                   readonly
                                                   class="readonly-input text-right font-bold text-gray-800 w-32 pl-3 pr-8 py-1 border border-gray-300 rounded text-sm">
                                            <span class="absolute right-2 top-1/2 transform -translate-y-1/2 text-xs text-gray-500">ZMW</span>
                                        </div>
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Processing Method:</span>
                                        <input type="text" 
                                               id="summaryProcessingMethod" 
                                               value="Manual Payment (Record Only)"
                                               readonly
                                               class="readonly-input text-right font-medium text-gray-800 w-32 px-3 py-1 border border-gray-300 rounded text-sm">
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Payment Method:</span>
                                        <input type="text" 
                                               id="summaryMethod" 
                                               value="Cash"
                                               readonly
                                               class="readonly-input text-right font-medium text-gray-800 w-32 px-3 py-1 border border-gray-300 rounded text-sm">
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Payment Date:</span>
                                        <input type="text" 
                                               id="summaryDate" 
                                               value="<?php echo e(date('M d, Y')); ?>"
                                               readonly
                                               class="readonly-input text-right font-medium text-gray-800 w-32 px-3 py-1 border border-gray-300 rounded text-sm">
                                    </div>
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Payment Reference:</span>
                                        <input type="text" 
                                               id="summaryReference" 
                                               value=""
                                               readonly
                                               class="readonly-input text-right font-medium text-gray-800 w-32 px-3 py-1 border border-gray-300 rounded text-sm">
                                    </div>
                                    
                                    
                                    <div id="gatewaySummaryInfo" class="hidden space-y-2 pt-3 border-t">
                                        <div class="flex items-center justify-between">
                                            <span class="text-sm text-gray-600">Gateway Status:</span>
                                            <span id="summaryGatewayStatus" class="text-sm font-medium text-blue-600">Pending</span>
                                        </div>
                                        <div class="flex items-center justify-between">
                                            <span class="text-sm text-gray-600">Transaction ID:</span>
                                            <span id="summaryTransactionId" class="text-sm font-mono text-gray-700 truncate" title="Click to copy"></span>
                                        </div>
                                        <div class="flex items-center justify-between">
                                            <span class="text-sm text-gray-600">Check Status:</span>
                                            <a id="summaryStatusLink" href="#" target="_blank" class="text-sm text-blue-600 hover:underline">View Status</a>
                                        </div>
                                    </div>
                                    
                                    <hr class="my-2">
                                    <div class="flex items-center justify-between">
                                        <span class="text-sm text-gray-600">Remaining Balance:</span>
                                        <div class="relative">
                                            <input type="text" 
                                                   id="summaryNewBalance" 
                                                   value="<?php echo e(number_format($totalBalance, 2)); ?>"
                                                   readonly
                                                   class="readonly-input text-right font-bold text-blue-800 w-32 pl-3 pr-8 py-1 border border-gray-300 rounded text-sm">
                                            <span class="absolute right-2 top-1/2 transform -translate-y-1/2 text-xs text-gray-500">ZMW</span>
                                        </div>
                                    </div>
                                    <div class="pt-3 border-t space-y-2">
                                        <div class="flex items-center justify-between text-sm">
                                            <span class="text-gray-600">Principal Remaining:</span>
                                            <div class="relative">
                                                <input type="text" 
                                                       id="summaryPrincipalRemaining" 
                                                       value="<?php echo e(number_format($principalBalance, 2)); ?>"
                                                       readonly
                                                       class="readonly-input text-right font-medium text-gray-800 w-28 pl-3 pr-7 py-1 border border-gray-300 rounded text-xs">
                                                <span class="absolute right-1 top-1/2 transform -translate-y-1/2 text-xs text-gray-500 text-[10px]">ZMW</span>
                                            </div>
                                        </div>
                                        <div class="flex items-center justify-between text-sm">
                                            <span class="text-gray-600">Interest Remaining:</span>
                                            <div class="relative">
                                                <input type="text" 
                                                       id="summaryInterestRemaining" 
                                                       value="<?php echo e(number_format($interestBalance, 2)); ?>"
                                                       readonly
                                                       class="readonly-input text-right font-medium text-gray-800 w-28 pl-3 pr-7 py-1 border border-gray-300 rounded text-xs">
                                                <span class="absolute right-1 top-1/2 transform -translate-y-1/2 text-xs text-gray-500 text-[10px]">ZMW</span>
                                            </div>
                                        </div>
                                        <div class="flex items-center justify-between text-sm">
                                            <span class="text-gray-600">Processing Fee Remaining:</span>
                                            <div class="relative">
                                                <input type="text" 
                                                       id="summaryProcessingRemaining" 
                                                       value="<?php echo e(number_format($processingBalance, 2)); ?>"
                                                       readonly
                                                       class="readonly-input text-right font-medium text-gray-800 w-28 pl-3 pr-7 py-1 border border-gray-300 rounded text-xs">
                                                <span class="absolute right-1 top-1/2 transform -translate-y-1/2 text-xs text-gray-500 text-[10px]">ZMW</span>
                                            </div>
                                        </div>
                                        <div class="flex items-center justify-between text-sm">
                                            <span class="text-gray-600">Penalty Remaining:</span>
                                            <div class="relative">
                                                <input type="text" 
                                                       id="summaryPenaltyRemaining" 
                                                       value="<?php echo e(number_format($penaltyBalance, 2)); ?>"
                                                       readonly
                                                       class="readonly-input text-right font-medium text-gray-800 w-28 pl-3 pr-7 py-1 border border-gray-300 rounded text-xs">
                                                <span class="absolute right-1 top-1/2 transform -translate-y-1/2 text-xs text-gray-500 text-[10px]">ZMW</span>
                                            </div>
                                        </div>
                                    </div>
                                </div>
                            </div>

                            
                            <div class="bg-white rounded-xl shadow-sm border p-5">
                                <div class="space-y-3">
                                    <button type="submit" 
                                            id="submitBtn"
                                            class="w-full px-4 py-3 bg-zedbank-blue text-white rounded-lg hover:bg-zedbank-blue-dark 
                                                   font-bold text-lg flex items-center justify-center gap-2">
                                        <i class="fas fa-check-circle"></i>
                                        Process Payment
                                    </button>
                                    
                                    <a href="<?php echo e(route('admin.loans.active')); ?>" 
                                       class="block w-full px-4 py-3 border border-gray-300 rounded-lg hover:bg-gray-50 
                                              text-center font-medium">
                                        Cancel
                                    </a>
                                    
                                    <div class="pt-3 border-t">
                                        <div class="flex items-center text-sm text-gray-600">
                                            <i class="fas fa-info-circle mr-2 text-blue-500"></i>
                                            <span id="allocationInfo">Payments auto-allocate: Penalty → Processing Fee → Interest → Principal</span>
                                        </div>
                                        <div class="mt-2 text-xs text-gray-500 hidden" id="apiProcessingInfo">
                                            <i class="fas fa-sync-alt mr-1"></i> This payment will be processed via API
                                        </div>
                                    </div>
                                </div>
                            </div>

                            
                            <?php if($loan->repayments()->exists()): ?>
                            <div class="bg-white rounded-xl shadow-sm border p-5">
                                <h3 class="font-bold text-gray-800 mb-3 border-b pb-2">Recent Payments</h3>
                                <div class="space-y-2 max-h-60 overflow-y-auto">
                                    <?php $__currentLoopData = $loan->repayments()->latest()->take(5)->get(); $__env->addLoop($__currentLoopData); foreach($__currentLoopData as $repayment): $__env->incrementLoopIndices(); $loop = $__env->getLastLoop(); ?>
                                    <div class="p-2 border-b border-gray-100 last:border-0">
                                        <div class="flex justify-between items-start">
                                            <div>
                                                <div class="flex items-center gap-2">
                                                    <input type="text" 
                                                           value="<?php echo e(number_format($repayment->rtotal, 2)); ?>"
                                                           readonly
                                                           class="readonly-input text-sm font-medium text-gray-800 w-24 px-2 py-1 border border-gray-300 rounded">
                                                    <span class="text-xs text-gray-500">ZMW</span>
                                                </div>
                                                <input type="text" 
                                                       value="<?php echo e(\Carbon\Carbon::parse($repayment->rdate)->format('M d, Y')); ?>"
                                                       readonly
                                                       class="readonly-input text-xs text-gray-500 w-full mt-1 px-2 py-1 border border-gray-200 rounded">
                                            </div>
                                            <span class="text-xs px-2 py-1 rounded-full 
                                                         <?php echo e($repayment->status === 'Verified' ? 'bg-green-100 text-green-800' : 'bg-yellow-100 text-yellow-800'); ?>">
                                                <?php echo e(ucfirst($repayment->status)); ?>

                                            </span>
                                        </div>
                                    </div>
                                    <?php endforeach; $__env->popLoop(); $loop = $__env->getLastLoop(); ?>
                                </div>
                            </div>
                            <?php endif; ?>
                        </div>
                    </div>
                </form>
            </div>
        </main>
    </div>

    <script>
        // Global constants for API URLs
        const API_BASE_URL = '<?php echo e($apiBaseUrl); ?>';
        const COLLECT_ENDPOINT = '<?php echo e($collectEndpoint); ?>';
        const STATUS_ENDPOINT = '<?php echo e($statusEndpoint); ?>';
        const CURRENT_USER_ID = <?php echo e($currentUserId); ?>;
        const CURRENT_COMPANY_ID = <?php echo e($currentCompanyId); ?>;
        const DEFAULT_ACCOUNT_ID = <?php echo e($defaultAccountId); ?>;

        // Loan balance data
        const loanBalances = {
            principal: <?php echo e($principalBalance); ?>,
            interest: <?php echo e($interestBalance); ?>,
            processing: <?php echo e($processingBalance); ?>,
            penalty: <?php echo e($penaltyBalance); ?>,
            total: <?php echo e($totalBalance); ?>

        };

        // Current allocation values (auto-calculated)
        let currentAllocations = {
            principal: 0,
            interest: 0,
            processing: 0,
            penalty: 0
        };

        // Manual allocation tracking
        let manualAllocations = {
            principal: { enabled: false, amount: 0 },
            interest: { enabled: false, amount: 0 },
            processing: { enabled: false, amount: 0 },
            penalty: { enabled: false, amount: 0 }
        };

        // Allocation mode
        let isManualMode = false;

        // Gateway state tracking
        let isGatewayProcessing = false;
        let gatewayReferenceId = null;
        let gatewayCheckInterval = null;

        // Initialize on page load
        $(document).ready(function() {
            // Set up event listeners
            $('#rtotal').on('input', calculateAllocations);
            $('#pmethod').on('change', function() {
                handlePaymentMethodChange();
                updateSummary();
            });
            $('#rdate').on('change', updateSummary);
            $('#processing_method').on('change', function() {
                handleProcessingMethodChange();
                updateSummary();
            });
            
            // Manual allocation event listeners
            $('#manualPenalty').on('input', () => updateManualAllocation('penalty'));
            $('#manualProcessing').on('input', () => updateManualAllocation('processing'));
            $('#manualInterest').on('input', () => updateManualAllocation('interest'));
            $('#manualPrincipal').on('input', () => updateManualAllocation('principal'));
            
            // Gateway mobile info listeners
            $(document).on('change', '#gateway_mobile_channel', updateSummary);
            $(document).on('input', '#gateway_mobile_number', updateSummary);
            
            // Initialize
            calculateAllocations();
            updateSummary();
            handlePaymentMethodChange();
            handleProcessingMethodChange();
            
            // Remove required attributes from dynamic fields initially
            removeRequiredFromDynamicFields();
            
            // Set payment reference field to readonly initially
            $('#paymentreference').prop('readonly', true);
        });

        // Helper functions to manage required attributes
        function removeRequiredFromDynamicFields() {
            // Remove required from all dynamic fields
            $('#payment_bank_fields input, #payment_bank_fields select').prop('required', false);
            $('#payment_mobile_fields input, #payment_mobile_fields select').prop('required', false);
            $('#gateway_mobile_info input, #gateway_mobile_info select').prop('required', false);
        }

        function addRequiredToBankFields() {
            // Add required to visible bank fields
            $('#payment_bank_fields input[name="bank_name"]').prop('required', true);
            $('#payment_bank_fields input[name="account_holder"]').prop('required', true);
            $('#payment_bank_fields input[name="account_number"]').prop('required', true);
            $('#payment_bank_fields input[name="bank_branch"]').prop('required', true);
            $('#payment_bank_fields input[name="bank_reference"]').prop('required', true);
            // swift_code is optional, so don't add required
        }

        function addRequiredToMobileFields() {
            // Add required to visible manual mobile fields
            $('#payment_mobile_fields select[name="mobile_channel"]').prop('required', true);
            $('#payment_mobile_fields input[name="mobile_number"]').prop('required', true);
        }

        function addRequiredToGatewayMobileFields() {
            // Add required to gateway mobile fields
            $('#gateway_mobile_info select[name="gateway_mobile_channel"]').prop('required', true);
            $('#gateway_mobile_info input[name="gateway_mobile_number"]').prop('required', true);
        }

        function handlePaymentMethodChange() {
            const method = $('#pmethod').val();
            const processingMethod = $('#processing_method').val();
            const paymentReferenceField = $('#paymentreference');
            const paymentReferenceHelp = $('#paymentReferenceHelp');
            
            // Remove required attributes from ALL dynamic fields first
            removeRequiredFromDynamicFields();
            
            // Hide all dynamic fields first
            $('#payment_bank_fields').addClass('hidden');
            $('#payment_mobile_fields').addClass('hidden');
            $('#gateway_mobile_info').addClass('hidden');
            
            // Clear validation attributes from hidden fields
            $('#payment_bank_fields input, #payment_bank_fields select').prop('required', false);
            $('#payment_mobile_fields input, #payment_mobile_fields select').prop('required', false);
            
            // Update payment reference field behavior
            if (processingMethod === 'manual') {
                if (method === 'Bank' || method === 'Mobile Money') {
                    // For manual bank/mobile money, make reference editable
                    paymentReferenceField.prop('readonly', false)
                                         .removeClass('readonly-input')
                                         .attr('placeholder', 'Enter transaction reference');
                    paymentReferenceHelp.text('Enter the transaction reference or ID');
                } else {
                    // For other manual methods, make reference optional and editable
                    paymentReferenceField.prop('readonly', false)
                                         .removeClass('readonly-input')
                                         .attr('placeholder', 'Optional reference');
                    paymentReferenceHelp.text('Optional reference for this payment');
                }
            } else if (processingMethod === 'gateway') {
                // For gateway, always make reference read-only and auto-filled
                paymentReferenceField.prop('readonly', true)
                                     .addClass('readonly-input')
                                     .attr('placeholder', 'Will be auto-generated for gateway payments');
                paymentReferenceHelp.text('For gateway payments, this will be auto-filled with the transaction reference');
            }
            
            // Show relevant fields based on selected method and processing mode
            if (processingMethod === 'manual') {
                if (method === 'Bank') {
                    $('#payment_bank_fields').removeClass('hidden');
                    // Add required attributes back for visible bank fields
                    addRequiredToBankFields();
                } else if (method === 'Mobile Money') {
                    $('#payment_mobile_fields').removeClass('hidden');
                    // Add required attributes back for visible mobile fields
                    addRequiredToMobileFields();
                }
            } else if (processingMethod === 'gateway') {
                if (method === 'Mobile Money') {
                    // Show gateway-specific mobile money info
                    showGatewayMobileMoneyInfo();
                    // Add required attributes for gateway mobile fields
                    addRequiredToGatewayMobileFields();
                }
            }
            
            updateSummary();
        }

        function handleProcessingMethodChange() {
            const processingMethod = $('#processing_method').val();
            const manualDesc = $('#manual_description');
            const gatewayDesc = $('#gateway_description');
            const gatewayMessage = $('#gateway_message');
            const gatewayInfo = $('#gateway_info');
            const apiProcessingInfo = $('#apiProcessingInfo');
            const pmethodSelect = $('#pmethod');
            const paccountSelect = $('#paccount');
            const paymentReferenceField = $('#paymentreference');
            const paymentReferenceHelp = $('#paymentReferenceHelp');
            
            // Hide all descriptions first
            manualDesc.addClass('hidden');
            gatewayDesc.addClass('hidden');
            gatewayMessage.addClass('hidden');
            gatewayInfo.addClass('hidden');
            apiProcessingInfo.addClass('hidden');
            
            // Reset gateway state
            resetGatewayState();
            
            // Hide gateway mobile info
            $('#gateway_mobile_info').addClass('hidden');
            
            // Clear payment reference when switching modes
            paymentReferenceField.val('');
            
            if (processingMethod === 'manual') {
                manualDesc.removeClass('hidden');
                $('#summaryProcessingMethod').val('Manual Payment (Record Only)');
                apiProcessingInfo.addClass('hidden');
                
                // Enable all fields for manual payment
                pmethodSelect.prop('disabled', false);
                paccountSelect.prop('disabled', false);
                handlePaymentMethodChange(); // This will update payment reference field
                
                // Update payment method options for manual
                updatePaymentMethodsForManual();
                
                // Remove gateway styling from account select
                paccountSelect.removeClass('gateway-auto-selected');
                $('#gateway_account_note').addClass('hidden');
                
            } else if (processingMethod === 'gateway') {
                gatewayDesc.removeClass('hidden');
                gatewayMessage.removeClass('hidden');
                gatewayInfo.removeClass('hidden');
                $('#summaryProcessingMethod').val('Payment Gateway (Process Now)');
                apiProcessingInfo.removeClass('hidden');
                
                // For gateway payments, restrict to Mobile Money only for now
                updatePaymentMethodsForGateway();
                
                // Auto-select a valid cash account for gateway payments
                autoSelectGatewayAccount();
                
                // Payment reference is read-only for gateway
                paymentReferenceField.prop('readonly', true)
                                     .addClass('readonly-input')
                                     .attr('placeholder', 'Will be auto-generated for gateway payments');
                paymentReferenceHelp.text('For gateway payments, this will be auto-filled with the transaction reference');
                
                // Hide bank/mobile specific fields (will be handled by gateway)
                $('#payment_bank_fields').addClass('hidden');
                $('#payment_mobile_fields').addClass('hidden');
                
                // Show mobile money info for gateway
                if (pmethodSelect.val() === 'Mobile Money') {
                    showGatewayMobileMoneyInfo();
                }
            }
            
            updateSummary();
        }

        function autoSelectGatewayAccount() {
            const paccountSelect = $('#paccount');
            const gatewayAccountNote = $('#gateway_account_note');
            
            // First check if we have a default account ID
            if (DEFAULT_ACCOUNT_ID) {
                paccountSelect.val(DEFAULT_ACCOUNT_ID);
                paccountSelect.prop('disabled', true);
                paccountSelect.addClass('gateway-auto-selected');
                gatewayAccountNote.removeClass('hidden');
                return;
            }
            
            // If no default ID, try to find a Mobile Money account
            let mobileMoneyAccountId = null;
            const options = paccountSelect.find('option');
            
            options.each(function() {
                const accountName = $(this).data('account-name') || $(this).text().toLowerCase();
                if (accountName.includes('mobile money') || 
                    accountName.includes('mobile') || 
                    accountName.includes('mtn') || 
                    accountName.includes('airtel') ||
                    accountName.includes('zamtel')) {
                    mobileMoneyAccountId = $(this).val();
                    return false; // Break the loop
                }
            });
            
            if (mobileMoneyAccountId) {
                paccountSelect.val(mobileMoneyAccountId);
                paccountSelect.prop('disabled', true);
                paccountSelect.addClass('gateway-auto-selected');
                gatewayAccountNote.removeClass('hidden');
            } else {
                // If no mobile money account found, use the first available account
                const firstOption = paccountSelect.find('option:not(:first)').first();
                if (firstOption.length) {
                    paccountSelect.val(firstOption.val());
                    paccountSelect.prop('disabled', false);
                }
            }
        }

        function updatePaymentMethodsForGateway() {
            const paymentMethodSelect = $('#pmethod');
            const currentValue = paymentMethodSelect.val();
            
            // For gateway, show only gateway-compatible methods (Mobile Money only for now)
            paymentMethodSelect.html(`
                <option value="">Select Payment Method</option>
                <option value="Mobile Money" ${currentValue === 'Mobile Money' ? 'selected' : ''}>Mobile Money</option>
            `);
            
            // Trigger change to update fields
            paymentMethodSelect.trigger('change');
        }

        function updatePaymentMethodsForManual() {
            const paymentMethodSelect = $('#pmethod');
            const currentValue = paymentMethodSelect.val();
            
            // Restore all payment methods for manual
            paymentMethodSelect.html(`
                <option value="">Select Payment Method</option>
                <option value="Cash" ${currentValue === 'Cash' ? 'selected' : ''}>Cash</option>
                <option value="Bank" ${currentValue === 'Bank' ? 'selected' : ''}>Bank Transfer</option>
                <option value="Mobile Money" ${currentValue === 'Mobile Money' ? 'selected' : ''}>Mobile Money</option>
                <option value="Cheque" ${currentValue === 'Cheque' ? 'selected' : ''}>Cheque</option>
                <option value="Card" ${currentValue === 'Card' ? 'selected' : ''}>Card Payment</option>
                <option value="Other" ${currentValue === 'Other' ? 'selected' : ''}>Other</option>
            `);
            
            // Trigger change to update fields
            paymentMethodSelect.trigger('change');
        }

        function showGatewayMobileMoneyInfo() {
            // Create or show the gateway mobile info section
            let gatewayMobileInfo = $('#gateway_mobile_info');
            
            if (gatewayMobileInfo.length === 0) {
                // Create the section if it doesn't exist
                const pmethodField = $('#pmethod').closest('div');
                gatewayMobileInfo = $(`
                    <div id="gateway_mobile_info" class="mt-4 pt-4 border-t border-gray-200">
                        <h4 class="font-medium text-gray-800 mb-3">Mobile Money Gateway Details</h4>
                        <div class="bg-blue-50 border border-blue-200 rounded-lg p-4 mb-3">
                            <div class="flex items-center">
                                <i class="fas fa-mobile-alt text-blue-500 mr-3 text-lg"></i>
                                <div>
                                    <h4 class="font-medium text-blue-800">Mobile Money Payment</h4>
                                    <p class="text-sm text-blue-700 mt-1">
                                        The customer will receive a mobile money prompt to approve the payment.
                                        Ensure the mobile number is correct for the selected network.
                                    </p>
                                </div>
                            </div>
                        </div>
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <div>
                                <label class="block compact-label font-medium mb-1">Mobile Network <span class="text-red-500">*</span></label>
                                <select id="gateway_mobile_channel" name="gateway_mobile_channel" class="w-full compact-input border border-gray-300 rounded-lg">
                                    <option value="">Select Network</option>
                                    <option value="mtn">MTN Mobile Money</option>
                                    <option value="airtel">Airtel Money</option>
                                    <option value="zamtel">Zamtel Kwacha</option>
                                </select>
                            </div>
                            <div>
                                <label class="block compact-label font-medium mb-1">Mobile Phone Number <span class="text-red-500">*</span></label>
                                <input type="text" 
                                       id="gateway_mobile_number" 
                                       name="gateway_mobile_number"
                                       class="w-full compact-input border border-gray-300 rounded-lg"
                                       placeholder="e.g., 0976379025">
                                <p class="text-xs text-gray-500 mt-1">Enter the customer's mobile number for payment</p>
                            </div>
                        </div>
                    </div>
                `);
                
                // Insert after the payment method field
                pmethodField.after(gatewayMobileInfo);
            } else {
                gatewayMobileInfo.removeClass('hidden');
            }
        }

        function calculateAllocations() {
            if (isManualMode) {
                calculateManualAllocation();
            } else {
                calculateAutoAllocation();
            }
        }

        function calculateAutoAllocation() {
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            const allocationDisplay = $('#autoAllocationDisplay');
            const manualDisplay = $('#manualAllocationDisplay');
            
            // Reset allocations
            currentAllocations = {
                principal: 0,
                interest: 0,
                processing: 0,
                penalty: 0
            };
            
            let remaining = paymentAmount;
            
            // Auto-allocate in priority order: Penalty → Processing Fee → Interest → Principal
            const priorityOrder = ['penalty', 'processing', 'interest', 'principal'];
            
            for (const type of priorityOrder) {
                if (remaining <= 0) break;
                
                const maxAllocation = Math.min(loanBalances[type], remaining);
                currentAllocations[type] = maxAllocation;
                remaining -= maxAllocation;
            }
            
            // Update hidden form fields
            updateHiddenAllocationFields();
            
            // Update allocation display inputs
            $('#allocPenalty').val(currentAllocations.penalty.toFixed(2));
            $('#allocProcessing').val(currentAllocations.processing.toFixed(2));
            $('#allocInterest').val(currentAllocations.interest.toFixed(2));
            $('#allocPrincipal').val(currentAllocations.principal.toFixed(2));
            
            const totalAllocated = Object.values(currentAllocations).reduce((a, b) => a + b, 0);
            $('#allocTotal').val(totalAllocated.toFixed(2));
            
            // Show/hide allocation display
            if (paymentAmount > 0) {
                allocationDisplay.css('display', 'block');
                manualDisplay.css('display', 'none');
                
                // Special styling for full payment
                if (paymentAmount === loanBalances.total) {
                    showFullPaymentAllocation();
                }
            } else {
                allocationDisplay.css('display', 'block');
                manualDisplay.css('display', 'none');
            }
            
            updateSummary();
        }

        function calculateManualAllocation() {
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            
            // Update manual allocation summary
            updateManualAllocationSummary();
            
            updateSummary();
        }

        function toggleAllocationMode() {
            const toggle = $('#manualAllocationToggle');
            const autoDisplay = $('#autoAllocationDisplay');
            const manualDisplay = $('#manualAllocationDisplay');
            const summaryMode = $('#summaryAllocationMode');
            const allocationInfo = $('#allocationInfo');
            
            isManualMode = toggle.is(':checked');
            $('#allocation_mode').val(isManualMode ? 'manual' : 'auto');
            
            if (isManualMode) {
                autoDisplay.css('display', 'none');
                manualDisplay.css('display', 'block');
                allocationInfo.text('Manual allocation selected. Check components and specify amounts.');
                
                // Initialize manual allocation with auto values
                initializeManualAllocation();
            } else {
                autoDisplay.css('display', 'block');
                manualDisplay.css('display', 'none');
                allocationInfo.text('Payments auto-allocate: Penalty → Processing Fee → Interest → Principal');
                
                // Recalculate auto allocation
                calculateAutoAllocation();
            }
            
            calculateAllocations();
        }

        function initializeManualAllocation() {
            // Initialize with auto allocation values
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            let remaining = paymentAmount;
            
            // Check which components have balance
            const hasPenalty = loanBalances.penalty > 0;
            const hasProcessing = loanBalances.processing > 0;
            const hasInterest = loanBalances.interest > 0;
            const hasPrincipal = loanBalances.principal > 0;
            
            // Auto-check components that have balance
            if (hasPenalty) {
                toggleAllocation('penalty');
                const amount = Math.min(loanBalances.penalty, remaining);
                $('#manualPenalty').val(amount.toFixed(2));
                manualAllocations.penalty.amount = amount;
                remaining -= amount;
            }
            
            if (hasProcessing && remaining > 0) {
                toggleAllocation('processing');
                const amount = Math.min(loanBalances.processing, remaining);
                $('#manualProcessing').val(amount.toFixed(2));
                manualAllocations.processing.amount = amount;
                remaining -= amount;
            }
            
            if (hasInterest && remaining > 0) {
                toggleAllocation('interest');
                const amount = Math.min(loanBalances.interest, remaining);
                $('#manualInterest').val(amount.toFixed(2));
                manualAllocations.interest.amount = amount;
                remaining -= amount;
            }
            
            if (hasPrincipal && remaining > 0) {
                toggleAllocation('principal');
                const amount = Math.min(loanBalances.principal, remaining);
                $('#manualPrincipal').val(amount.toFixed(2));
                manualAllocations.principal.amount = amount;
                remaining -= amount;
            }
            
            updateManualAllocationSummary();
        }

        function toggleAllocation(type) {
            const checkbox = $(`#${type}AllocationItem .custom-checkbox`);
            const input = $(`#manual${type.charAt(0).toUpperCase() + type.slice(1)}`);
            const allocationItem = $(`#${type}AllocationItem`);
            
            manualAllocations[type].enabled = !manualAllocations[type].enabled;
            
            if (manualAllocations[type].enabled) {
                checkbox.addClass('checked');
                input.prop('disabled', false);
                allocationItem.addClass('active');
                
                // Set default amount to balance if not already set
                if (input.val() === '0' || input.val() === '') {
                    input.val(Math.min(loanBalances[type], parseFloat($('#rtotal').val()) || 0).toFixed(2));
                    manualAllocations[type].amount = parseFloat(input.val()) || 0;
                }
            } else {
                checkbox.removeClass('checked');
                input.prop('disabled', true);
                allocationItem.removeClass('active');
                input.val('0');
                manualAllocations[type].amount = 0;
            }
            
            updateManualAllocationSummary();
        }

        function updateManualAllocation(type) {
            const input = $(`#manual${type.charAt(0).toUpperCase() + type.slice(1)}`);
            const amount = parseFloat(input.val()) || 0;
            const maxAmount = loanBalances[type];
            
            // Validate amount
            if (amount > maxAmount) {
                input.val(maxAmount.toFixed(2));
                manualAllocations[type].amount = maxAmount;
            } else if (amount < 0) {
                input.val('0');
                manualAllocations[type].amount = 0;
            } else {
                manualAllocations[type].amount = amount;
            }
            
            updateManualAllocationSummary();
        }

        function updateManualAllocationSummary() {
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            
            // Update individual allocation displays
            $('#penaltyAmount').text(manualAllocations.penalty.amount.toFixed(2));
            $('#processingAmount').text(manualAllocations.processing.amount.toFixed(2));
            $('#interestAmount').text(manualAllocations.interest.amount.toFixed(2));
            $('#principalAmount').text(manualAllocations.principal.amount.toFixed(2));
            
            // Calculate total allocated
            const totalAllocated = Object.values(manualAllocations).reduce((total, item) => total + item.amount, 0);
            $('#manualTotalAllocated').text(totalAllocated.toFixed(2));
            $('#manualPaymentAmount').text(paymentAmount.toFixed(2));
            
            // Calculate remaining to allocate
            const remaining = paymentAmount - totalAllocated;
            $('#manualRemainingAmount').text(remaining.toFixed(2));
            
            // Update hidden fields
            $('#rprincipal').val(manualAllocations.principal.amount.toFixed(2));
            $('#rinterest').val(manualAllocations.interest.amount.toFixed(2));
            $('#processing_fees_amount_paid').val(manualAllocations.processing.amount.toFixed(2));
            $('#penalties').val(manualAllocations.penalty.amount.toFixed(2));
            
            // Update current allocations for summary
            currentAllocations.principal = manualAllocations.principal.amount;
            currentAllocations.interest = manualAllocations.interest.amount;
            currentAllocations.processing = manualAllocations.processing.amount;
            currentAllocations.penalty = manualAllocations.penalty.amount;
            
            // Show validation messages
            const errorDiv = $('#manualAllocationError');
            const successDiv = $('#manualAllocationSuccess');
            
            if (Math.abs(remaining) > 0.01) {
                if (remaining > 0) {
                    $('#manualAllocationErrorMessage').text(
                        `Under-allocated by ${remaining.toFixed(2)}. Please allocate the full payment amount.`
                    );
                } else {
                    $('#manualAllocationErrorMessage').text(
                        `Over-allocated by ${Math.abs(remaining).toFixed(2)}. Please adjust allocations.`
                    );
                }
                errorDiv.removeClass('hidden');
                successDiv.addClass('hidden');
            } else {
                errorDiv.addClass('hidden');
                successDiv.removeClass('hidden');
            }
            
            // Update remaining balances display
            $('#penaltyBalanceRemaining').text(
                (loanBalances.penalty - manualAllocations.penalty.amount).toFixed(2)
            );
            $('#processingBalanceRemaining').text(
                (loanBalances.processing - manualAllocations.processing.amount).toFixed(2)
            );
            $('#interestBalanceRemaining').text(
                (loanBalances.interest - manualAllocations.interest.amount).toFixed(2)
            );
            $('#principalBalanceRemaining').text(
                (loanBalances.principal - manualAllocations.principal.amount).toFixed(2)
            );
        }

        function updateHiddenAllocationFields() {
            $('#rprincipal').val(currentAllocations.principal.toFixed(2));
            $('#rinterest').val(currentAllocations.interest.toFixed(2));
            $('#processing_fees_amount_paid').val(currentAllocations.processing.toFixed(2));
            $('#penalties').val(currentAllocations.penalty.toFixed(2));
        }

        function showFullPaymentAllocation() {
            const allocationDisplay = $('#autoAllocationDisplay');
            if (allocationDisplay.length) {
                const paymentAmount = parseFloat($('#rtotal').val()) || 0;
                
                if (paymentAmount === loanBalances.total) {
                    // Change styling for full payment
                    allocationDisplay.html(`
                        <div class="bg-green-50 border border-green-200 auto-allocation-indicator rounded-lg">
                            <div class="flex items-center justify-between mb-4">
                                <h4 class="font-medium text-green-800">
                                    <i class="fas fa-check-circle mr-2"></i> Full Settlement Allocation
                                </h4>
                                <span class="text-xs bg-green-100 text-green-800 px-2 py-1 rounded-full">
                                    Complete Payment
                                </span>
                            </div>
                            
                            <div class="space-y-4">
                                <div class="flex items-center justify-between">
                                    <label class="text-sm font-medium text-gray-700">Penalty:</label>
                                    <div class="relative w-48">
                                        <input type="text" 
                                               id="allocPenalty" 
                                               value="${currentAllocations.penalty.toFixed(2)}"
                                               readonly
                                               class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-300 rounded text-sm font-bold text-red-700 text-right">
                                        <span class="currency-indicator">ZMW</span>
                                    </div>
                                </div>
                                
                                <div class="flex items-center justify-between">
                                    <label class="text-sm font-medium text-gray-700">Processing Fee:</label>
                                    <div class="relative w-48">
                                        <input type="text" 
                                               id="allocProcessing" 
                                               value="${currentAllocations.processing.toFixed(2)}"
                                               readonly
                                               class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-300 rounded text-sm font-bold text-purple-600 text-right">
                                        <span class="currency-indicator">ZMW</span>
                                    </div>
                                </div>
                                
                                <div class="flex items-center justify-between">
                                    <label class="text-sm font-medium text-gray-700">Interest:</label>
                                    <div class="relative w-48">
                                        <input type="text" 
                                               id="allocInterest" 
                                               value="${currentAllocations.interest.toFixed(2)}"
                                               readonly
                                               class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-300 rounded text-sm font-bold text-yellow-700 text-right">
                                        <span class="currency-indicator">ZMW</span>
                                    </div>
                                </div>
                                
                                <div class="flex items-center justify-between">
                                    <label class="text-sm font-medium text-gray-700">Principal:</label>
                                    <div class="relative w-48">
                                        <input type="text" 
                                               id="allocPrincipal" 
                                               value="${currentAllocations.principal.toFixed(2)}"
                                               readonly
                                               class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-300 rounded text-sm font-bold text-red-600 text-right">
                                        <span class="currency-indicator">ZMW</span>
                                    </div>
                                </div>
                                
                                <div class="flex items-center justify-between pt-4 border-t border-green-200">
                                    <label class="text-sm font-medium text-gray-700">Total Allocated:</label>
                                    <div class="relative w-48">
                                        <input type="text" 
                                               id="allocTotal" 
                                               value="${Object.values(currentAllocations).reduce((a, b) => a + b, 0).toFixed(2)}"
                                               readonly
                                               class="readonly-input block w-full pl-3 pr-10 py-2 bg-white border border-green-400 rounded text-sm font-bold text-green-600 text-right">
                                        <span class="currency-indicator">ZMW</span>
                                    </div>
                                </div>
                            </div>
                            
                            <div class="mt-4 pt-4 border-t border-green-200">
                                <p class="text-sm text-green-700 font-medium">
                                    <i class="fas fa-check-circle mr-1"></i>
                                    This payment will fully settle the loan.
                                </p>
                            </div>
                        </div>
                    `);
                    allocationDisplay.css('display', 'block');
                }
            }
        }

        function resetGatewayState() {
            isGatewayProcessing = false;
            gatewayReferenceId = null;
            
            if (gatewayCheckInterval) {
                clearInterval(gatewayCheckInterval);
                gatewayCheckInterval = null;
            }
            
            // Reset submit button
            const submitBtn = $('#submitBtn');
            submitBtn.prop('disabled', false)
                    .html('<i class="fas fa-check-circle"></i> Process Payment')
                    .removeClass('bg-green-600')
                    .addClass('bg-zedbank-blue');
            
            // Hide gateway status
            $('#gateway_status').addClass('hidden');
            
            // Hide gateway summary info
            $('#gatewaySummaryInfo').addClass('hidden');
            
            // Clear payment reference if it was set by gateway
            if ($('#processing_method').val() === 'gateway') {
                $('#paymentreference').val('');
                $('#summaryReference').val('');
            }
        }

        async function initiateGatewayPayment(formData) {
            try {
                // Show processing state
                $('#submitBtn').prop('disabled', true)
                              .html('<div class="loading-spinner inline-block mr-2"></div> Initiating Payment...');
                
                // Prepare gateway request data with dynamic URLs
                const gatewayData = {
                    amount: formData.rtotal,
                    account_number: formData.gateway_mobile_number,
                    paymentmethod: formData.gateway_mobile_channel,
                    narration: `Loan repayment for <?php echo e($loan->loannumber); ?>`,
                    currency: 'ZMW',
                    loan_id: <?php echo e($loan->loanid); ?>,
                    company_id: CURRENT_COMPANY_ID,
                    user_id: CURRENT_USER_ID,
                    // Add these fields to help with repayment recording
                    pmethod: 'Mobile Money', // Valid payment method
                    paccount: $('#paccount').val(), // Use the selected account
                    rtotal: formData.rtotal,
                    allocations: formData.allocations || currentAllocations,
                };
                
                // Log the request with full URL
                console.log('Gateway payment request:', {
                    url: COLLECT_ENDPOINT,
                    data: gatewayData,
                    baseUrl: '<?php echo e($baseUrl); ?>'
                });
                
                // Make API call to initiate payment with dynamic URL
                const response = await $.ajax({
                    url: COLLECT_ENDPOINT,
                    method: 'POST',
                    data: JSON.stringify(gatewayData),
                    contentType: 'application/json',
                    dataType: 'json',
                    headers: {
                        'Accept': 'application/json',
                        'Content-Type': 'application/json'
                    }
                });
                
                if (response.success) {
                    // Store reference ID and start status polling
                    gatewayReferenceId = response.data.reference_id;
                    isGatewayProcessing = true;
                    
                    // Update payment reference field with the gateway reference ID
                    $('#paymentreference').val(gatewayReferenceId);
                    $('#summaryReference').val(gatewayReferenceId);
                    
                    // Update UI to show payment is processing
                    $('#submitBtn').html('<i class="fas fa-hourglass-half"></i> Payment Processing...');
                    
                    // Show gateway status
                    showGatewayStatus('pending', 'Payment initiated. Waiting for customer approval...');
                    
                    // Show reference ID and check status URL to user
                    const checkStatusUrl = `${STATUS_ENDPOINT}/${gatewayReferenceId}`;
                    showMessage(`Payment initiated. Reference: ${gatewayReferenceId}. Waiting for customer confirmation...`, 'info');
                    
                    // Log the status check URL
                    console.log('Payment status check URL:', checkStatusUrl);
                    
                    // Start polling for payment status
                    startGatewayStatusPolling(gatewayReferenceId);
                    
                    return true;
                } else {
                    throw new Error(response.error || 'Payment initiation failed');
                }
                
            } catch (error) {
                console.error('Gateway payment error:', {
                    error: error.message,
                    status: error.status,
                    responseText: error.responseText,
                    endpoint: COLLECT_ENDPOINT
                });
                
                let errorMessage = 'Failed to initiate gateway payment';
                if (error.status === 404) {
                    errorMessage += ': Endpoint not found. Please check the API URL configuration.';
                } else if (error.status === 500) {
                    errorMessage += ': Server error. Please try again later.';
                } else if (error.responseJSON && error.responseJSON.message) {
                    errorMessage += ': ' + error.responseJSON.message;
                } else {
                    errorMessage += ': ' + error.message;
                }
                
                showMessage(errorMessage, 'error');
                resetGatewayState();
                return false;
            }
        }

        function startGatewayStatusPolling(referenceId) {
            // Poll every 5 seconds for up to 5 minutes
            let attempts = 0;
            const maxAttempts = 60; // 5 minutes at 5-second intervals
            
            gatewayCheckInterval = setInterval(async () => {
                try {
                    attempts++;
                    
                    // Construct dynamic status check URL
                    const statusUrl = `${STATUS_ENDPOINT}/${referenceId}`;
                    
                    // Check payment status
                    const response = await $.ajax({
                        url: statusUrl,
                        method: 'GET',
                        dataType: 'json',
                        headers: {
                            'Accept': 'application/json'
                        }
                    });
                    
                    if (response.success) {
                        const status = response.data.status.toLowerCase();
                        
                        if (status === 'successful' || status === 'success' || status === 'paid' || status === 'completed') {
                            // Payment successful
                            clearInterval(gatewayCheckInterval);
                            showGatewayStatus('success', 'Payment completed successfully!');
                            
                            // Update UI with completion details
                            updateGatewayCompletionUI(response.data);
                            
                            // Complete the repayment process
                            await completeGatewayPayment(referenceId, response.data);
                            
                        } else if (status === 'failed' || status === 'failure' || status === 'rejected' || status === 'declined') {
                            // Payment failed
                            clearInterval(gatewayCheckInterval);
                            showGatewayStatus('failed', `Payment failed: ${response.data.message || 'Unknown error'}`);
                            showMessage('Gateway payment failed. Please try again or use manual payment.', 'error');
                            resetGatewayState();
                            
                        } else if (status === 'pending') {
                            // Still pending, update status
                            showGatewayStatus('pending', `Payment pending... (${attempts}/60 checks)`);
                            
                            // Update attempt counter in UI
                            updatePollingAttempts(attempts, maxAttempts);
                        }
                    }
                    
                    // Stop polling after max attempts
                    if (attempts >= maxAttempts) {
                        clearInterval(gatewayCheckInterval);
                        showGatewayStatus('timeout', 'Payment timeout. Please check payment status manually.');
                        
                        // Provide manual check URL
                        const manualCheckUrl = `${STATUS_ENDPOINT}/${referenceId}`;
                        showMessage(`Payment timeout. Please check status manually: ${manualCheckUrl}`, 'warning');
                        resetGatewayState();
                    }
                    
                } catch (error) {
                    console.error('Status polling error:', {
                        error: error.message,
                        url: `${STATUS_ENDPOINT}/${referenceId}`,
                        attempt: attempts
                    });
                    
                    if (attempts >= maxAttempts) {
                        clearInterval(gatewayCheckInterval);
                        showGatewayStatus('error', 'Error checking payment status');
                        showMessage('Unable to verify payment status. Please check manually.', 'warning');
                        resetGatewayState();
                    }
                }
            }, 5000); // Check every 5 seconds
        }

        function updatePollingAttempts(current, max) {
            const statusMessage = $('#gateway_status_message');
            if (statusMessage.length) {
                statusMessage.text(`Payment pending... (${current}/${max} checks)`);
            }
        }

        function updateGatewayCompletionUI(paymentData) {
            // Update summary with gateway info
            updateGatewaySummaryInfo('Completed', gatewayReferenceId);
            
            // Update main status display
            showGatewayStatus('success', `Payment of ${paymentData.amount} ZMW completed successfully!`);
        }

        function showGatewayStatus(status, message) {
            let statusDiv = $('#gateway_status');
            
            if (statusDiv.length === 0) {
                statusDiv = $(`
                    <div id="gateway_status" class="mt-4">
                        <div class="rounded-lg p-4">
                            <div class="flex items-center">
                                <i class="fas fa-spinner fa-spin mr-3"></i>
                                <div>
                                    <h4 class="font-medium" id="gateway_status_title">Processing Payment</h4>
                                    <p class="text-sm mt-1" id="gateway_status_message">${message}</p>
                                </div>
                            </div>
                        </div>
                    </div>
                `);
                
                // Insert after the payment details card
                $('.bg-white.rounded-xl.shadow-sm.border.overflow-hidden').last().after(statusDiv);
            } else {
                statusDiv.removeClass('hidden');
            }
            
            // Update status based on state
            const statusContainer = statusDiv.find('.rounded-lg');
            const statusIcon = statusDiv.find('i');
            const statusTitle = $('#gateway_status_title');
            const statusMessage = $('#gateway_status_message');
            
            statusContainer.removeClass('gateway-status-pending gateway-status-success gateway-status-failed gateway-status-timeout');
            statusIcon.removeClass('fa-spinner fa-spin fa-check-circle fa-times-circle fa-exclamation-triangle');
            
            switch (status) {
                case 'pending':
                    statusContainer.addClass('gateway-status-pending');
                    statusIcon.addClass('fa-spinner fa-spin');
                    statusTitle.text('Processing Payment');
                    break;
                case 'success':
                    statusContainer.addClass('gateway-status-success');
                    statusIcon.addClass('fa-check-circle');
                    statusTitle.text('Payment Successful');
                    break;
                case 'failed':
                    statusContainer.addClass('gateway-status-failed');
                    statusIcon.addClass('fa-times-circle');
                    statusTitle.text('Payment Failed');
                    break;
                case 'timeout':
                case 'error':
                    statusContainer.addClass('gateway-status-timeout');
                    statusIcon.addClass('fa-exclamation-triangle');
                    statusTitle.text('Payment Status Unknown');
                    break;
            }
            
            statusMessage.text(message);
            
            // Also update the summary
            updateGatewaySummaryInfo(status, gatewayReferenceId);
        }

        function updateGatewaySummaryInfo(status, referenceId) {
            const summaryInfo = $('#gatewaySummaryInfo');
            const statusSpan = $('#summaryGatewayStatus');
            const transactionIdSpan = $('#summaryTransactionId');
            const statusLink = $('#summaryStatusLink');
            
            if (referenceId) {
                summaryInfo.removeClass('hidden');
                transactionIdSpan.text(referenceId);
                
                // Add copy functionality
                transactionIdSpan.attr('title', 'Click to copy').css('cursor', 'pointer');
                transactionIdSpan.off('click').on('click', function() {
                    navigator.clipboard.writeText(referenceId).then(() => {
                        showMessage('Transaction ID copied to clipboard!', 'success');
                    });
                });
                
                // Update status link
                const statusUrl = `${STATUS_ENDPOINT}/${referenceId}`;
                statusLink.attr('href', statusUrl);
                
                // Update status with appropriate color
                statusSpan.text(status);
                statusSpan.removeClass('text-blue-600 text-green-600 text-red-600 text-yellow-600');
                
                if (status.toLowerCase().includes('pending')) {
                    statusSpan.addClass('text-blue-600');
                } else if (status.toLowerCase().includes('success') || status.toLowerCase().includes('completed')) {
                    statusSpan.addClass('text-green-600');
                } else if (status.toLowerCase().includes('failed')) {
                    statusSpan.addClass('text-red-600');
                } else {
                    statusSpan.addClass('text-yellow-600');
                }
            }
        }

           async function completeGatewayPayment(referenceId, paymentData) {
    try {
        // Get the cash/bank account ID
        const paccountId = $('#paccount').val();
        if (!paccountId) {
            throw new Error('No cash/bank account selected');
        }
        
        // Map gateway channel to controller's expected values
        const gatewayChannel = $('#gateway_mobile_channel').val();
        let mobileChannel = 'Mobile Money';
        switch(gatewayChannel.toLowerCase()) {
            case 'mtn':
                mobileChannel = 'MTN Mobile';
                break;
            case 'airtel':
                mobileChannel = 'Airtel Mobile';
                break;
            case 'zamtel':
                mobileChannel = 'Zamtel Mobile';
                break;
            default:
                mobileChannel = 'Mobile Money';
        }
        
        // Get current user ID from the view
        const currentUserId = <?php echo e(auth()->id() ?? 'null'); ?>;
        
        // Prepare repayment data for the API with valid values
        const repaymentData = {
            rtotal: $('#rtotal').val(),
            pmethod: 'Mobile Money', // Valid payment method value
            paccount: paccountId, // Use the selected cash/bank account ID
            rdate: $('#rdate').val(),
            paymentreference: referenceId, // Use gateway reference as payment reference
            notes: `Payment processed via payment gateway. Reference: ${referenceId}. Mobile: ${$('#gateway_mobile_number').val()}. Network: ${gatewayChannel}`,
            rprincipal: $('#rprincipal').val(),
            rinterest: $('#rinterest').val(),
            processing_fees_amount_paid: $('#processing_fees_amount_paid').val(),
            penalties: $('#penalties').val(),
            allocation_mode: $('#allocation_mode').val(),
            source: 'gateway',
            // Add user ID for the API
            user_id: currentUserId,
            // Add mobile money specific fields
            mobile_channel: mobileChannel,
            mobile_number: $('#gateway_mobile_number').val(),
            mobile_transaction_id: referenceId,
        };
        
        console.log('Submitting repayment data:', repaymentData);
        
        // Construct dynamic repayment API URL
        const repaymentUrl = `${API_BASE_URL}/loans/<?php echo e($loan->loanid); ?>/repayments`;
        
        console.log('Submitting repayment to:', repaymentUrl);
        
        // Submit to your existing repayment API
        const response = await $.ajax({
            url: repaymentUrl,
            method: 'POST',
            data: repaymentData,
            headers: {
                'X-CSRF-TOKEN': '<?php echo e($csrfToken); ?>',
                'Accept': 'application/json',
                'Content-Type': 'application/x-www-form-urlencoded'
            }
        });
        
        if (response.success) {
            showMessage('Payment completed and recorded successfully!', 'success');
            
            // Update submit button to show success
            $('#submitBtn').prop('disabled', true)
                          .html('<i class="fas fa-check-circle"></i> Payment Completed')
                          .removeClass('bg-zedbank-blue')
                          .addClass('bg-green-600');
            
            // Show completion details
            $('#completionDetails').removeClass('hidden').html(`
                <div class="bg-green-50 border border-green-200 rounded-lg p-4 mt-4">
                    <div class="flex items-center">
                        <i class="fas fa-check-circle text-green-500 mr-3 text-lg"></i>
                        <div>
                            <h4 class="font-medium text-green-800">Payment Successfully Completed</h4>
                            <div class="text-sm text-green-700 mt-1">
                                <p><strong>Reference:</strong> ${referenceId}</p>
                                <p><strong>Amount:</strong> ${paymentData.amount} ZMW</p>
                                <p><strong>Mobile:</strong> ${$('#gateway_mobile_number').val()} (${gatewayChannel.toUpperCase()})</p>
                                <p><strong>Journal Entry:</strong> ${response.data.journal_entry?.number || 'N/A'}</p>
                                <p><strong>New Balance:</strong> ${response.data.loan?.new_balance ? response.data.loan.new_balance + ' ZMW' : 'N/A'}</p>
                            </div>
                        </div>
                    </div>
                </div>
            `);
            
            // Update summary with new balance
            if (response.data.loan?.new_balance) {
                $('#summaryNewBalance').val(parseFloat(response.data.loan.new_balance).toFixed(2));
            }
            
            // Redirect after 5 seconds
            setTimeout(() => {
                window.location.href = "<?php echo e(route('admin.loans.active')); ?>";
            }, 5000);
            
            return true;
        } else {
            throw new Error(response.message || 'Failed to record payment');
        }
        
    } catch (error) {
        console.error('Complete payment error:', error);
        
        let errorMsg = 'Payment processed but recording failed. ';
        if (error.responseJSON) {
            if (error.responseJSON.errors) {
                const errors = error.responseJSON.errors;
                errorMsg += Object.entries(errors).map(([field, messages]) => 
                    `${field}: ${Array.isArray(messages) ? messages.join(', ') : messages}`
                ).join('; ');
            } else if (error.responseJSON.message) {
                errorMsg += error.responseJSON.message;
            }
        } else {
            errorMsg += error.message;
        }
        
        showMessage(errorMsg, 'warning');
        showGatewayStatus('error', 'Payment processed but recording failed. Please contact support.');
        
        // Provide manual completion option with proper values
        $('#manualCompletionOption').removeClass('hidden').html(`
            <div class="bg-yellow-50 border border-yellow-200 rounded-lg p-4 mt-4">
                <div class="flex items-center">
                    <i class="fas fa-exclamation-triangle text-yellow-500 mr-3 text-lg"></i>
                    <div class="flex-1">
                        <h4 class="font-medium text-yellow-800">Manual Completion Required</h4>
                        <p class="text-sm text-yellow-700 mt-1">
                            The payment was processed (Reference: ${referenceId}) but automatic recording failed.
                            Please record this payment manually with the following details:
                        </p>
                        <div class="mt-2 text-sm bg-white p-3 rounded border">
                            <p><strong>Reference:</strong> ${referenceId}</p>
                            <p><strong>Amount:</strong> ${$('#rtotal').val()} ZMW</p>
                            <p><strong>Payment Method:</strong> Mobile Money</p>
                            <p><strong>Mobile Network:</strong> ${$('#gateway_mobile_channel').val().toUpperCase()}</p>
                            <p><strong>Mobile Number:</strong> ${$('#gateway_mobile_number').val()}</p>
                            <p><strong>Allocations:</strong></p>
                            <ul class="ml-4">
                                <li>Principal: ${$('#rprincipal').val()} ZMW</li>
                                <li>Interest: ${$('#rinterest').val()} ZMW</li>
                                <li>Processing Fee: ${$('#processing_fees_amount_paid').val()} ZMW</li>
                                <li>Penalty: ${$('#penalties').val()} ZMW</li>
                            </ul>
                        </div>
                    </div>
                    <button onclick="retryGatewayCompletion('${referenceId}')" 
                            class="ml-4 px-4 py-2 bg-yellow-100 text-yellow-800 rounded-lg hover:bg-yellow-200 font-medium">
                        Retry Recording
                    </button>
                </div>
            </div>
        `);
        
        return false;
    }
}
        async function retryGatewayCompletion(referenceId) {
            try {
                // First check current payment status
                const statusResponse = await $.ajax({
                    url: `${STATUS_ENDPOINT}/${referenceId}`,
                    method: 'GET',
                    dataType: 'json'
                });
                
                if (statusResponse.success && statusResponse.data.status.toLowerCase() === 'successful') {
                    // Retry completion
                    await completeGatewayPayment(referenceId, statusResponse.data);
                } else {
                    showMessage('Payment is not yet successful. Please wait and try again.', 'warning');
                }
            } catch (error) {
                showMessage('Failed to retry: ' + error.message, 'error');
            }
        }

        function useScheduledAmount(amount) {
            $('#rtotal').val(amount.toFixed(2));
            calculateAllocations();
        }

        function updateSummary() {
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            const paymentMethod = $('#pmethod');
            const paymentDate = $('#rdate');
            const processingMethod = $('#processing_method').val();
            const paymentReference = $('#paymentreference').val();
            
            // Update summary display
            $('#summaryAmount').val(paymentAmount.toFixed(2));
            $('#summaryReference').val(paymentReference || 'N/A');
            
            // Update processing method display
            if (processingMethod === 'gateway') {
                $('#summaryProcessingMethod').val('Payment Gateway (Process Now)');
                
                // Add gateway-specific info
                if (paymentMethod.val() === 'Mobile Money') {
                    const mobileChannel = $('#gateway_mobile_channel').val();
                    const mobileNumber = $('#gateway_mobile_number').val();
                    
                    if (mobileChannel && mobileNumber) {
                        $('#summaryMethod').val(`Mobile Money (${mobileChannel.toUpperCase()}) - ${mobileNumber}`);
                    } else {
                        $('#summaryMethod').val('Mobile Money (Gateway)');
                    }
                }
            } else {
                $('#summaryProcessingMethod').val('Manual Payment (Record Only)');
                $('#summaryMethod').val(paymentMethod.val() ? 
                    paymentMethod.find('option:selected').text() : '-');
            }
            
            if (paymentDate.val()) {
                const date = new Date(paymentDate.val());
                $('#summaryDate').val(date.toLocaleDateString('en-US', {
                    month: 'short',
                    day: 'numeric',
                    year: 'numeric'
                }));
            }
            
            // Calculate new balances
            const newTotalBalance = loanBalances.total - paymentAmount;
            $('#summaryNewBalance').val(newTotalBalance.toFixed(2));
            
            // Calculate remaining for each component
            $('#summaryPrincipalRemaining').val(
                (loanBalances.principal - currentAllocations.principal).toFixed(2)
            );
            $('#summaryInterestRemaining').val(
                (loanBalances.interest - currentAllocations.interest).toFixed(2)
            );
            $('#summaryProcessingRemaining').val(
                (loanBalances.processing - currentAllocations.processing).toFixed(2)
            );
            $('#summaryPenaltyRemaining').val(
                (loanBalances.penalty - currentAllocations.penalty).toFixed(2)
            );
            
            // Update balance displays in the top cards
            $('#principalBalanceDisplay').val(
                (loanBalances.principal - currentAllocations.principal).toFixed(2)
            );
            $('#interestBalanceDisplay').val(
                (loanBalances.interest - currentAllocations.interest).toFixed(2)
            );
            $('#processingBalanceDisplay').val(
                (loanBalances.processing - currentAllocations.processing).toFixed(2)
            );
            $('#penaltyBalanceDisplay').val(
                (loanBalances.penalty - currentAllocations.penalty).toFixed(2)
            );
            $('#totalBalanceDisplay').val(newTotalBalance.toFixed(2));
        }

        function isValidMobileNumber(number) {
            // Remove non-digits
            const cleaned = number.replace(/\D/g, '');
            
            // Check if it's a valid Zambian mobile number (10 digits starting with 0, or 9 digits, or 12 digits with 260)
            if (cleaned.length === 10 && cleaned.startsWith('0')) {
                return true;
            }
            
            if (cleaned.length === 9) {
                return true;
            }
            
            if (cleaned.length === 12 && cleaned.startsWith('260')) {
                return true;
            }
            
            return false;
        }

        // Form validation and submission
        $('#paymentForm').on('submit', async function(e) {
            e.preventDefault();
            
            const paymentAmount = parseFloat($('#rtotal').val()) || 0;
            const paymentMethod = $('#pmethod').val();
            const processingMethod = $('#processing_method').val();
            const loanId = <?php echo e($loan->loanid); ?>;
            
            // Basic validation
            if (paymentAmount <= 0) {
                showMessage('Please enter a valid payment amount.', 'error');
                $('#rtotal').addClass('border-red-500').focus();
                return false;
            }
            
            if (paymentAmount > loanBalances.total) {
                showMessage('Payment amount cannot exceed total balance.', 'error');
                $('#rtotal').addClass('border-red-500').focus();
                return false;
            }
            
            if (!processingMethod) {
                showMessage('Please select a payment processing method.', 'error');
                $('#processing_method').addClass('border-red-500').focus();
                return false;
            }
            
            // Clear any previous validation messages
            $('input:invalid, select:invalid').removeClass('border-red-500');
            
            // Handle manual payments
            if (processingMethod === 'manual') {
                if (!paymentMethod) {
                    showMessage('Please select a payment method for manual payment.', 'error');
                    $('#pmethod').addClass('border-red-500').focus();
                    return false;
                }
                
                // Additional validation for manual payment methods
                if (paymentMethod === 'Bank') {
                    // Check if bank fields are visible and validate them
                    if ($('#payment_bank_fields').is(':visible')) {
                        const bankFields = ['bank_name', 'account_holder', 'account_number', 'bank_branch', 'bank_reference'];
                        let missingFields = [];
                        
                        bankFields.forEach(field => {
                            const $field = $(`[name="${field}"]`);
                            if (!$field.val() || $field.val().trim() === '') {
                                missingFields.push(field.replace('_', ' '));
                                $field.addClass('border-red-500');
                            } else {
                                $field.removeClass('border-red-500');
                            }
                        });
                        
                        if (missingFields.length > 0) {
                            showMessage(`Please fill in all required bank transfer details: ${missingFields.join(', ')}`, 'error');
                            // Focus on first missing field
                            $(`[name="${bankFields[0]}"]`).focus();
                            return false;
                        }
                    }
                }
                
                if (paymentMethod === 'Mobile Money') {
                    // Check if mobile fields are visible and validate them
                    if ($('#payment_mobile_fields').is(':visible')) {
                        const mobileFields = ['mobile_channel', 'mobile_number'];
                        let missingFields = [];
                        
                        mobileFields.forEach(field => {
                            const $field = $(`[name="${field}"]`);
                            if (!$field.val() || $field.val().trim() === '') {
                                missingFields.push(field.replace('_', ' '));
                                $field.addClass('border-red-500');
                            } else {
                                $field.removeClass('border-red-500');
                            }
                        });
                        
                        if (missingFields.length > 0) {
                            showMessage(`Please fill in all required mobile money details: ${missingFields.join(', ')}`, 'error');
                            // Focus on first missing field
                            $(`[name="${mobileFields[0]}"]`).focus();
                            return false;
                        }
                    }
                }
                
                // Validate allocations
                if (isManualMode) {
                    const totalAllocated = Object.values(manualAllocations).reduce((total, item) => total + item.amount, 0);
                    if (Math.abs(paymentAmount - totalAllocated) > 0.01) {
                        showMessage('Total manual allocation does not match payment amount. Please adjust your allocations.', 'error');
                        return false;
                    }
                }
                
                // Prepare form data
                const formData = $(this).serialize();
                
                // Show loading state
                $('#submitBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Processing...');
                
                // Submit via API
                $.ajax({
                    url: '<?php echo e(url("/api/loans")); ?>/'+loanId +'/repayments', // Use Laravel URL helper
                    method: 'POST',
                    data: formData,
                    headers: {
                        'X-CSRF-TOKEN': '<?php echo e($csrfToken); ?>',
                        'Accept': 'application/json',
                        'Content-Type': 'application/x-www-form-urlencoded'
                    },
                    success: function(response) {
                        if (response.success) {
                            showMessage(response.message, 'success');
                            
                            // Update submit button
                            $('#submitBtn').prop('disabled', true)
                                          .html('<i class="fas fa-check-circle"></i> Payment Completed')
                                          .removeClass('bg-zedbank-blue')
                                          .addClass('bg-green-600');
                            
                            // Redirect after 2 seconds
                            setTimeout(function() {
                                window.location.href = "<?php echo e(route('admin.loans.active')); ?>";
                            }, 2000);
                        } else {
                            showMessage('Error: ' + response.message, 'error');
                            $('#submitBtn').prop('disabled', false).html('<i class="fas fa-check-circle"></i> Process Payment');
                        }
                    },
                    error: function(xhr) {
                        let errorMessage = 'Payment failed. Please try again.';
                        
                        if (xhr.responseJSON && xhr.responseJSON.errors) {
                            const errors = xhr.responseJSON.errors;
                            errorMessage = Object.values(errors).flat().join('<br>');
                        } else if (xhr.responseJSON && xhr.responseJSON.message) {
                            errorMessage = xhr.responseJSON.message;
                        }
                        
                        showMessage(errorMessage, 'error');
                        $('#submitBtn').prop('disabled', false).html('<i class="fas fa-check-circle"></i> Process Payment');
                    }
                });
                
            } else if (processingMethod === 'gateway') {
                // Gateway payment handling
                if (!paymentMethod) {
                    showMessage('Please select a payment method for gateway processing.', 'error');
                    $('#pmethod').addClass('border-red-500').focus();
                    return false;
                }
                
                if (paymentMethod !== 'Mobile Money') {
                    showMessage('Gateway payment is currently only available for Mobile Money.', 'error');
                    $('#pmethod').addClass('border-red-500').focus();
                    return false;
                }
                
                // Validate gateway mobile money fields
                const mobileChannel = $('#gateway_mobile_channel').val();
                const mobileNumber = $('#gateway_mobile_number').val();
                
                if (!mobileChannel || !mobileNumber) {
                    showMessage('Please provide mobile network and phone number for gateway payment.', 'error');
                    
                    // Highlight missing fields
                    if (!mobileChannel) {
                        $('#gateway_mobile_channel').addClass('border-red-500');
                    }
                    if (!mobileNumber) {
                        $('#gateway_mobile_number').addClass('border-red-500');
                    }
                    
                    // Focus on first missing field
                    if (!mobileChannel) {
                        $('#gateway_mobile_channel').focus();
                    } else {
                        $('#gateway_mobile_number').focus();
                    }
                    
                    return false;
                }
                
                // Validate mobile number format
                if (!isValidMobileNumber(mobileNumber)) {
                    showMessage('Please enter a valid mobile phone number (e.g., 0976379025).', 'error');
                    $('#gateway_mobile_number').addClass('border-red-500').focus();
                    return false;
                }
                
                // Validate allocations for manual mode
                if (isManualMode) {
                    const totalAllocated = Object.values(manualAllocations).reduce((total, item) => total + item.amount, 0);
                    if (Math.abs(paymentAmount - totalAllocated) > 0.01) {
                        showMessage('Total manual allocation does not match payment amount. Please adjust your allocations.', 'error');
                        return false;
                    }
                }
                
                // Add URL validation
                if (!COLLECT_ENDPOINT || COLLECT_ENDPOINT === '/api') {
                    showMessage('Payment gateway endpoint is not properly configured.', 'error');
                    return false;
                }
                
                // Confirm gateway payment
                if (!confirm('This will initiate a mobile money payment request to the customer. Continue?')) {
                    return false;
                }
                
                // Prepare form data for gateway
                const formData = {
                    rtotal: paymentAmount,
                    gateway_mobile_channel: mobileChannel,
                    gateway_mobile_number: mobileNumber,
                    // Include allocation data if needed for the gateway request
                    allocations: isManualMode ? manualAllocations : currentAllocations
                };
                
                // Initiate gateway payment
                const initiated = await initiateGatewayPayment(formData);
                
                if (!initiated) {
                    // Gateway initiation failed
                    return false;
                }
                
                // Gateway payment initiated successfully - polling will handle the rest
                return false; // Prevent default form submission
            }
        });

        // Show message function
        function showMessage(message, type = 'info') {
            const messageContainer = $('#messageContainer');
            
            let bgColor = 'bg-blue-50';
            let borderColor = 'border-blue-200';
            let textColor = 'text-blue-700';
            let icon = 'fa-info-circle';
            
            if (type === 'success') {
                bgColor = 'bg-green-50';
                borderColor = 'border-green-200';
                textColor = 'text-green-700';
                icon = 'fa-check-circle';
            } else if (type === 'error') {
                bgColor = 'bg-red-50';
                borderColor = 'border-red-200';
                textColor = 'text-red-700';
                icon = 'fa-exclamation-circle';
            } else if (type === 'warning') {
                bgColor = 'bg-yellow-50';
                borderColor = 'border-yellow-200';
                textColor = 'text-yellow-700';
                icon = 'fa-exclamation-triangle';
            }
            
            messageContainer.html(`
                <div class="mb-6 ${bgColor} border ${borderColor} ${textColor} px-4 py-3 rounded-lg message-container">
                    <i class="fas ${icon} mr-2"></i> ${message}
                </div>
            `);
            
            // Auto-hide success messages after 5 seconds
            if (type === 'success') {
                setTimeout(() => {
                    messageContainer.fadeOut(300);
                }, 5000);
            }
        }
    </script>
</body>
</html><?php /**PATH /home/topacvbu/public_html/zedbankos.cloud/manageportal/resources/views/admin/loans/payment.blade.php ENDPATH**/ ?>